/* GTK - The GIMP Toolkit
 * Copyright (C) 1995-1997 Peter Mattis, Spencer Kimball and Josh MacDonald
 * GtkMultiRow Copyright (C) 1998 Stefan Ondrejicka <ondrej@idata.sk>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gtkmultirow.h"
#include <gtk/gtksignal.h>
#include <stdlib.h>

enum {
	ARG_0,
	ARG_VSPACING,
	ARG_HSPACING
};

static void   gtk_multirow_class_init	(GtkMultiRowClass	*klass);
static void   gtk_multirow_init		(GtkMultiRow		*multirow);
static void   gtk_multirow_get_arg	(GtkObject		*object,
					 GtkArg			*arg,
					 guint			arg_id);
static void   gtk_multirow_set_arg	(GtkObject		*object,
					 GtkArg			*arg,
					 guint			arg_id);
static void   gtk_multirow_size_request	(GtkWidget		*widget,
					 GtkRequisition		*requisition);
static void   gtk_multirow_size_allocate(GtkWidget		*widget,
					 GtkAllocation		*allocation);

guint gtk_multirow_get_type ()
{
	static guint multirow_type = 0;

	if (!multirow_type)
	{
		GtkTypeInfo multirow_info =
		{
			"GtkMultiRow",
			sizeof (GtkMultiRow),
			sizeof (GtkMultiRowClass),
			(GtkClassInitFunc) gtk_multirow_class_init,
			(GtkObjectInitFunc) gtk_multirow_init,
			NULL ,
			NULL ,
		};

		multirow_type = gtk_type_unique (gtk_box_get_type (), &multirow_info);
	}

	return multirow_type;
}

static void gtk_multirow_class_init (GtkMultiRowClass *class)
{
	GtkWidgetClass *widget_class;
	GtkObjectClass *object_class;

	widget_class = (GtkWidgetClass*) class;
	object_class = (GtkObjectClass*) class;

	widget_class->size_request = gtk_multirow_size_request;
	widget_class->size_allocate = gtk_multirow_size_allocate;

	gtk_object_add_arg_type ("GtkMultiRow::hspacing", GTK_TYPE_INT, GTK_ARG_READWRITE, ARG_HSPACING);
	gtk_object_add_arg_type ("GtkMultiRow::vspacing", GTK_TYPE_INT, GTK_ARG_READWRITE, ARG_VSPACING);

	object_class->set_arg = gtk_multirow_set_arg;
	object_class->get_arg = gtk_multirow_get_arg;

}

static void gtk_multirow_init (GtkMultiRow *multirow)
{
	multirow->vspacing = 3;
	multirow->hspacing = 3;
}

static void gtk_multirow_get_arg (
GtkObject	*object,
GtkArg		*arg,
guint		arg_id)
{
	GtkMultiRow *multirow = GTK_MULTIROW(object);

	switch (arg_id)
	{
		case ARG_VSPACING:
			GTK_VALUE_INT (*arg) = multirow->vspacing;
			break;
		case ARG_HSPACING:
			GTK_VALUE_INT (*arg) = multirow->hspacing;
			break;
		default:
			arg->type = GTK_TYPE_INVALID;
			break;
	}
}

static void gtk_multirow_set_arg (
GtkObject	*object,
GtkArg		*arg,
guint		arg_id)
{
	GtkMultiRow	*multirow = GTK_MULTIROW(object);

	switch (arg_id)
	{
		case ARG_VSPACING:
			gtk_multirow_set_spacing (multirow, multirow->hspacing , GTK_VALUE_INT (*arg));
			break;
		case ARG_HSPACING:
			gtk_multirow_set_spacing (multirow , GTK_VALUE_INT (*arg), multirow->vspacing);
			break;
		default:
			arg->type = GTK_TYPE_INVALID;
			break;
    }
}


GtkWidget* gtk_multirow_new (
gint ncols)
{
	GtkMultiRow *multirow;

	multirow = gtk_type_new (gtk_multirow_get_type ());

	multirow->ncols = ncols;

	return GTK_WIDGET (multirow);
}

static void gtk_multirow_size_request (
GtkWidget	*widget,
GtkRequisition	*requisition)
{
	GtkBox *box;
	GtkMultiRow *multirow;
	GtkBoxChild *child;
	GList *children;
	gint maxwidth,maxheight,nm,nrow,diff,diffp;

	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_MULTIROW (widget));
	g_return_if_fail (requisition != NULL);

	box = GTK_BOX (widget);
	multirow = GTK_MULTIROW (widget);

	maxwidth = 0;
	maxheight = 0;
	nm = 0;

	children = box->children;

	while (children)
	{
		child = children->data;
		children = children->next;

		if (GTK_WIDGET_VISIBLE (child->widget))
		{
			gtk_widget_size_request (child->widget, &child->widget->requisition);

			maxwidth = MAX (maxwidth , (child->widget->requisition.width + child->padding * 2));
	  		maxheight = MAX (maxheight , child->widget->requisition.height + child->padding * 2);

			nm++;
		}
	}

	if (multirow->ncols)
		nrow = nm/multirow->ncols + (nm % multirow->ncols != 0);
	else
	{
		nrow = 0;
		diff = 20000000;
		do
		{
			diffp = diff;
			nrow++;
			diff = abs((nm/nrow + (nm % nrow != 0)) * maxwidth - nrow * maxheight);
		}while(diffp > diff);

		nrow--;
	}

	if (nm)
	{
		requisition->height = nrow * (maxheight + multirow->hspacing) + multirow->hspacing;

		requisition->width = (nm/nrow + (nm % nrow != 0)) * (maxwidth + multirow->vspacing) + multirow->vspacing;
	}

	requisition->width += GTK_CONTAINER (box)->border_width * 2;
	requisition->height += GTK_CONTAINER (box)->border_width * 2;
}

static void gtk_multirow_size_allocate (
GtkWidget	*widget,
GtkAllocation	*allocation)
{
	GtkBox *box;
	GtkMultiRow *multirow;
	GtkBoxChild *child;
	GList *children;
	GList *pchildren;
	GtkAllocation child_allocation;
	gint maxwidth,maxheight,ncol,nm,row,col;

	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_MULTIROW (widget));
	g_return_if_fail (allocation != NULL);

	box = GTK_BOX (widget);
	multirow = GTK_MULTIROW (widget);

	widget->allocation = *allocation;

	if (!GTK_WIDGET_NO_WINDOW(widget) && GTK_WIDGET_REALIZED (widget))
	{
		gdk_window_move_resize (widget->window,
			allocation->x + GTK_CONTAINER (widget)->border_width,
			allocation->y + GTK_CONTAINER (widget)->border_width,
			allocation->width,
			allocation->height);

		gdk_window_raise(widget->window);
	}

	maxwidth = 0;
	maxheight = 0;
	nm = 0;

	children = box->children;
	while (children)
	{
		child = children->data;
		children = children->next;

		if (GTK_WIDGET_VISIBLE (child->widget))
		{
			gtk_widget_size_request (child->widget, &child->widget->requisition);

			maxwidth = MAX (maxwidth , (child->widget->requisition.width + child->padding * 2));
	  		maxheight = MAX (maxheight , child->widget->requisition.height + child->padding * 2);

			nm++;
		}
	}

	children = box->children;
	if (nm)
	{
		if (multirow->ncols)
			ncol = multirow->ncols;
		else
			ncol = (allocation->width - multirow->hspacing) /
				(maxwidth + multirow->hspacing);

		col = 0;
		row = 0;

		pchildren = NULL;

		while (children)
		{
			child = children->data;
			pchildren = children;
			children = children->next;

			if ((child->pack == GTK_PACK_START) && GTK_WIDGET_VISIBLE (child->widget))
			{
				child_allocation.x = allocation->x + multirow->hspacing +
						col * (maxwidth + multirow->hspacing);
				child_allocation.y = allocation->y + multirow->vspacing +
						row * (maxheight + multirow->vspacing);

				child_allocation.width = maxwidth;
				child_allocation.height = maxheight;

				gtk_widget_size_allocate (child->widget, &child_allocation);
			
				col++;

				if (col >= ncol)
				{
					col = 0;
					row++;
				}
			}
		}

		children = pchildren;

		while (children)
		{
			child = children->data;
			children = children->prev;

			if ((child->pack == GTK_PACK_END) && GTK_WIDGET_VISIBLE (child->widget))
			{
				child_allocation.x = allocation->x + multirow->hspacing +
						col * (maxwidth + multirow->hspacing);
				child_allocation.y = allocation->y + multirow->vspacing +
						row * (maxheight + multirow->vspacing);

				child_allocation.width = maxwidth;
				child_allocation.height = maxheight;

				gtk_widget_size_allocate (child->widget, &child_allocation);
			
				col++;

				if (col >= ncol)
				{
					col = 0;
					row++;
				}
			}
		}
	}
}

void gtk_multirow_set_spacing (
GtkMultiRow	*multirow ,
gint		hspacing ,
gint		vspacing)
{
	if (multirow->hspacing == hspacing && multirow->vspacing == vspacing) return;

	multirow->hspacing = hspacing;
	multirow->vspacing = vspacing;
	
	if (GTK_WIDGET_VISIBLE (multirow))
		gtk_widget_queue_resize (GTK_WIDGET (multirow));
}

void gtk_multirow_set_number_of_columns (
GtkMultiRow	*multirow ,
gint		ncols)
{
	if (multirow->ncols == ncols) return;

	multirow->ncols = ncols;

	if (GTK_WIDGET_VISIBLE (multirow))
		gtk_widget_queue_resize (GTK_WIDGET (multirow));
}
