/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: ViewController.cxx,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: rt $ $Date: 2007/04/03 16:00:06 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#include "precompiled_sd.hxx"

#include "framework/ViewController.hxx"

#include "framework/FrameworkHelper.hxx"
#include "framework/FactoryContainer.hxx"
#include "FrameView.hxx"

#ifndef _COM_SUN_STAR_DRAWING_FRAMEWORK_XCONTROLLERMANAGER_HPP_
#include <com/sun/star/drawing/framework/XControllerManager.hpp>
#endif
#ifndef _COM_SUN_STAR_DRAWING_FRAMEWORK_XMODULECONTROLLER_HPP_
#include <com/sun/star/drawing/framework/XModuleController.hpp>
#endif
#ifndef _COM_SUN_STAR_LANG_ILLEGALARGUMENTEXCEPTION_HPP_
#include <com/sun/star/lang/IllegalArgumentException.hpp>
#endif

#include <comphelper/stl_types.hxx>
#include <vcl/svapp.hxx>
#include <vos/mutex.hxx>

#include <boost/bind.hpp>
#include <hash_map>

using namespace ::com::sun::star;
using namespace ::com::sun::star::uno;
using namespace ::com::sun::star::drawing::framework;
using ::rtl::OUString;

#undef VERBOSE

namespace sd { namespace framework {


Reference<XInterface> SAL_CALL ViewController_createInstance (
    const Reference<XComponentContext>& rxContext)
{
    return Reference<XInterface>(static_cast<XWeak*>(new ViewController(rxContext)), UNO_QUERY);
}




::rtl::OUString ViewController_getImplementationName (void) throw(RuntimeException)
{
    return ::rtl::OUString(
        RTL_CONSTASCII_USTRINGPARAM("com.sun.star.comp.Draw.framework.view.ViewController"));
}




Sequence<rtl::OUString> SAL_CALL ViewController_getSupportedServiceNames (void)
    throw (RuntimeException)
{
	static const ::rtl::OUString sServiceName(
        ::rtl::OUString::createFromAscii("com.sun.star.drawing.framework.ViewController"));
	return Sequence<rtl::OUString>(&sServiceName, 1);
}




namespace {
    class ViewDescriptor
    {
    public:
        Reference<XResourceId> mxViewId;
        Reference<XView> mxView;
        Reference<XViewFactory> mxFactory;
        static bool CompareView (const ViewDescriptor& rDescriptor, const Reference<XView>& rxView)
        { return rDescriptor.mxView.get()==rxView.get(); }
        static bool ComparePaneURL (const ViewDescriptor& rDescriptor, const OUString& rsPaneURL)
        { return rDescriptor.mxViewId->isBoundToURL(rsPaneURL, AnchorBindingMode_DIRECT); }
    };
}




/** The view container stores maps from pane URLs to ViewDescriptor objects.
*/
class ViewController::ViewContainer
    : public ::std::vector<ViewDescriptor>
{
public:
    ViewContainer (void) {}
};




//===== ViewController ========================================================



ViewController::ViewController (
    const Reference<XComponentContext>& rxContext)
    throw()
    : ViewControllerInterfaceBase(MutexOwner::maMutex),
      mxConfigurationController(),
      mpFactoryContainer(new FactoryContainer()),
      mpViewContainer(new ViewContainer())
{
    (void)rxContext;
}




ViewController::~ViewController (void) throw()
{
}




void SAL_CALL ViewController::disposing (void)
{
    mxConfigurationController = NULL;

    while ( ! mpViewContainer->empty())
        ReleaseView(mpViewContainer->front().mxView);

    mpFactoryContainer.reset();
    mpViewContainer.reset();
}




//----- XViewController -------------------------------------------------------
    
void SAL_CALL ViewController::addViewFactory(
    const ::rtl::OUString& sViewURL,
    const Reference<XViewFactory >& rxViewFactory)
    throw (RuntimeException)
{
    ThrowIfDisposed();
    mpFactoryContainer->AddFactory(sViewURL, rxViewFactory);
}




void SAL_CALL ViewController::removeViewFactoryForURL(
    const ::rtl::OUString& rsViewURL)
    throw (::com::sun::star::uno::RuntimeException)
{
    ThrowIfDisposed();
    mpFactoryContainer->RemoveFactoryForURL(rsViewURL);
}




void SAL_CALL ViewController::removeViewFactoryForReference(
    const Reference<XViewFactory >& rxViewFactory)
    throw (::com::sun::star::uno::RuntimeException)
{
    ThrowIfDisposed();
    mpFactoryContainer->RemoveFactoryForReference(rxViewFactory);
}




void SAL_CALL ViewController::updateStart (
    const Reference<XConfiguration>& rxRequestedConfiguration,
    const Reference<XConfiguration>& rxCurrentConfiguration,
    const Sequence<Reference<XResourceId> >& rResourcesToDeactivate)
    throw (::com::sun::star::uno::RuntimeException)
{
    (void)rxRequestedConfiguration;

    ThrowIfDisposed();

    for (sal_Int32 nIndex=0; nIndex<rResourcesToDeactivate.getLength(); ++nIndex)
    {
        Reference<XResourceId> xResourceId (rResourcesToDeactivate[nIndex]);
        if (xResourceId->getResourceURL().match(FrameworkHelper::msViewURLPrefix))
        {
            ViewContainer::const_iterator iView;
            for (iView=mpViewContainer->begin(); iView!=mpViewContainer->end(); ++iView)
            {
                if (iView->mxViewId->compareTo(xResourceId) == 0)
                {
                    if (ReleaseView(iView->mxView))
                        rxCurrentConfiguration->removeResource(xResourceId);
                    break; 
                }
            }
        }
    }
}




void SAL_CALL ViewController::updateEnd (
    const Reference<XConfiguration>& rxRequestedConfiguration,
    const Reference<XConfiguration>& rxCurrentConfiguration,
    const Sequence<Reference<XResourceId> >& rResourcesToActivate)
    throw (::com::sun::star::uno::RuntimeException)
{
    (void)rxRequestedConfiguration;

    ThrowIfDisposed();

    // First the view in the center pane.
    for (sal_Int32 nIndex=0; nIndex<rResourcesToActivate.getLength(); ++nIndex)
    {
        Reference<XResourceId> xResourceId (rResourcesToActivate[nIndex]);
        if (xResourceId->isBoundToURL(
            FrameworkHelper::msCenterPaneURL, AnchorBindingMode_DIRECT)
            && xResourceId->getResourceURL().match(FrameworkHelper::msViewURLPrefix))
        {
            if (CreateView(xResourceId))
                rxCurrentConfiguration->addResource(xResourceId);
            break;
        }
    }

    for (sal_Int32 nIndex=0; nIndex<rResourcesToActivate.getLength(); ++nIndex)
    {
        Reference<XResourceId> xResourceId (rResourcesToActivate[nIndex]);
        if ( ! xResourceId->isBoundToURL(
            FrameworkHelper::msCenterPaneURL, AnchorBindingMode_DIRECT)
            && xResourceId->getResourceURL().match(FrameworkHelper::msViewURLPrefix))
        {
            if (CreateView(xResourceId))
                rxCurrentConfiguration->addResource(xResourceId);
        }
    }
}




Reference<XView> SAL_CALL ViewController::getView (const Reference<XResourceId>& rxResourceId)
    throw (RuntimeException)
{
    ThrowIfDisposed();
    ::osl::MutexGuard aGuard (maMutex);

    Reference<XView> xView;
    
    ViewContainer::const_iterator iView;
    for (iView=mpViewContainer->begin(); iView!=mpViewContainer->end(); ++iView)
    {
        if (iView->mxViewId->compareTo(rxResourceId) == 0)
        {
            xView = iView->mxView;
            break;
        }
    }

    return xView;
}




Reference<XView> SAL_CALL ViewController::getFirstViewForAnchor (
    const Reference<XResourceId>& rxAnchorId)
    throw (RuntimeException)
{
    ThrowIfDisposed();
    ::osl::MutexGuard aGuard (maMutex);

    Reference<XView> xView;
    
    ViewContainer::const_iterator iView;
    for (iView=mpViewContainer->begin(); iView!=mpViewContainer->end(); ++iView)
    {
        if (iView->mxViewId->isBoundTo(rxAnchorId, AnchorBindingMode_DIRECT))
        {
            xView = iView->mxView;
            break;
        }
    }

    return xView;
}




//----- XInitialization -------------------------------------------------------
    
void SAL_CALL ViewController::initialize(
    const Sequence<Any>& aArguments)
    throw (Exception, RuntimeException)
{
    ::osl::MutexGuard aGuard (maMutex);
    if (aArguments.getLength() > 0)
    {
        mxController = Reference<frame::XController>(aArguments[0], UNO_QUERY);
        
        // Cast the first argument to XControllerManager to make sure
        // that the right object is given.
        Reference<drawing::framework::XControllerManager> xControllerManager (
            mxController, UNO_QUERY);
        if (xControllerManager.is())
        {
            // Get a reference to the configuration controller.
            mxConfigurationController = xControllerManager->getConfigurationController();
            if ( ! mxConfigurationController.is())
                throw RuntimeException();

            // Get a reference to the module controller.
            mpFactoryContainer->SetModuleController(xControllerManager->getModuleController());
        }
    }
}




//-----------------------------------------------------------------------------

bool ViewController::CreateView (const Reference<XResourceId>& rxViewId)
{
    ::osl::MutexGuard aGuard (maMutex);

    bool bSuccess (false);

    Reference<XViewFactory> xFactory (
        mpFactoryContainer->GetFactory(rxViewId->getResourceURL()), UNO_QUERY);
    if (xFactory.is())
    {
#ifdef VERBOSE
        OSL_TRACE("creating view %s",
            OUStringToOString(
                FrameworkHelper::ResourceIdToString(rxViewId),
                RTL_TEXTENCODING_UTF8).getStr());
#endif

        // Insert the (still empty) descriptor right away to avoid recursive
        // calls to this method: the constructor of the ViewShell actively
        // shows its window; that causes the update methods of the
        // ViewController to be called again; when the entry for the
        // specified pane in the mpViewContainer is still empty at that
        // point then CreateView is called a second time.
        ViewDescriptor aDescriptor;
        aDescriptor.mxViewId = rxViewId;
        aDescriptor.mxFactory = xFactory;
        mpViewContainer->push_back(aDescriptor);
        
        // Create the view.
        Reference<XView> xView;
        try
        {
            xView = xFactory->createView(rxViewId, mxController);
        }
        catch (lang::DisposedException)
        {
            // The factory is disposed and can be removed from the list of
            // registered factories.
            xView = NULL;
            mpFactoryContainer->RemoveFactoryForReference(xFactory);
        }
        
        
        if (xView.is())
        {
            // Update the descriptor.
            mpViewContainer->back().mxView = xView;

            // Notifiy the new view to listeners of the ConfigurationController.
            ConfigurationChangeEvent aEvent;
            aEvent.Type = FrameworkHelper::msResourceActivationEvent;
            aEvent.ResourceId = rxViewId;
            aEvent.ResourceObject = xView;
            if (mxConfigurationController.is())
            {
                try
                {
                    mxConfigurationController->notifyEvent(aEvent);
                }
                catch (lang::DisposedException)
                {
                    mxConfigurationController = NULL;
                }
            }

            bSuccess = true;
        }
        else
        {
#ifdef VERBOSE
            OSL_TRACE("    creating view failed");
#endif
            // We have to remove the descriptor.
            mpViewContainer->erase(mpViewContainer->end()-1);
        }
    }
    
    return bSuccess;
}




/** This method is supposed to work with views that are already disposed.
*/
bool ViewController::ReleaseView (const Reference<XView>& rxView)
{
    ::osl::MutexGuard aGuard (maMutex);

    bool bSuccess (false);
    
    ViewContainer::iterator iView (
        ::std::find_if(
            mpViewContainer->begin(),
            mpViewContainer->end(),
            ::boost::bind(&ViewDescriptor::CompareView, _1, rxView)));
    if (iView != mpViewContainer->end())
    {
        OSL_TRACE("releasing view %s",
            OUStringToOString(FrameworkHelper::ResourceIdToString(iView->mxViewId),
                RTL_TEXTENCODING_UTF8).getStr());

        // Notify the deactivation of the view to listeners of the ConfigurationController.
        ConfigurationChangeEvent aEvent;
        aEvent.Type = FrameworkHelper::msResourceDeactivationEvent;
        aEvent.ResourceId  = iView->mxViewId;
        aEvent.ResourceObject = rxView;
        if (mxConfigurationController.is())
        {
            try
            {
                mxConfigurationController->notifyEvent(aEvent);
            }
            catch(lang::DisposedException)
            {
                mxConfigurationController = NULL;
            }
        }

        // Deactivate the view.  We release our reference to the XView
        // object before calling the factory so that the factory, as the
        // owner of the view, is the only one (that should be) holding a
        // reference and can check that.
        Reference<XViewFactory> xFactory (iView->mxFactory);
        Reference<XView> xView (rxView);
        mpViewContainer->erase(iView);
        try
        {
            xFactory->releaseView(xView);
        }
        catch(lang::DisposedException)
        {
            // The factory is disposed and can be removed from the list of
            // registered factories.
            mpFactoryContainer->RemoveFactoryForReference(xFactory);
        }

        bSuccess = true;
    }

    return bSuccess;
}




void ViewController::ThrowIfDisposed (void) const
    throw (::com::sun::star::lang::DisposedException)
{
	if (rBHelper.bDisposed || rBHelper.bInDispose)
	{
        throw lang::DisposedException (
            OUString(RTL_CONSTASCII_USTRINGPARAM(
                "ViewController object has already been disposed")),
            const_cast<uno::XWeak*>(static_cast<const uno::XWeak*>(this)));
    }
}


} } // end of namespace sd::framework
