/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: FileRepository.java,v $
 *
 *  $Revision: 1.3 $
 *
 *  last change: $Author: hr $ $Date: 2007/08/03 09:52:20 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2007 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *    Copyright 2007 by Pentaho Corporation
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/


package com.sun.star.report.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import com.sun.star.report.InputRepository;
import com.sun.star.report.OutputRepository;

public class FileRepository implements InputRepository, OutputRepository
{
  private File baseDirectory;
  private ManifestWriter manifestWriter;
  private boolean manifestDirty;

  public FileRepository(final File baseDirectory)
  {
    if (baseDirectory == null)
    {
      throw new NullPointerException();
    }
    this.baseDirectory = baseDirectory;
    this.manifestWriter = new ManifestWriter();
  }

  public InputStream createInputStream(final String name)
      throws IOException
  {
    return new FileInputStream(new File(baseDirectory, name));
  }

  /**
   * Returns a unique identifier for this repository. Two repositories accessing the same location should return the
   * same id. The identifier must never be null.
   *
   * @return the repository id
   */
  public Object getId()
  {
    return baseDirectory;
  }

  /**
   * This returns an version number for the given resource. Return zero, if the resource is not versionable, else return
   * a unique number for each version. As rule of thumb: Increase the version number by at least one for each change
   * made to the resource.
   *
   * @param name the name of the resource
   * @return the version number
   */
  public long getVersion(final String name)
  {
    return new File(baseDirectory, name).lastModified();
  }

  /**
   * Creates an output stream for writing the data. If there is an entry with that name already contained in the
   * repository, try to overwrite it.
   *
   * @param name
   * @return the outputstream
   * @throws java.io.IOException if opening the stream fails
   */
  public synchronized OutputStream createOutputStream(final String name, final String mimeType)
      throws IOException
  {
    final File file = new File(baseDirectory, name);
    final File parentFile = file.getParentFile();
    parentFile.mkdirs();
    manifestWriter.addEntry(name, mimeType);
    manifestDirty = true;
    return new FileOutputStream(file);
  }

  public synchronized boolean exists(final String name)
  {
    return new File(baseDirectory, name).exists();
  }

  public boolean isWritable(final String name)
  {
    return new File(baseDirectory, name).canWrite();
  }

  public boolean isReadable(final String name)
  {
    return new File(baseDirectory, name).canRead();
  }

  public void initialize (final String contentType)
  {
    manifestWriter.addEntry("/", contentType);
  }

  public void close() throws IOException
  {
    // Write the manifest ...
    if (manifestDirty)
    {
      manifestWriter.write(this);
      manifestDirty = false;
    }
  }
}
