#include <gtk/gtk.h>

#include "guiutils.h"
#include "fprompt.h"
#include "editclist.h"

#include "images/icon_add_16x16.xpm"
#include "images/icon_edit_16x16.xpm"
#include "images/icon_remove_16x16.xpm"
#include "images/icon_add_20x20.xpm"
#include "images/icon_edit_20x20.xpm"
#include "images/icon_remove_20x20.xpm"


static gint EditCListButtonPressEventCB(
        GtkWidget *widget, GdkEventButton *button, gpointer data
);
static void EditCListSelectRowCB(
        GtkCList *clist, gint row, gint column, GdkEvent *event,
        gpointer data
);
static void EditCListUnselectRowCB(
        GtkCList *clist, gint row, gint column, GdkEvent *event,
        gpointer data
);
static void EditCListAddCB(
        GtkWidget *widget, gpointer data
);
static void EditCListEditCB(
        GtkWidget *widget, gpointer data
);
static void EditCListRemoveCB(
        GtkWidget *widget, gpointer data
);
static void EditCListShiftUpCB(
        GtkWidget *widget, gpointer data
);
static void EditCListShiftDownCB(
        GtkWidget *widget, gpointer data
);

static void EditCListFPromptCancelCB(gpointer data);
static void EditCListFPromptApplyCB(
        gpointer data, const gchar *value
);
static void EditCListFPromptMap(
	editclist_struct *ec, gint row, gint column
);


GtkWidget *EditCListGetCList(editclist_struct *ec);

editclist_struct *EditCListNew(
	GtkWidget *parent,	/* A GtkBox. */
	gint width, gint height,
	gboolean expand, gboolean fill, guint padding,
	gchar **heading, gint columns,
	void (*row_changed_cb)(gpointer, GtkCList *, gint, gint, gpointer),
        gpointer client_data
);
void EditCListUpdateMenus(editclist_struct *ec);
void EditCListMap(editclist_struct *ec);
void EditCListUnmap(editclist_struct *ec);
void EditCListDelete(editclist_struct *ec);



/*
 *	GtkCList "button_press_event" signal callback.
 */
static gint EditCListButtonPressEventCB(
        GtkWidget *widget, GdkEventButton *button, gpointer data
)
{
	gint status = FALSE;
	gint etype;
	GtkCList *clist;
        editclist_struct *ec = EDITCLIST(data);
        if(ec == NULL)
            return(status);

	clist = (GtkCList *)EDITCLIST_CLIST(ec);
	if(clist == NULL)
	    return(status);

	if(clist->clist_window != ((GdkEventAny *)button)->window)
	    return(status);

	/* Get event type. */
	etype = button->type;

	if(TRUE)
	{
	    gint row, column;
	    GtkWidget *w;


            /* Find row and column based on given coordinates. */
            if(!gtk_clist_get_selection_info(
                clist, (gint)button->x, (gint)button->y, &row, &column
            ))
            {
                row = -1;
                column = 0;
            }

	    /* Handle by event type. */
	    switch(etype)
	    {
	      case GDK_BUTTON_PRESS:
		/* Handle by button number. */
		switch(button->button)
		{
		  case 3:
                    /* Need to select this cell so that operations on
		     * menu operate on this cell.
		     */
		    gtk_clist_select_row(clist, row, column);

		    /* Get right click menu widget and map it. */
		    w = ec->menu;
		    if(w != NULL)
			gtk_menu_popup(
			    GTK_MENU(w), NULL, NULL,
			    NULL, NULL,
			    button->button, button->time
			);
		    break;

		  case 2:
                    /* Map floating prompt over the clicked on cell. */
		    EditCListFPromptMap(ec, row, column);
		    break;
		}
		status = TRUE;
		break;

              case GDK_2BUTTON_PRESS:
                /* Handle by button number. */
                switch(button->button)
                {
                  case 1:
                    /* Map floating prompt over the clicked on cell. */
                    EditCListFPromptMap(ec, row, column);
                    break;
		}
		status = TRUE;
		break;
	    }
	}

        return(status);
}

/*
 *	GtkCList "select_row" signal callback.
 */
static void EditCListSelectRowCB(
        GtkCList *clist, gint row, gint column, GdkEvent *event,
        gpointer data
)
{
        editclist_struct *ec = EDITCLIST(data);
        if(ec == NULL)
            return;

        /* Record selected row and column. */
        ec->sel_row = row;
        ec->sel_column = column;

	/* Check if selected row is fully visible, if not then
	 * adjust scroll position to try and make it visible.
	 */
	if(gtk_clist_row_is_visible(clist, row) != GTK_VISIBILITY_FULL)
	    gtk_clist_moveto(
		clist,
		row, -1,	/* Row, column. */
		0.5, 0.0	/* Row, column. */
	    );

        EditCListUpdateMenus(ec);
}

/*
 *      GtkCList "unselect_row" signal callback.
 */
static void EditCListUnselectRowCB(
        GtkCList *clist, gint row, gint column, GdkEvent *event,
        gpointer data
)
{
        editclist_struct *ec = EDITCLIST(data);
        if(ec == NULL)
            return;

	/* Mark current row and column as unselected. */
	ec->sel_row = -1;
	ec->sel_column = -1;

        EditCListUpdateMenus(ec);
}

/*
 *	Add row callback.
 */
static void EditCListAddCB(GtkWidget *widget, gpointer data)
{
        gint sel_row, new_row;
	gchar **strv;
        GList *glist;
        GtkCList *clist;
        editclist_struct *ec = EDITCLIST(data);
        if(ec == NULL)
            return;

        clist = (GtkCList *)EDITCLIST_CLIST(ec);
        if(clist == NULL)
            return;

        /* Get selected row. */
        glist = clist->selection_end;
        sel_row = (glist != NULL) ? (gint)glist->data : -1;

	/* Allocate row text data. */
	strv = (gchar **)g_malloc(clist->columns * sizeof(gchar *));
	if(strv != NULL)
	{
	    gint i;
	    for(i = 0; i < clist->columns; i++)
		strv[i] = g_strdup("");
	}

        /* Check if selected row is in bounds. */
        if((sel_row >= 0) && (sel_row < clist->rows))
	{
	    /* Selected row in bounds, insert a new row. */
	    new_row = gtk_clist_insert(
		clist, sel_row, strv
	    );
	}
	else
	{
	    /* Selected row out of bounds, append a new row. */
	    new_row = gtk_clist_append(
		clist, strv
	    );
	}

        /* Deallocate row text data. */
        if(strv != NULL)
        {
            gint i;
            for(i = 0; i < clist->columns; i++)
                g_free(strv[i]);
	    g_free(strv);
	    strv = NULL;
        }

	/* Select new row. */
	gtk_clist_select_row(clist, new_row, 0);

	/* Call row changed callback. */
        if((new_row > -1) && (ec->row_changed_cb != NULL))
	    ec->row_changed_cb(
		ec, clist, new_row, 0, ec->client_data
	    );
}

/*
 *	Edit selected cell callback.
 */
static void EditCListEditCB(GtkWidget *widget, gpointer data)
{
        GtkCList *clist;
        editclist_struct *ec = EDITCLIST(data);
        if(ec == NULL)
            return;

        clist = (GtkCList *)EDITCLIST_CLIST(ec);
        if(clist == NULL)
            return;

	/* Map floating prompt over the last selected cell. */
	EditCListFPromptMap(
	    ec, ec->sel_row, ec->sel_column
	);

        EditCListUpdateMenus(ec);
}

/*
 *	Remove selected row callback.
 */
static void EditCListRemoveCB(GtkWidget *widget, gpointer data)
{
	gint sel_row;
	GList *glist;
	GtkCList *clist;
        editclist_struct *ec = EDITCLIST(data);
        if(ec == NULL)
            return;

	clist = (GtkCList *)EDITCLIST_CLIST(ec);
	if(clist == NULL)
	    return;

	/* Get selected row. */
	glist = clist->selection_end;
	sel_row = (glist != NULL) ? (gint)glist->data : -1;

	/* Check if selected row is in bounds. */
	if((sel_row >= 0) && (sel_row < clist->rows))
	{
            /* Call row changed callback (before removal). */
            if(ec->row_changed_cb != NULL)
                ec->row_changed_cb(
                    ec, clist, sel_row, 0, ec->client_data
                );

	    /* Remove row. */
	    gtk_clist_remove(clist, sel_row);
	}

	EditCListUpdateMenus(ec);
}

/*
 *	Shift selected row up callback.
 */
static void EditCListShiftUpCB(GtkWidget *widget, gpointer data)
{
        gint sel_row;
        GList *glist;
        GtkCList *clist;
        editclist_struct *ec = EDITCLIST(data);
        if(ec == NULL)
            return;

        clist = (GtkCList *)EDITCLIST_CLIST(ec);
        if(clist == NULL)
            return;

	/* Get selected row. */
	glist = clist->selection_end;
	sel_row = (glist != NULL) ? (gint)glist->data : -1;

	/* Can shift up? */
	if(sel_row > 0)
	{
	    /* Shift selected row up. */
	    gtk_clist_swap_rows(clist, sel_row, sel_row - 1);

            /* Call row changed callback. */
            if(ec->row_changed_cb != NULL)
	    {
                ec->row_changed_cb(
                    ec, clist, sel_row, 0, ec->client_data
                );
                ec->row_changed_cb(
                    ec, clist, sel_row - 1, 0, ec->client_data
                );
	    }
	}

        EditCListUpdateMenus(ec);
}

/*
 *	Shift selected row down callback.
 */
static void EditCListShiftDownCB(GtkWidget *widget, gpointer data)
{
        gint sel_row;
        GList *glist;
        GtkCList *clist;
        editclist_struct *ec = EDITCLIST(data);
        if(ec == NULL)
            return;

        clist = (GtkCList *)EDITCLIST_CLIST(ec);
        if(clist == NULL)
            return;

        /* Get selected row. */
        glist = clist->selection_end;
        sel_row = (glist != NULL) ? (gint)glist->data : -1;

        /* Can shift down? */
        if((sel_row > -1) && (sel_row < (clist->rows - 1)))
        {
            /* Shift selected row down. */
            gtk_clist_swap_rows(clist, sel_row, sel_row + 1);

            /* Call row changed callback. */
            if(ec->row_changed_cb != NULL)
            {
                ec->row_changed_cb(
                    ec, clist, sel_row, 0, ec->client_data
                );
                ec->row_changed_cb(
                    ec, clist, sel_row + 1, 0, ec->client_data
                );
            }
        }

        EditCListUpdateMenus(ec);
}


/*
 *	Floating prompt cancel callback.
 */
static void EditCListFPromptCancelCB(gpointer data)
{
        editclist_struct *ec = EDITCLIST(data);
        if(ec == NULL)
            return;

	/* Do nothing. */
}

/*
 *	Floating prompt apply callback.
 */
static void EditCListFPromptApplyCB(gpointer data, const gchar *value)
{
        GtkCList *clist;
        gint row, column;
	editclist_struct *ec = EDITCLIST(data);
        if(ec == NULL)
            return;

        clist = (GtkCList *)EDITCLIST_CLIST(ec);
        row = ec->fprompt_row;
        column = ec->fprompt_column;

        /* Inputs valid? */
        if((ec != NULL) && (clist != NULL) && (row >= 0) && (column >= 0) &&
           (value != NULL)
        )
        {
            gchar *strptr;
            guint8 spacing;
            GdkPixmap *pixmap;
            GdkBitmap *mask;

	    /* Set new value from floating prompt to the selected cell
	     * on the clist.
	     */
	    switch((gint)gtk_clist_get_cell_type(
		clist, row, column
	    ))
	    {
	      case GTK_CELL_TEXT:
		gtk_clist_set_text(
		    clist, row, column, value
		);
		break;
              case GTK_CELL_PIXTEXT:
                gtk_clist_get_pixtext(
                    clist, row, column, &strptr,
                    &spacing, &pixmap, &mask
                );
                gtk_clist_set_pixtext(
                    clist, row, column, value,
                    spacing, pixmap, mask
                );
                break;
            }

            /* Call row changed callback. */
            if(ec->row_changed_cb != NULL)
                ec->row_changed_cb(
                    ec, clist, row, column, ec->client_data
                );
	}
}

/*
 *	Maps the float prompt over the specified cell on the given
 *	editable clist.
 */
static void EditCListFPromptMap(
        editclist_struct *ec, gint row, gint column
)
{
        gint cx, cy, px, py, pwidth, pheight;
	gchar *value;
        GtkCList *clist;

        gchar *strptr;
        guint8 spacing;
        GdkPixmap *pixmap;
        GdkBitmap *mask;


	if(ec == NULL)
	    return;

	clist = (GtkCList *)EDITCLIST_CLIST(ec);
	if(clist == NULL)
	    return;

        if(FPromptIsQuery())
            return;

        /* Make sure given row index is in bounds. */
        if((row < 0) || (row >= clist->rows))
            return;

        /* Get clist cell geometry. */
        if(GUICListGetCellGeometry(
            clist, column, row,
            &cx, &cy, &pwidth, &pheight
        ))
            return;

        /* Get root window relative coordinates. */
        px = 0;
        py = 0;
        gdk_window_get_deskrelative_origin(
            clist->clist_window, &px, &py
        );
        px += cx + 0;
        py += cy - 2;   /* Move up a little. */


	/* Get cell text as coppied value. */
	strptr = NULL;
	switch((gint)gtk_clist_get_cell_type(
	    clist, row, column
	))
	{
	  case GTK_CELL_TEXT:
	    gtk_clist_get_text(
		clist, row, column, &strptr
	    );
	    break;
	  case GTK_CELL_PIXTEXT:
	    gtk_clist_get_pixtext(
		clist, row, column, &strptr,
		&spacing, &pixmap, &mask
	    );
	    break;
	}
	value = (strptr != NULL) ? g_strdup(strptr) : NULL;


        /* Is the clicked on column the name column or given column is -1
         * implying any column is acceptable?
         */
        if(value != NULL)
        {
	    /* Record row and column float prompt mapped over. */
	    ec->fprompt_row = row;
	    ec->fprompt_column = column;

            /* Map floating prompt to change values. */
            FPromptSetTransientFor(
		gtk_widget_get_toplevel(ec->toplevel)
	    );
            FPromptSetPosition(px, py);
            FPromptMapQuery(
                NULL,                   /* No label. */
                value,
                NULL,                   /* No tooltip message. */
                FPROMPT_MAP_NO_MOVE,    /* Map code. */
                pwidth, -1,             /* Width and height. */
/*              FPROMPT_FLAG_OK_BTN | FPROMPT_FLAG_CANCEL_BTN, */
                0,      /* No buttons. */
                ec,			/* Client data. */
                NULL,			/* No browse callback. */
                EditCListFPromptApplyCB,
		EditCListFPromptCancelCB
            );
	}

	g_free(value);
}


/*
 *	Returns the GtkCList widget on the given editable clist.
 */
GtkWidget *EditCListGetCList(editclist_struct *ec)
{
	return(EDITCLIST_CLIST(ec));
}

/*
 *	Creates a new editable clist.
 */
editclist_struct *EditCListNew(
        GtkWidget *parent,      /* A GtkBox. */
        gint width, gint height,
        gboolean expand, gboolean fill, guint padding,
        gchar **heading, gint columns,
        void (*row_changed_cb)(gpointer, GtkCList *, gint, gint, gpointer),
        gpointer client_data
)
{
/*	gint border_minor = 2, border_major = 5; */
	gint bw = 20, bh = 20;
	gpointer mclient_data;
        gpointer accel_group;
        GtkWidget *w, *fw, *menu, *parent2, *parent3, *parent4;
        GtkCList *clist;
	editclist_struct *ec = (editclist_struct *)g_malloc0(
	    sizeof(editclist_struct)
	);
	if(ec == NULL)
	    return(ec);


	/* Reset values. */
	ec->initialized = TRUE;
	ec->map_state = FALSE;
	ec->sel_row = -1;
	ec->sel_column = -1;
	ec->fprompt_row = -1;
	ec->fprompt_column = -1;
	ec->row_changed_cb = row_changed_cb;
	ec->client_data = client_data;

        /* Main vbox. */
	if(parent != NULL)
	{
	    ec->toplevel = w = gtk_vbox_new(FALSE, 0);
	    gtk_widget_set_usize(w, width, height);
	    if(GTK_IS_BOX(parent))
		gtk_box_pack_start(GTK_BOX(parent), w, expand, fill, padding);
	    else if(GTK_IS_CONTAINER(parent))
		gtk_container_add(GTK_CONTAINER(parent), w);
	    parent2 = w;
	}
	else
	{
	    parent2 = NULL;
	}

	/* Buttons hbox. */
	w = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
        parent3 = w;
        /* Alignment to align to the right. */
        w = gtk_alignment_new(1.0, 0.5, 0.0, 0.0);
        gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
        gtk_widget_show(w);
        parent3 = w;
        /* Buttons set hbox. */
        w = gtk_hbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent3), w);
        gtk_widget_show(w);
        parent3 = w;

	/* Add button. */
	ec->add_btn = w = (GtkWidget *)GUIButtonPixmap(
	    (u_int8_t **)icon_add_16x16_xpm
	);
        gtk_widget_set_usize(w, bw, bh);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EditCListAddCB), ec
        );
        gtk_widget_show(w);

        /* Edit button. */
        ec->edit_btn = w = (GtkWidget *)GUIButtonPixmap(
            (u_int8_t **)icon_edit_16x16_xpm
        );
        gtk_widget_set_usize(w, bw, bh);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EditCListEditCB), ec
        );
        gtk_widget_show(w);

        /* Remove button. */
        ec->remove_btn = w = (GtkWidget *)GUIButtonPixmap(
            (u_int8_t **)icon_remove_16x16_xpm
        );
        gtk_widget_set_usize(w, bw, bh);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EditCListRemoveCB), ec
        );
        gtk_widget_show(w);

        /* Shift up button. */
        ec->up_btn = w = gtk_button_new();
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EditCListShiftUpCB), ec
        );
        gtk_widget_set_usize(w, bw, bh);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent4 = w;
        /* Arrow. */
        w = gtk_arrow_new(GTK_ARROW_UP, GTK_SHADOW_OUT);
        gtk_container_add(GTK_CONTAINER(parent4), w);
        gtk_widget_show(w);

        /* Shift down button. */
        ec->down_btn = w = gtk_button_new();
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(EditCListShiftDownCB), ec
        );
        gtk_widget_set_usize(w, bw, bh);
        gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
        gtk_widget_show(w);
        parent4 = w;
        /* Arrow. */
        w = gtk_arrow_new(GTK_ARROW_DOWN, GTK_SHADOW_OUT);
        gtk_container_add(GTK_CONTAINER(parent4), w);
        gtk_widget_show(w);



	/* Scrolled window for clist. */
        w = gtk_scrolled_window_new(NULL, NULL);
        gtk_scrolled_window_set_policy(
            GTK_SCROLLED_WINDOW(w),
            GTK_POLICY_AUTOMATIC,
            GTK_POLICY_AUTOMATIC
        );
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
        gtk_widget_show(w);
        parent3 = w;

        /* Contents clist. */
        ec->clist = w = gtk_clist_new_with_titles(columns, heading);
        clist = GTK_CLIST(w);
        gtk_widget_add_events(w, GDK_BUTTON_PRESS_MASK);
        gtk_signal_connect_after(
            GTK_OBJECT(w), "button_press_event",
            GTK_SIGNAL_FUNC(EditCListButtonPressEventCB), ec
        );
        gtk_container_add(GTK_CONTAINER(parent3), w);
        gtk_signal_connect(
            GTK_OBJECT(w), "select_row",
            GTK_SIGNAL_FUNC(EditCListSelectRowCB), ec
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "unselect_row",
            GTK_SIGNAL_FUNC(EditCListUnselectRowCB), ec
        );
        gtk_widget_show(w);


	/* Right-click menu for clist. */
	ec->menu = menu = (GtkWidget *)GUIMenuCreate();
        accel_group = NULL;
        mclient_data = ec;
        if(menu != NULL)
        {
	    guint8 **icon;
	    gint accel_key;
	    guint accel_mods;
	    const gchar *label;
	    void (*func_cb)(GtkWidget *, gpointer);

#define DO_ADD_MENU_ITEM_LABEL				\
{ w = GUIMenuItemCreate(				\
  menu, GUI_MENU_ITEM_TYPE_LABEL, accel_group,		\
  icon, label, accel_key, accel_mods, (void **)&fw,	\
  mclient_data, func_cb					\
 ); }
#define DO_ADD_MENU_SEP					\
{ w = GUIMenuItemCreate(				\
  menu, GUI_MENU_ITEM_TYPE_SEPARATOR, NULL,		\
  NULL, NULL, 0, 0, NULL,				\
  NULL, NULL						\
 ); }

            icon = (guint8 **)icon_add_20x20_xpm;
            label = "Add";
            accel_key = 0;
            accel_mods = 0;
            func_cb = EditCListAddCB;
            DO_ADD_MENU_ITEM_LABEL
	    ec->add_mi = w;

            icon = (guint8 **)icon_edit_20x20_xpm;
            label = "Edit";
            accel_key = 0;
            accel_mods = 0;
            func_cb = EditCListEditCB;
            DO_ADD_MENU_ITEM_LABEL
            ec->edit_mi = w;

            icon = (guint8 **)icon_remove_20x20_xpm;
            label = "Remove";
            accel_key = 0;
            accel_mods = 0;
            func_cb = EditCListRemoveCB;
            DO_ADD_MENU_ITEM_LABEL
            ec->remove_mi = w;

	    DO_ADD_MENU_SEP

            icon = NULL;
            label = "Shift Up";
            accel_key = 0;
            accel_mods = 0;
            func_cb = EditCListShiftUpCB;
            DO_ADD_MENU_ITEM_LABEL
            ec->up_mi = w;

            icon = NULL;
            label = "Shift Down";
            accel_key = 0;
            accel_mods = 0;
            func_cb = EditCListShiftDownCB;
            DO_ADD_MENU_ITEM_LABEL
            ec->down_mi = w;

#undef DO_ADD_MENU_ITEM_LABEL
#undef DO_ADD_MENU_SEP
	}


	EditCListUpdateMenus(ec);

	return(ec);
}

/*
 *	Updates the menus and widgets on the given editable clist to match
 *	its current values.
 */
void EditCListUpdateMenus(editclist_struct *ec)
{
	gboolean sensitive;
	gint sel_row;
	GList *glist;
	GtkWidget *w;
	GtkCList *clist;


        if(ec == NULL)
            return;

#define DO_SET_SENSITIVE	\
{ if(w != NULL) { gtk_widget_set_sensitive(w, sensitive); } }

        /* Get selected row on clist. */
	clist = (GtkCList *)EDITCLIST_CLIST(ec);
	glist = (clist != NULL) ? clist->selection_end : NULL;
	sel_row = (glist != NULL) ? (gint)glist->data : -1;

	/* Update widgets. */
	sensitive = (sel_row > -1) ? TRUE : FALSE;
	w = ec->edit_btn;
	DO_SET_SENSITIVE
	w = ec->remove_btn;
	DO_SET_SENSITIVE
        w = ec->edit_mi;
        DO_SET_SENSITIVE
        w = ec->remove_mi;
        DO_SET_SENSITIVE

	sensitive = (sel_row > 0) ? TRUE : FALSE;
	w = ec->up_btn;
        DO_SET_SENSITIVE
        w = ec->up_mi;
        DO_SET_SENSITIVE

        sensitive = ((sel_row > -1) && (sel_row < (clist->rows - 1))) ?
	    TRUE : FALSE;
        w = ec->down_btn;
        DO_SET_SENSITIVE
        w = ec->down_mi;
        DO_SET_SENSITIVE

#undef DO_SET_SENSITIVE
}

/*
 *	Maps the given editable clist.
 */
void EditCListMap(editclist_struct *ec)
{
        GtkWidget *w;

        if(ec == NULL)
            return;

        w = ec->toplevel;
        if(w == NULL)
            return;

        if(!ec->map_state)
        {
            gtk_widget_show(w);
            ec->map_state = TRUE;
        }
}

/*
 *	Unmaps the given editable clist.
 */
void EditCListUnmap(editclist_struct *ec)
{
        GtkWidget *w;

        if(ec == NULL)
            return;

        w = ec->toplevel;
        if(w == NULL)
            return;

        if(ec->map_state)
        {
            gtk_widget_hide(w);
            ec->map_state = FALSE;
        }
}

/*
 *	Deallocates all resources on the given editable clist and
 *	deallocates the structure itself.
 */
void EditCListDelete(editclist_struct *ec)
{
        GtkWidget **w;


        if(ec == NULL)
            return;

        if(ec->initialized)
        {
#define DO_DESTROY_WIDGET	\
{ if(*w != NULL) { gtk_widget_destroy(*w); *w = NULL; } }

            w = &ec->menu;
	    ec->add_mi = NULL;
            ec->edit_mi = NULL;
            ec->remove_mi = NULL;
            ec->up_mi = NULL;
            ec->down_mi = NULL;
            DO_DESTROY_WIDGET

            w = &ec->clist;
            DO_DESTROY_WIDGET
            w = &ec->add_btn;
            DO_DESTROY_WIDGET
            w = &ec->edit_btn;
            DO_DESTROY_WIDGET
            w = &ec->remove_btn;
            DO_DESTROY_WIDGET
            w = &ec->up_btn;
            DO_DESTROY_WIDGET
            w = &ec->down_btn;
            DO_DESTROY_WIDGET

            w = &ec->toplevel;
            DO_DESTROY_WIDGET

#undef DO_DESTROY_WIDGET
        }

        /* Deallocate structure itself. */
        g_free(ec);
}
