/*
   Linked List functions
   --------------------------------------------------------------------
   VCHE - Virtual Console Hex Editor

   Copyright (C) 1998, 1999 Diego Javier Grigna <diego@grigna.com>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "common.h"

/* File linked list */
struct ll_file *ll_file_head;    /* Pointer to head of the linked list     */
struct ll_file *ll_file_current; /* Pointer to the current file            */
struct ll_file *ll_file_tail;    /* Pointer to the tail of the linked list */
int ll_file_qty;                 /* Quantity of nodes in the linked list   */

struct ll_file *ll_file_browse_curr; /* Current browsing pointer */
struct ll_file *ll_file_browse_top;  /* Top of page              */

int ll_file_browse_pos;              /* Position of the bar      */

struct ll_edit *ll_edit_head;        /* Pointer to head of edit changes ll */


void ll_file_init( void)
{
 ll_file_head    = NULL;
 ll_file_tail    = NULL;
 ll_file_current = NULL;
 ll_file_qty     = 0;
}

struct ll_file *ll_file_alloc( char *filename)
{
 struct ll_file *p;
 
 p = ( struct ll_file *) allocate_mem( sizeof( struct ll_file) + 1);

 p->filename = ( char *) allocate_mem( strlen( filename) + 1);

 if( filename != NULL)
     strcpy( p->filename, filename);

 p->filepos = 0;

 p->next    = NULL;
 p->prev    = NULL;

 return p;
}

void ll_file_add( char *filename)
{
 struct ll_file *p;

 p = ll_file_alloc( filename);

 if( ll_file_head == NULL) {
     ll_file_head = p;
     ll_file_tail = p;
 } else {
     p->prev = ll_file_tail;
     ll_file_tail->next = p;
     ll_file_tail = p;
 }

 ll_file_qty++;
}

void ll_file_insert( char *filename)
{
 if( ll_file_find( filename) == NULL)
     ll_file_add( filename);
}

void ll_file_free( void)
{
 struct ll_file *p, *v;

 p = ll_file_head;

 while( p != NULL) {
        v = p->next;
        if( p->filename != NULL) 
            free( p->filename);
        if( p != NULL)
            free( p);
        p = v;
 }

 ll_file_init();
}

struct ll_file *ll_file_find( char *filename)
{
 struct ll_file *p, *v;

 p = ll_file_head;

 while( p != NULL) {
        v = p->next;
        if( p != NULL) 
            if( !strcmp( filename, p->filename)) return p;
        p = v;
 }

 return NULL;
}

void ll_file_delete( void)
{
 struct ll_file *p;

 if( ll_file_current == NULL)
     return;

 if( ll_file_current->next == NULL && ll_file_current->prev == NULL) {
     free( ll_file_current->filename);
     free( ll_file_current);
     ll_file_init();
     return; 
 }

 p = ll_file_current;

 if( ll_file_current->next == NULL) {
     if( p == ll_file_browse_curr || p == ll_file_browse_top) {
         ll_file_browse_top  = ll_file_head;
         ll_file_browse_curr = ll_file_head;
         ll_file_browse_pos  = 0;
     }

     ll_file_tail    = p->prev;
     ll_file_current = p->prev;
     p->prev->next   = NULL;
     free( p->filename);
     free( p);
     ll_file_qty--;
     return;
 }

 if( ll_file_current->prev == NULL) {
     ll_file_head    = p->next;
     ll_file_current = p->next;
     p->next->prev   = NULL;

     if( p == ll_file_browse_curr || p == ll_file_browse_top) {
         ll_file_browse_top  = ll_file_head;
         ll_file_browse_curr = ll_file_head;
         ll_file_browse_pos  = 0;
     }

     free( p->filename);
     free( p);
     ll_file_qty--;
     return;
 }

 if( p == ll_file_browse_curr || p == ll_file_browse_top) {
     ll_file_browse_top  = ll_file_head;
     ll_file_browse_curr = ll_file_head;
     ll_file_browse_pos  = 0;
 }

 ll_file_current = p->next;

 p->next->prev = p->prev;
 p->prev->next = p->next;

 free( p->filename);
 free( p);
 
 ll_file_qty--;
}

void ll_file_browse( void)
{
 int c;
#if defined(VCHE_RAW) || defined(VCHE_NC)
 char *bottom_line = "Use         to move           ENTER to select           F10 to ABORT";
 int i;
#endif
#if defined(VCHE_NC)
 WINDOW *w;
 WINDOW *w2;
#endif

#if defined(VCHE_VC) || defined(VCHE_RAW)
 lib_save_screen();
#endif

#if defined(VCHE_VC)
 vc_box( 0, 0, vc_cols, vc_lines - 1, VC_COLOR_BACKCYAN);
 vc_do_title( " File selection ");
 vc_put_bottom_line( "Use \x18 \x19 \x1A \x1B to move          ENTER to select          F10 to ABORT");
#elif defined(VCHE_RAW)
 raw_box( 0, 0, vc_cols, vc_lines - 1, VC_COLOR_BACKCYAN);
 raw_do_title( " File selection ");
 raw_put_bottom_line( bottom_line);

 i = ( vc_cols - strlen( bottom_line)) / 2;

 term_enable_graf_map();
 raw_putc( i + 50, vc_lines - 1, RAW_LINE_VERT);
 raw_putc( i + 24, vc_lines - 1, RAW_LINE_VERT);
 raw_putc( i +  4, vc_lines - 1, RAW_UARROW   );
 raw_putc( i +  6, vc_lines - 1, RAW_DARROW   );
 raw_putc( i +  8, vc_lines - 1, RAW_RARROW   );
 raw_putc( i + 10, vc_lines - 1, RAW_LARROW   );
 term_disable_graf_map();

#elif defined(VCHE_NC)
 w2 = nc_box( 0, 0, vc_cols, vc_lines, VC_COLOR_BACKCYAN);

 nc_put_bottom_line( w2, bottom_line);

 wattrset( w2, VC_COLOR_BACKGRAY);

 i = ( vc_cols - strlen( bottom_line)) / 2;

 mvwaddch( w2, vc_lines - 1, i + 50, ACS_VLINE );
 mvwaddch( w2, vc_lines - 1, i + 24, ACS_VLINE );
 mvwaddch( w2, vc_lines - 1, i +  4, ACS_UARROW);
 mvwaddch( w2, vc_lines - 1, i +  6, ACS_DARROW);
 mvwaddch( w2, vc_lines - 1, i +  8, ACS_RARROW);
 mvwaddch( w2, vc_lines - 1, i + 10, ACS_LARROW);

 wrefresh( w2);

 w = nc_box( 0, 0, vc_cols, vc_lines - 1, VC_COLOR_BACKCYAN);
 nc_do_title( w, " File selection ");
#endif

 while( 1) {
        ll_file_browse_show(
#if defined(VCHE_NC)
 w
#endif
                            );

        c = key_get();

        switch( c) {
                case '\n': 
#if defined(VCHE_VC) || defined(VCHE_RAW)
                           lib_load_screen();
#elif defined(VCHE_NC)
                           delwin( w);
                           delwin( w2);
#endif
                           ll_file_current->filepos = fpos;
                           ll_file_current = ll_file_browse_curr;
                           file_open( ll_file_current->filename);
                           fpos = lseek( fd, (long) ll_file_current->filepos, SEEK_SET);
                           draw_filename( ll_file_current->filename);
                           hex_showfile();
                           lib_flush();
                           return;
                case 'q' : 
                case 'Q' :
                case VC_KEY_F10  : 
#if defined(VCHE_VC) || defined(VCHE_RAW)
                                   lib_load_screen(); 
#elif defined(VCHE_NC)
                                   delwin( w);
                                   delwin( w2);
                                   refresh();
#endif
                                   return;
                case VC_KEY_LEFT : 
                case VC_KEY_UP   : if( ll_file_browse_curr->prev == NULL) {
                                       term_beep();
                                       break;
                                   }
                                   ll_file_browse_curr=ll_file_browse_curr->prev;
                                   ll_file_browse_pos--;
                                   if( ll_file_browse_pos < 0) {
                                       if( ll_file_browse_back( vc_lines - 5) == -1) {
                                           term_beep();
                                           ll_file_browse_pos++;
                                           break;
                                       }
                                       ll_file_browse_pos = vc_lines - 6;
                                   }
                                   break;
                case VC_KEY_DOWN : 
                case VC_KEY_RIGHT: if( ll_file_browse_curr->next == NULL) {
                                       term_beep();
                                       break;
                                   }
                                   ll_file_browse_curr=ll_file_browse_curr->next;
                                   ll_file_browse_pos++;
                                   if( ll_file_browse_pos > vc_lines - 6) {
                                       if( ll_file_browse_next( vc_lines - 5) == -1) {
                                           term_beep();
                                           ll_file_browse_pos--;
                                           break;
                                       }
                                       ll_file_browse_pos = 0;
                                   }
                                   break;
                case VC_KEY_HOME : ll_file_browse_top  = ll_file_head;
                                   ll_file_browse_curr = ll_file_head;
                                   ll_file_browse_pos  = 0;
                                   break;
                case VC_KEY_END  : while( ll_file_browse_next( vc_lines - 5) != -1);
                                   ll_file_browse_curr = ll_file_browse_top;
                                   ll_file_browse_pos  = 0;
                                   break;
                case VC_KEY_PGUP : if( ll_file_browse_back( vc_lines - 5) == -1) {
                                       term_beep();
                                       break;
                                   }
                                   ll_file_browse_pos  = 0;
                                   ll_file_browse_curr = ll_file_browse_top;
                                   break;
                case VC_KEY_PGDN : if( ll_file_browse_next( vc_lines - 5) == -1) {
                                       term_beep();
                                       break;
                                   }
                                   ll_file_browse_pos  = 0;
                                   ll_file_browse_curr = ll_file_browse_top;
                                   break;
        } /* end switch( c) */
 } /* end while( 1) */

}

#if defined(VCHE_VC)

void ll_file_browse_show( void)
{
 struct ll_file *p = ll_file_browse_top;
 char buf[ 2048];
 int len;
 int y = 2;

 while( p != NULL && y < vc_lines - 3) {

        len = strlen( p->filename);

        sprintf( buf, "%-255.255s", p->filename);

        if( p == ll_file_browse_curr)
            vc_color_puts( 1, y, vc_cols - 2, buf,          VC_COLOR_WHITE );
        else
            vc_color_puts( 1, y, vc_cols - 2, buf, VC_COLOR_BACKCYAN );

        if( len > vc_cols - 2)
            vc_color_putc( vc_cols - 2, y, '\x1A', VC_COLOR_ATTENTION );
        y++;
        p = p->next;
 }

 while( y < vc_lines - 3) {
        vc_color_puts( 1, y, vc_cols - 2, space_buffer, VC_COLOR_BACKCYAN);
        y++;
 }

 if( p != NULL) {
     vc_color_putc( 1            , vc_lines - 3, '\x19', VC_COLOR_ATTENTION);
     vc_color_putc( vc_cols - 2, vc_lines - 3, '\x19', VC_COLOR_ATTENTION);
 } else {
     vc_color_putc( 1            , vc_lines - 3, ' ', VC_COLOR_BACKCYAN);
     vc_color_putc( vc_cols - 2, vc_lines - 3, ' ', VC_COLOR_BACKCYAN);
 }

 if( ll_file_browse_top->prev != NULL) {
     vc_color_putc( 1            , 1, '\x18', VC_COLOR_ATTENTION);
     vc_color_putc( vc_cols - 2, 1, '\x18', VC_COLOR_ATTENTION);
 } else {
     vc_color_putc( 1            , 1, ' ', VC_COLOR_BACKCYAN);
     vc_color_putc( vc_cols - 2, 1, ' ', VC_COLOR_BACKCYAN);
 }

}

#elif defined(VCHE_RAW)

void ll_file_browse_show( void)
{
 struct ll_file *p = ll_file_browse_top;
 char buf[ 2048];
 int len;
 int y = 2;

 term_color( VC_COLOR_BACKCYAN);

 while( p != NULL && y < vc_lines - 3) {

        len = strlen( p->filename);

        sprintf( buf, "%-255.255s", p->filename);

        if( p == ll_file_browse_curr) {
            term_color( VC_COLOR_WHITE);
            raw_puts( 1, y, vc_cols - 2, buf);
            term_color( VC_COLOR_BACKCYAN);
        } else
            raw_puts( 1, y, vc_cols - 2, buf);

        if( len > vc_cols - 2) {
            term_color( VC_COLOR_ATTENTION);
            term_enable_graf_map();
            raw_putc( vc_cols - 2, y, RAW_RARROW);
            term_disable_graf_map();
            term_color( VC_COLOR_BACKCYAN);
        }
        y++;
        p = p->next;
 }

 term_color( VC_COLOR_BACKCYAN);
 while( y < vc_lines - 3) {
        raw_puts( 1, y, vc_cols - 2, space_buffer);
        y++;
 }

 if( p != NULL) {
     term_color( VC_COLOR_ATTENTION);
     term_enable_graf_map(); /* Keep it here, because of bug/feature in rxvt
                                we must call term_enable_graf_map() after
                                term_color() */
     raw_putc( 1            , vc_lines - 3, RAW_DARROW);
     raw_putc( vc_cols - 2, vc_lines - 3, RAW_DARROW);
 } else {
     term_color( VC_COLOR_BACKCYAN);
     term_enable_graf_map();
     raw_putc( 1            , vc_lines - 3, ' ');
     raw_putc( vc_cols - 2, vc_lines - 3, ' ');
 }

 if( ll_file_browse_top->prev != NULL) {
     term_color( VC_COLOR_ATTENTION);
     term_enable_graf_map();
     raw_putc( 1            , 1, RAW_UARROW);
     raw_putc( vc_cols - 2, 1, RAW_UARROW);
 } else {
     term_color( VC_COLOR_BACKCYAN);
     term_enable_graf_map();
     raw_putc( 1            , 1, ' ');
     raw_putc( vc_cols - 2, 1, ' ');
 }
 term_disable_graf_map();

}

#elif defined(VCHE_NC)

void ll_file_browse_show( WINDOW *w)
{
 struct ll_file *p = ll_file_browse_top;
 char buf[ 2048];
 int len;
 int y = 2;

 while( p != NULL && y < vc_lines - 3) {

        len = strlen( p->filename);

        sprintf( buf, "%-255.255s", p->filename);

        if( p == ll_file_browse_curr)
            nc_color_wputs( w, 1, y, vc_cols - 2, buf, VC_COLOR_WHITE    );
        else
            nc_color_wputs( w, 1, y, vc_cols - 2, buf, VC_COLOR_BACKCYAN );

        if( len > vc_cols - 2)
            nc_color_wputc( w, vc_cols - 2, y, ACS_RARROW, VC_COLOR_ATTENTION );
        y++;
        p = p->next;
 }

 while( y < vc_lines - 3) {
        nc_color_wputs( w, 1, y, vc_cols - 2, space_buffer, VC_COLOR_BACKCYAN);
        y++;
 }

 if( p != NULL) {
     nc_color_wputc( w, 1       , vc_lines - 3, ACS_DARROW, VC_COLOR_ATTENTION);
     nc_color_wputc( w, vc_cols - 2, vc_lines - 3, ACS_DARROW, VC_COLOR_ATTENTION);
 } else {
     nc_color_wputc( w, 1       , vc_lines - 3, ' ', VC_COLOR_BACKCYAN);
     nc_color_wputc( w, vc_cols - 2, vc_lines - 3, ' ', VC_COLOR_BACKCYAN);
 }

 if( ll_file_browse_top->prev != NULL) {
     nc_color_wputc( w, 1       , 1, ACS_UARROW, VC_COLOR_ATTENTION);
     nc_color_wputc( w, vc_cols - 2, 1, ACS_UARROW, VC_COLOR_ATTENTION);
 } else {
     nc_color_wputc( w, 1       , 1, ' ', VC_COLOR_BACKCYAN);
     nc_color_wputc( w, vc_cols - 2, 1, ' ', VC_COLOR_BACKCYAN);
 }

}

#endif

int ll_file_browse_next( int nitems)
{
 struct ll_file *p = ll_file_browse_top;
 int pos = 0;

 while( p != NULL && pos++ < nitems)
        p = p->next;

 if( p == NULL) return -1;

 ll_file_browse_top = p;

 return 1;
}

int ll_file_browse_back( int nitems)
{
 struct ll_file *p = ll_file_browse_top;
 int pos = 0;

 while( p != NULL && pos++ < nitems)
        p = p->prev;

 if( p == NULL) return -1;

 ll_file_browse_top = p;

 return 1;
}

struct ll_edit *ll_edit_alloc( char byte)
{
 struct ll_edit *p;

 p = ( struct ll_edit *) allocate_mem( sizeof( struct ll_edit) + 1);

 p->byte = byte;

 p->filepos = 0;

 p->next    = NULL;

 return p;
}

void ll_edit_add( char byte, long filepos, struct ll_edit *tmp)
{
 struct ll_edit *p = ll_edit_alloc( byte);

 p->filepos = filepos;

 if( ll_edit_head == NULL) {
     ll_edit_head = p;
     ll_edit_head->next = NULL;
     return;
 }

 if( filepos < ll_edit_head->filepos) {
     p->next = ll_edit_head;
     ll_edit_head = p;
     return;
 }

 p->next = tmp->next;
 tmp->next = p;
}

void ll_edit_free( void)
{
 struct ll_edit *p = ll_edit_head;

 while( p != NULL) {
        free( p);
        p = p->next;
 }

 ll_edit_head = NULL;
}

