/*
 * UFRaw - Unidentified Flying Raw converter for digital camera images
 * by udi Fuchs
 *
 * ufraw.h - Common definitions for UFRaw.
 * 
 * based on the gimp plug-in by Pawel T. Jochym jochym at ifj edu pl,
 *
 * based on the gimp plug-in by Dave Coffin
 * http://www.cybercom.net/~dcoffin/
 *
 * UFRaw is licensed under the GNU General Public License.
 * It uses "dcraw" code to do the actual raw decoding.
 */

#ifndef _UFRAW_H
#define _UFRAW_H

#define MAXSMPL 0xFFFF /* Max input sample */
#define MAXOUT 255 /* Max output sample */

#define max_curves 20
#define max_profiles 20
#define max_path 200
#define max_name 80

enum { preserve_wb, camera_wb, auto_wb };
enum { rgb_histogram, r_g_b_histogram, luminosity_histogram, value_histogram,
       saturation_histogram };
enum { full_interpolation, four_color_interpolation, quick_interpolation,
       half_interpolation};
enum { load_preserve, load_default, load_auto };
enum { gamma_curve, log_curve, linear_curve, camera_curve };
enum { in_profile, out_profile, profile_types};
enum { raw_expander, wb_expander, color_expander,
       curve_expander, live_expander };
enum { ppm8_type, ppm16_type, tiff8_type, tiff16_type, jpeg_type };

typedef struct {
    char *name;
    float red, blue;
} wb_data;

typedef struct {
    int rgbMax;
    int max;
    int rgbWB[4];
    float exposure, gamma;
    void *profile[2];
    char file[2][max_path];
    int intent;
    gboolean updateTransform;
    void *colorTransform;
    float saturation;
    guint16 gammaCurve[0x10000];
    guint16 saturationCurve[0x10000];
    guint16 toneCurve[0x10000];
} developer_data;

typedef guint16 image_type[4];

typedef struct {
    float gamma, contrast, saturation, brightness, black, shadow, depth;
    char name[max_name];
    char file[max_path];
    int indexSize, curveSize;
    struct {guint16 index; guint8 value;} curve[257];
} curve_data;

typedef struct {
    char name[max_name];
    char file[max_path];
    char productName[max_name];
    float gamma;
    int curve;
} profile_data;

/* cfg holds infomation that is kept for the next image load */
typedef struct {
    int size, version;
    char inputFilename[max_path], outputFilename[max_path];
    int wbLoad, curveLoad, exposureLoad, type, compression;
    int wb, spotSize, histogram, expander[5], interpolation, shrink;
    gfloat temperature, green, exposure;
    gboolean unclip, overExp, underExp, overwrite, losslessCompress;
    int curveIndex, curveCount;
    curve_data curve[max_curves];
    char curvePath[max_path];
    int profileIndex[profile_types], profileCount[profile_types];
    profile_data profile[profile_types][max_profiles];
    int intent;
    char profilePath[max_path];
} cfg_data;

typedef struct {
    char filename[max_path];
    int height, width, trim, rgbMax;
    float preMul[4];
    image_type *image;
    void *raw;
    developer_data *developer;
    cfg_data *cfg;
    int gimpImage;
} image_data;

extern const cfg_data cfg_default;
extern const wb_data wb_preset[];
extern const int wb_preset_count;
extern const char raw_ext[];
extern const char *file_type[];

/* prototypes for functions in ufraw_ufraw.c */
image_data *ufraw_open(char *filename);
int ufraw_config(image_data *image, cfg_data *cfg);
int ufraw_load_raw(image_data *image, int interactive);
int ufraw_convert_image(image_data *image, image_data *rawImage);
void ufraw_close(image_data *image);
int ufraw_set_wb(image_data *image);
void ufraw_auto_adjust(image_data *image);

/* prototypes for functions in ufraw_preview.c */
int ufraw_preview(image_data *image, int plugin, int (*write_func)());
void ufraw_messenger(char *message, void *parentWindow);
void preview_progress(char *text, float progress);

/* prototypes for functions in ufraw_routines.c */
void set_curve(guint16 *curve, int maxValue, int curveType, curve_data *cp);
void curve_convert(curve_data *cp, int curveSize, guint8 *curve);
int curve_load(curve_data *cp, char *filename);
developer_data *developer_init();
void developer_destroy(developer_data *d);
void developer_profile(developer_data *d, int type, profile_data *p);
void developer_prepare(developer_data *d, int rgbMax, float exposure,
    int unclip, float temperature, float green, float pre_mul[4],
    profile_data *in, profile_data *out, int intent,
    float saturation, int curveType, curve_data *curve);
inline void develope(void *po, guint16 pix[4], developer_data *d, int mode,
    guint16 *buf, int count);
void pixbuf_reverse(int x, int y, guchar *pixbuf, int width, int height,
    int rowstride);
void RGB_to_temperature(float *rgb, float *temperature, float *green);
int load_configuration(cfg_data *c);
int save_configuration(cfg_data *c, char *buf, int bufSize);

/* status numbers from dcraw and ufraw */
#define UFRAW_SUCCESS 0
#define UFRAW_DCRAW_ERROR 1 /* General dcraw unrecoverable error */
#define UFRAW_DCRAW_UNSUPPORTED 2
#define UFRAW_DCRAW_NO_CAMERA_WB 3
#define UFRAW_DCRAW_VERBOSE 4
#define UFRAW_DCRAW_OPEN_ERROR 5
#define UFRAW_ERROR 100
#define UFRAW_VERBOSE 101
#define UFRAW_OPEN_ERROR 102
#define UFRAW_CANCEL 103
#define UFRAW_ABORT_SAVE 104
#define UFRAW_NO_CONFIG 105 /* Couldn't open configuration file (I) */
#define UFRAW_RC_VERSION 106 /* Mismatch in version from .ufrawrc */
#define UFRAW_RC_ERROR 107 /* General error in .ufrawrc parsing */
#define UFRAW_WARNING 108
#define UFRAW_MESSAGE 109
#define UFRAW_GET_MESSAGE 200 /* Return the message buffer */
#define UFRAW_REPORT 201 /* Report previews messages */
#define UFRAW_CLEAN 202 /* Clean the message buffer */
#define UFRAW_SET_PARENT 203 /* Set parent window for message dialog */
#define UFRAW_LCMS_WARNING 0x1000
#define UFRAW_LCMS_RECOVERABLE 0x2000
#define UFRAW_LCMS_ABORTED 0x3000

char *ufraw_message_handler(int code, char *message);
#define ufraw_message(code, format, ...) {\
    char uf_tmp_msg[200];\
    if (format!=NULL) {\
        g_snprintf (uf_tmp_msg, 200, format, ## __VA_ARGS__);\
        ufraw_message_handler(code, uf_tmp_msg);\
    } else\
    ufraw_message_handler(code, NULL);\
}

#endif /*_UFRAW_H*/
