/*
 * font.c - Internationalization font module
 *
 * Copyright (c) 2001,2002 Seiichi SATO <ssato@sh.rim.or.jp>
 *
 * licensed under the GPL.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <X11/Xlib.h>
#include <X11/xpm.h>
#include <X11/extensions/shape.h>

#include <stdio.h>

#if STDC_HEADERS
# include <stdlib.h>
# include <stddef.h>
#else
# if HAVE_STDLIB_H
#  include <stdlib.h>
# endif
#endif

#if HAVE_STRING_H
# if !STDC_HEADERS && HAVE_MEMORY_H
#  include <memory.h>
# endif
# include <string.h>
#endif
#if HAVE_STRINGS_H
# include <strings.h>
#endif


#if TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif

#if HAVE_UNISTD_H
# include <unistd.h>
#else
/* We are in trouble. */
#endif

#if HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif

#if HAVE_X11_XLOCALE_H
# include <X11/Xlocale.h>
#else
#  if HAVE_LOCALE_H
#    include <locale.h>
#  endif
#endif

#include "font.h"
#include "mem.h"

/* XLFD pattern matching */
static char
*xlfd_get_element(const char *xlfd, int index)
{
    const char *p = xlfd;
    char *buf;

    while (*p != 0) {
	if (*p == '-' && --index == 0) {
	    const char *end = strchr(p + 1, '-');
	    size_t len;
	    if (end == 0)
		end = p + strlen(p);
	    len = end - (p + 1);
	    buf = malloc_wrap(len);
	    memcpy(buf, p + 1, len);
	    buf[len] = 0;
	    return buf;
	}
	p++;
    }

    buf = malloc_wrap(sizeof(char) * 2);
    strcpy(buf, "*");
    return buf;
}

/* XLFD pattern matching */
static char
*generalize_xlfd(const char *xlfd)
{
    char *buf;
    int len;
    char *weight = xlfd_get_element(xlfd, 3);
    char *slant = xlfd_get_element(xlfd, 4);
    char *pxlsz = xlfd_get_element(xlfd, 7);

#define Xstrlen(A) ((A)?strlen(A):0)
    len = Xstrlen(xlfd) + Xstrlen(weight) + Xstrlen(slant) +
	Xstrlen(pxlsz) * 2 + 56;
#undef Xstrlen

    buf = malloc_wrap(len + 1);
    sprintf(buf, "%s,-*-*-%s-%s-*-*-%s-*-*-*-*-*-*-*,"
	    "-*-*-*-*-*-*-%s-*-*-*-*-*-*-*,*",
	    xlfd, weight, slant, pxlsz, pxlsz);

    free(pxlsz);
    free(slant);
    free(weight);

    return buf;
}

FontI18N *CreateFont(Display * dpy, char *fontname)
{
    FontI18N *font;
    char **missing_charset_list_return;
    int missing_charset_count_return;
    char *def_string_return;
    XFontSetExtents *font_extents;
    char *base_fontname;
    int i;
    char *locale;
    char *ptr;

    font = malloc_wrap(sizeof(FontI18N));

#if defined(HAVE_SETLOCALE) || defined(HAVE_XSETLOCALE)
    locale = setlocale(LC_ALL, "");

    if (locale == NULL || !strcmp(locale, "C") || !strcmp(locale, "POSIX")
	|| !XSupportsLocale())
	font->use_fontset = False;
    else
	font->use_fontset = True;

    if (font->use_fontset) {
	font->fontset = XCreateFontSet(dpy, fontname,
				       &missing_charset_list_return,
				       &missing_charset_count_return,
				       &def_string_return);

	if (missing_charset_count_return != 0) {
	    /* for non-iso8859-1 language and iso8859-1 specification
	       (this fontset is only for pattern analysis) */
	    if (font->fontset == NULL) {
		XFreeStringList(missing_charset_list_return);
		setlocale(LC_CTYPE, "C");
		font->fontset = XCreateFontSet(dpy, fontname,
					       &missing_charset_list_return,
					       &missing_charset_count_return,
					       &def_string_return);
		setlocale(LC_CTYPE, "");
	    }

	    base_fontname = generalize_xlfd(fontname);

	    if (missing_charset_count_return != 0)
		XFreeStringList(missing_charset_list_return);
	    if (font->fontset != NULL)
		XFreeFontSet(dpy, font->fontset);

	    if ((font->fontset = XCreateFontSet(dpy, base_fontname,
						&missing_charset_list_return,
						&missing_charset_count_return,
						&def_string_return)) ==
		NULL) {
		fprintf(stderr, "unable to open fontset \"%s\"\n",
			fontname);
		exit(1);
	    }

	    free(base_fontname);
	}

	for (i = 0; i < missing_charset_count_return; i++) {
	    printf("warning: font for charset %s lacking.\n",
		   missing_charset_list_return[i]);
	}

	font->fontstruct = NULL;
	font_extents = XExtentsOfFontSet(font->fontset);
	font->height = font_extents->max_logical_extent.height;
	font->y = font_extents->max_logical_extent.y * -1;
	return font;
    }
#endif

    if ((ptr = strchr(fontname, ','))) {
	base_fontname = malloc_wrap(ptr - fontname + 1);
	strncpy(base_fontname, fontname, ptr - fontname);
	base_fontname[ptr - fontname] = '\0';
    } else {
	base_fontname = malloc_wrap(sizeof(char) * (strlen(fontname) + 1));
	strcpy(base_fontname, fontname);
    }

    if ((font->fontstruct = XLoadQueryFont(dpy, base_fontname)) == NULL) {
	printf("warning: could not load font %s. Trying fixed.\n",
	       fontname);
	if ((font->fontstruct = XLoadQueryFont(dpy, "fixed")) == NULL) {
	    fprintf(stderr, "can't open font %s\n", fontname);
	    exit(1);
	}
    }

    free(base_fontname);

    font->fontset = NULL;
    font->height = font->fontstruct->ascent + font->fontstruct->descent;
    font->y = font->fontstruct->ascent;
    return font;
}

void
DrawString(Display * dpy, Drawable d, FontI18N * font, GC gc, int x, int y,
	   char *str, int len)
{
    if (font->use_fontset) {
	XmbDrawString(dpy, d, font->fontset, gc, x, y, str, len);
    } else {
	XDrawString(dpy, d, gc, x, y, str, len);
    }
}

int
GetStringWidth(FontI18N * font, char *str, int len)
{
    if (font->use_fontset) {
	return XmbTextEscapement(font->fontset, str, len);
    } else {
	return XTextWidth(font->fontstruct, str, len);
    }
}
