/***************************************************************************
 *   Copyright (C) 2003 by Stephen Allewell                                *
 *   stephen@mirramar.fsnet.co.uk                                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#include <klocale.h>
#include "schemeparser.h"
#include "flossscheme.h"
#include "floss.h"

const char* validTags[] =
{
  "flossScheme",
  "title",
  "floss",
  "name",
  "description",
  "color",
  "red",
  "green",
  "blue",
  0
};

SchemeParser::SchemeParser()
  : m_errorCode(0),
    m_scheme(0)
{
}

SchemeParser::~SchemeParser()
{
  // nothing to do here
  // m_scheme will be deleted by the calling code if necessary
}

FlossScheme* SchemeParser::flossScheme()
{
  return m_scheme;
}

bool SchemeParser::startDocument()
{
  m_scheme = new FlossScheme;
  return (m_scheme)?true:false;
}

bool SchemeParser::startElement(const QString&, const QString&, const QString& qName, const QXmlAttributes& )
{
  QString name = qName.simplifyWhiteSpace();
  int tag;
  for (tag = 0 ; validTags[tag] && name != validTags[tag] ; tag++) ;
  if (validTags[tag])
  {
    m_elements.push(new QString(name));
    return true;
  }
  // invalid tag found
  m_errorCode = 3;
  return false;
}

bool SchemeParser::characters(const QString& d)
{
  QString data = d.simplifyWhiteSpace();
  if (data.isEmpty()) return true;
  if (*m_elements.top() == "title")
    m_scheme->setSchemeName(data);
  else
    m_flossMap.insert(*m_elements.top(),data);
  if (*m_elements.top() == "red" || *m_elements.top() == "green" || *m_elements.top() == "blue")
    if (data.toInt() > 255)
    {
      // color value is not valid
      m_errorCode = 4;
      return false;
    }
  return true;
}

bool SchemeParser::endElement(const QString&, const QString&, const QString& qName)
{
  QString name = qName.simplifyWhiteSpace();
  QString *s = m_elements.pop();
  if (*s == name)
  {
    if (name == "floss")
    {
      if (m_flossMap.contains("name") && m_flossMap.contains("description") && m_flossMap.contains("red") && m_flossMap.contains("green") && m_flossMap.contains("blue"))
      {
        m_scheme->addFloss(new Floss(m_flossMap["name"],m_flossMap["description"],QColor(m_flossMap["red"].toInt(),m_flossMap["green"].toInt(),m_flossMap["blue"].toInt())));
        m_flossMap.clear();
      }
      else
      {
        // not all elements defined for a floss
        m_errorCode = 1;
      }
    }
  }
  else
    // mismatched tags
    m_errorCode = 2;

  delete s;

  if (m_errorCode)
    return false;
  else
    return true;
}

QString SchemeParser::errorString()
{
  QString error;
  switch (m_errorCode)
  {
    case 0:
      error = i18n("No error");
      break;
    case 1:
      error = i18n("Floss not completely defined");
      break;
    case 2:
      error = i18n("Unmatched element tags");
      break;
    case 3:
      error = i18n("Unknown element tag");
      break;
    case 4:
      error = i18n("Color value is invalid");
      break;
    default:
      error = i18n("Unknown error code");
      break;
  }
  return error;
}
