/***************************************************************************
 *   Copyright (C) 2003 by Stephen Allewell                                *
 *   stephen@mirramar.fsnet.co.uk                                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#include <qstringlist.h>
#include <qdir.h>
#include <qiconview.h>
#include <qcursor.h>
#include <qlabel.h>
#include <qtooltip.h>
#include <qpopupmenu.h>
#include <qpushbutton.h>
#include <qmap.h>
#include <qcolor.h>
#include <kapplication.h>
#include <kstandarddirs.h>
#include <klistview.h>
#include <klocale.h>
#include <klineeditdlg.h>
#include <kmessagebox.h>
#include "patternlibrarydialog.h"
#include "kxstitchmimesource.h"

PatternLibraryDialog::PatternLibraryDialog(QWidget *parent, KXStitchMimeSource* data)
  : PatternLibraryDlg(parent,"PatternLibraryDlg",true),
    m_patternData(data),
    m_listViewItem(0),
    m_tooltip(0)
{
  fillLibraryList();
  fillPatternList();

  if (m_patternData)
    InsertButton->setText(i18n("Add"));
  else
    InsertButton->setText(i18n("Insert"));

  connect(LibraryList,SIGNAL(contextMenu(KListView*,QListViewItem*,const QPoint&)),this,SLOT(showContextMenu(KListView*,QListViewItem*,const QPoint&)));
  connect(LibraryList,SIGNAL(selectionChanged(QListViewItem*)),this,SLOT(libraryChanged(QListViewItem*)));
  connect(PatternPreviews,SIGNAL(selectionChanged(QIconViewItem*)),this,SLOT(patternChanged(QIconViewItem*)));
  connect(PatternPreviews,SIGNAL(doubleClicked(QIconViewItem*)),this,SLOT(patternDoubleClicked(QIconViewItem*)));
  connect(PatternPreviews,SIGNAL(onItem(QIconViewItem*)),this,SLOT(showToolTip(QIconViewItem*)));
  connect(PatternPreviews,SIGNAL(onViewport()),this,SLOT(removeToolTip()));
  connect(HelpButton,SIGNAL(clicked()),this,SLOT(contextHelp()));
}

PatternLibraryDialog::~PatternLibraryDialog()
{
}

KXStitchMimeSource* PatternLibraryDialog::pattern()
{
  return m_patternData;
}

void PatternLibraryDialog::showContextMenu(KListView*,QListViewItem* item,const QPoint&)
{
  if (m_patternData)
  {
    // only require a context menu if adding a new pattern to the library
    m_listViewItem = item;
    QPopupMenu* context = new QPopupMenu;
    context->insertItem(i18n("New Catagory"),this,SLOT(newCatagory()));
    context->popup(QCursor::pos());
  }
}

void PatternLibraryDialog::accept()
{
  delete m_tooltip;
  if (m_patternData)
  {
    // store pattern data in writable directory for catagory
    QListViewItem* item = LibraryList->selectedItem();
    QString path;
    if (item)
    {
      for (int i = 0 ; !(path = item->text(i)).isNull() ; i++)
      {
        QFileInfo dir(path);
        if (dir.isWritable())
          break;
      }
      if (path.isNull())
      {
        path = item->text(1);
        int i = path.find("library");
        path.replace(0,i,"");
        path = kapp->dirs()->saveLocation("appdata",path);
      }
      path += "kxstitch.library";
      QFile library(path);
      if (library.open(IO_ReadWrite))
      {
        Q_UINT32 patternCount = 0;
        QDataStream stream(&library);
        QByteArray data;
        if (!stream.atEnd())
          stream >> patternCount;
        patternCount++;
        library.reset();
        stream << patternCount; // increment the number of patterns stored
        library.flush();
        library.at(library.size());
        data = m_patternData->encodedData("application/kxstitch");
        stream << qChecksum(data.data(),data.size());
        stream << qCompress(data);
        library.flush();
        library.close();
      }
    }
  }
  else
  {
    // read pattern data related to current selection
    QIconViewItem* item = PatternPreviews->currentItem();
    if (item)
    {
      QString path = item->key();    // file path in format 'path/kxstitch.library##index'
      QIODevice::Offset offset = path.section("##",1,1).toULong();
      path = path.section("##",0,0);

      QFile library(path);
      if (library.open(IO_ReadOnly))
      {
        QDataStream stream(&library);
        Q_UINT16 checksum;
        QByteArray compressed;
        QByteArray data;
        stream.device()->at(offset);
        stream >> checksum;
        stream >> compressed;
        data = qUncompress(compressed);
        if (checksum == qChecksum(data.data(),data.size()))
          m_patternData = new KXStitchMimeSource(data);
        else
          KMessageBox::detailedSorry(this,i18n("Unable to read the pattern."),i18n("KXStitch could not read a pattern\nfrom %1\nIt appears to be corrupt.").arg(path));
        library.close();
      }
    }
  }
  QDialog::accept();
}

void PatternLibraryDialog::contextHelp()
{
  kapp->invokeHelp("PatternLibraryDialog");
}

void PatternLibraryDialog::libraryChanged(QListViewItem* item)
{
  if (item)
  {
    if (m_patternData)
      InsertButton->setEnabled(true);
    else
      fillPatternList(item);
  }
}

void PatternLibraryDialog::patternChanged(QIconViewItem*)
{
  InsertButton->setEnabled(true);
}


void PatternLibraryDialog::patternDoubleClicked(QIconViewItem*)
{
  accept();
}

void PatternLibraryDialog::newCatagory()
{
  QString         catagory;
  bool            ok = false;

  while (!ok)
  {
    QListViewItem*  item;
    catagory = KLineEditDlg::getText(i18n("Catagory"),"",&ok,this);
    if (!ok)
      // user pressed cancel
      break;
    if (m_listViewItem)
    {
      for (item = m_listViewItem->firstChild() ; item ; item = item->nextSibling())
        if (item->text(0) == catagory)
          break;
    }
    else
      item = LibraryList->findItem(catagory,0,Qt::ExactMatch);

    if (item)
    {
      KMessageBox::sorry(this,i18n("This catagory already exists."),i18n("Catagory Exists"));
      ok = false;
    }
  }
  if (ok)
  {
    QFileInfo fileInfo;
    KListViewItem* item;
    if (m_listViewItem)
      item = new KListViewItem(m_listViewItem,catagory);
    else
      item = new KListViewItem(LibraryList,catagory);

    QString path;
    if (m_listViewItem)
    {
      for (int i = 1 ; (path = m_listViewItem->text(i)) != QString::null ; i++)
      {
        fileInfo.setFile(path);
        if (fileInfo.isWritable())
        {
          break;
        }
      }
      if (path == QString::null)
      {
        path = m_listViewItem->text(1);
        int i = path.find("library");
        path.replace(0,i,"");
        path = kapp->dirs()->saveLocation("appdata",path);
        fileInfo.setFile(path);
      }
    }
    else
    {
      path = kapp->dirs()->saveLocation("appdata","library/");
      fileInfo.setFile(path);
    }

    if (fileInfo.dir().mkdir(catagory))
    {
      path += catagory;
      path += "/";
      item->setText(1,path);
    }

    LibraryList->ensureItemVisible(item);
  }
  m_listViewItem = 0;
}

void PatternLibraryDialog::showToolTip(QIconViewItem* item)
{
  delete m_tooltip;
  m_tooltip = 0;
  if (item)
  {
    QString tip = "<table cellspacing=0 cellpadding=0>";
    const char* start = "<tr><td>";
    const char* mid = "</td><td>";
    const char* end = "</td></tr>";
    QString key = item->key();
    QString scheme = key.section("##",2,2);
    QString size = key.section("##",3,3);
    QString colors = key.section("##",4,4);
    tip += start + i18n("Scheme:") + mid + scheme + end;
    tip += start + i18n("Stitches:") + mid + size + end;
    tip += start + i18n("Colors:") + mid + colors + end;
    tip += "</table>";
    m_tooltip = new QLabel(tip,0,"PreviewToolTip",WStyle_StaysOnTop | WStyle_Customize | WStyle_NoBorder | WStyle_Tool);
    m_tooltip->setFrameStyle(QFrame::Plain | QFrame::Box);
    m_tooltip->setLineWidth(1);
    m_tooltip->setAlignment(AlignLeft | AlignTop);
    m_tooltip->move(QCursor::pos()+QPoint(10,10));
    m_tooltip->adjustSize();
    m_tooltip->setFont(QToolTip::font());
    m_tooltip->setPalette(QToolTip::palette(),true);
    m_tooltip->show();
  }
}

void PatternLibraryDialog::removeToolTip()
{
  delete m_tooltip;
  m_tooltip = 0;
}

void PatternLibraryDialog::fillLibraryList()
{
  QStringList libraries = kapp->dirs()->findDirs("appdata","library");
  QDir        dir;
  for (QStringList::Iterator it = libraries.begin() ; it != libraries.end() ; ++it)
  {
    QString path = *it;
    dir.setPath(path);
    const QFileInfoList* entries = dir.entryInfoList();
    QPtrListIterator<QFileInfo> files(*entries);
    for (QFileInfo* file = 0 ; (file = files.current()); ++files)
    {
      if (file->isDir() && file->fileName() != "." && file->fileName() != "..")
      {
        QString subPath = QString("%1%2/").arg(path).arg(file->fileName());
        KListViewItem* item = (KListViewItem*)(LibraryList->findItem(file->fileName(),0,Qt::ExactMatch));
        if (item)
        {
          // root node already exists, add this path to it.
          for (int i = 1 ; i ; i++)
            // find the first empty column
            if (item->text(i) == QString::null)
            {
              item->setText(i,subPath);
              break;
            }
        }
        else
          item = new KListViewItem(LibraryList,file->fileName(),subPath);

        recurseLibrary(item,subPath);
      }
    }
  }
}

void PatternLibraryDialog::recurseLibrary(QListViewItem* parentItem, QString path)
{
  QDir  dir(path);
  const QFileInfoList* entries = dir.entryInfoList();
  QPtrListIterator<QFileInfo> files(*entries);
  for (QFileInfo* file = 0 ; (file = files.current()) ; ++files)
  {
    if (file->isDir() && file->fileName() != "." && file->fileName() != "..")
    {
      QString subPath = QString("%1%2/").arg(path).arg(file->fileName());
      QListViewItem* item = parentItem->firstChild();
      while (item)
      {
        if (item->text(0) == file->fileName())
          break;
        else
          item = item->nextSibling();
      }
      // item will either point to a child item with this name or will be null
      if (item)
      {
        // root node already exists, add this path to it.
        for (int i = 1 ; i ; i++)
          // find the first empty column
          if (item->text(i) == QString::null)
          {
            item->setText(i,subPath);
            break;
          }
      }
      else
        item = new KListViewItem(parentItem,file->fileName(),subPath);

      recurseLibrary(item,subPath);
    }
  }
}

void PatternLibraryDialog::fillPatternList(QListViewItem* item)
{
  PatternPreviews->clear();
  if (m_patternData)
  {
    // PatternPreviews should only show the image being added
    PatternPreviews->setSelectionMode(QIconView::NoSelection);
    QImage img;
    QPixmap adding;
    img.loadFromData(m_patternData->encodedData("image/png"));
    adding.convertFromImage(img.smoothScale(48,48,QImage::ScaleMax));
    new QIconViewItem(PatternPreviews,0,adding);
  }
  else
  {
    if (item)
    {
      QString path;
      for (int i = 1 ; !(path = item->text(i)).isNull() ; i++)
      {
        path += "/kxstitch.library";
        QFile file(path);
        if (file.open(IO_ReadOnly))
        {
          QDataStream stream(&file);
          Q_UINT32 patterns;
          stream >> patterns;
          while (patterns--)
          {
            QIODevice::Offset filePos = stream.device()->at();
            Q_UINT16 checksum;
            QByteArray data;
            QByteArray compressed;
            stream >> checksum;
            stream >> compressed;
            data = qUncompress(compressed);
            if (checksum == qChecksum(data.data(),data.size()))
            {
              QDataStream stream(data,IO_ReadOnly);
              QString scheme;
              Q_INT32 width;
              Q_INT32 height;
              QMap<Q_UINT32,int> colors;
              QPoint start;
              QPoint end;
              Q_UINT32 color;       // stored as a QColor, but read as a Q_UINT32 so it can be used as a QMap key
              Q_INT32 backstitches;
              Q_INT32 knots;
              Q_INT8 count;
              Q_INT8 type;
              stream >> scheme >> width >> height;
              for (int dy = 0 ; dy < height ; dy++)
                for (int dx = 0 ; dx < width ; dx++)
                {
                  stream >> count;
                  while (count--)
                  {
                    stream >> type >> color;
                    colors[color]++;
                  }
                }
              stream >> backstitches;
              while (backstitches--)
              {
                stream >> start >> end >> color;
                colors[color]++;
              }
              stream >> knots;
              while (knots--)
              {
                stream >> start >> color;
                colors[color]++;
              }
              KXStitchMimeSource mimeSource(data);
              QPixmap adding;
              QImage img;
              img.loadFromData(mimeSource.encodedData("image/png"));
              adding.convertFromImage(img.smoothScale(64,64,QImage::ScaleMax));
              QIconViewItem* newItem = new QIconViewItem(PatternPreviews,0,adding);
              newItem->setKey(QString("%1##%2##%3##%4 x %5##%6").arg(path).arg(filePos).arg(scheme).arg(width).arg(height).arg(colors.count()));
            }
          }
          file.close();
        }
      }
    }
  }
}
