/***************************************************************************
 *   Copyright (C) 2003 by Stephen Allewell                                *
 *   stephen@mirramar.fsnet.co.uk                                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#ifndef KXSTITCHDOC_H
#define KXSTITCHDOC_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qobject.h>
#include <qstring.h>
#include <qptrlist.h>
#include <qptrvector.h>
#include <qpointarray.h>
#include <kurl.h>
#include "kxstitch.h"
#include "floss.h"
#include "stitch.h"
#include "patterncanvas.h"
#include <Magick++.h>

class FlossPalette;
class PatternPropertiesDialog;
class KXStitchView;
class KScanDialog;

/** KXStitchDoc provides a document object for a document-view model.
    The KXStitchDoc class provides a document object that can be used in conjunction
    with the classes KXStitchApp and KXStitchView to create a document-view model for
    standard KDE applications based on KApplication and KMainWindow. Thereby, the
    document object is created by the KXStitchApp instance and contains the document
    structure with the according methods for manipulation of the document data by
    KXStitchView objects. Also, KXStitchDoc contains the methods for serialization of
    the document data from and to files.
    @author Stephen P. Allewell
  */
class KXStitchDoc : public QObject
{
  Q_OBJECT
public:
  friend class PreView;
  friend class EditView;
  friend class PaletteView;

/** Constructor
    @param parent Pointer to parent widget
    @param name Internal name for QObject
  */
  KXStitchDoc(QWidget *parent, const char *name=0);
/** Destructor
  */
  ~KXStitchDoc();

/** adds a view to the document which represents the document contents. Usually this is the main view.
    @param view The view to be added
  */
  void addView(KXStitchView *view);
/** removes a view from the list of currently connected views
    @param view The view to be removed
  */
  void removeView(KXStitchView *view);
/** sets the modified flag for the document after a modifying action on the view connected to the document
    @param _m true if document has been modified, false otherwise
  */
  void setModified(bool modified=true);
/** returns if the document is modified or not. Use this to determine if your document needs saving by the user on closing
    @return TRUE if document has been modified
  */
  bool isModified(){ return m_modified; };
/** save modified - asks the user for saving if the document is modified
  */
  bool saveModified();
/** deletes the document's contents
  */
  void deleteContents();
/** initializes the document generally
  */
  bool newDocument();
/** closes the acutal document
  */
  void closeDocument();
/** load a KXStitch format file from the supplied data stream
    @param s QDataStream containing the file to be read
  */
  bool readKXStitchDocument(QDataStream& s);
/** load a PCStitch version 5 format file from the supplied data stream
    @param s QDataStream containing the file to be read
  */
  bool readPCStitch5Document(QDataStream& s);
/** load a string from a PCStitch file, format is a 16 bit value followed by text.
  */
  QString readPCStitchString(QDataStream& s);
/** loads the document by filename and format and emits the updateViews() signal
    @param url The @ref URL of the document
  */
  bool openDocument(const KURL& url, const char* format=0);
/** import an Image as a new document
  */
  bool importImage();
/** scan an image as a new document
  */
  bool scanImage();
/** convert a Magick::Image to a pattern
  */
  void convertImage(const Magick::Image&);
/** saves the document under filename and format
    @return TRUE if document saved FALSE otherwise
    @param url The URL of the document
  */
  bool saveDocument(const KURL& url, const char *format=0);
/** modify the properties of the document
  */
  void fileProperties();
/** returns the KURL of the document
    @return The URL of the document
  */
  const KURL& URL() const;
/** sets the URL of the document
    @param url The URL of the document
  */
  void setURL(const KURL& url);
/** resize the document to w, h
  */
  void resize(int w, int h);
/** extend the pattern
    @param l left margin
    @param t top margin
    @param r right margin
    @param b bottom margin
  */
  void extendPattern(int l, int t, int r, int b);
/** center the pattern to the canvas
  */
  void centerPattern();
/** crop the canvas to the pattern
  */
  void cropCanvasToPattern();
/** crop the canvas to a rectangle
    @param rect Define the top left and width height of the required area
  */
  void cropCanvasToRect(QRect rect);
/** add a stitch in the cell
  */
  void addStitch(QPoint cell, Stitch::Type type);
/** add a stitch at a location with a color
    @param x column
    @param y row
    @param i color index from palette
  */
  void addStitch(int x, int y, Stitch::Type t, int i);
/** add a french knot
  */
  void addFrenchKnot(QPoint);
/** delete a french knot taking account of color mask */
  void deleteFrenchKnot(QPoint, bool);
/** add stitches to at cells in list
    @param cells QPointArray of cells
  */
  void addStitches(QPointArray cells, Stitch::Type type);
/** delete the stitches in the cell
    @param cell to be deleted
  */
  void deleteStitch(QPoint cell, Stitch::Type type=Stitch::Delete, bool colorMask=false);
/** add a backstitch from start to finish
  */
  void addBackstitch(QPoint start, QPoint end);
/** delete a backstitch
  */
  void deleteBackstitch(BackStitch *bs);
/** copy a section of the pattern to a QByteArray
  */
  QByteArray copyArea(QRect area);
/** copy the selected stitches to the selection list
    deleting the originals if cutting
  */
  void copyToSelection(QRect area, Stitch::Type stitchMask, bool colorMask, bool cut);
/** cut the selected area
    @param cells QPointArray defining cells to be cut
  */
  void cutSelection(QRect area, Stitch::Type stitchMask=Stitch::Delete, bool colorMask=false);
/** copy the selected area
    @param cells QPointArray defining cells to be copied
  */
  void copySelection(QRect area, Stitch::Type stitchMask=Stitch::Delete, bool colorMask=false);
/** paste the selected area
    @param p QPoint defining start cell to be written over
    @param data QByteArray containing the encoded data
    @param merge TRUE if selection to be pasted is merged with original
  */
  void pasteSelection(QPoint p, QByteArray data, bool merge);
/** return a pointer to the pattern canvas
  */
  PatternCanvas *canvas();
/** return a pointer to the palette
  */
  FlossPalette *palette();
/** find the color of the stitch in a cell
  */
  int findColor(QPoint cell,Stitch::Type);
/** remove colors from palette
  */
  bool removeColors();
/** remove all unused colors from palette
  */
  bool removeUnusedColors();
/** swap colors
  */
  void swapColors(int);
/** replace color
  */
  void replaceColor(int);
/** return a pointer to the pixmap
  */
  QPixmap *pixmap();
/** render the pixmap image of the file
  */
  void renderPixmap();
/** update a section of the pixmap
  */
  void updatePixmap();
  void updatePixmap(QPoint cell);
  void updatePixmap(QRect rect);
/** get the title of the pattern
  */
  QString title();
/** get the author of the pattern
  */
  QString author();
/** get the copyright string
  */
  QString copyright();
/** get the instructions
  */
  QString instructions();
/** get the color of the canvas
  */
  QColor canvasColor();
/** get the floss usage
  */
  UsageMap flossUsage();
/** the list of the views currently connected to the document
  */
  static QPtrList<KXStitchView> *viewList;

public slots:
/** resets all the views when the document changes size or attributes
  */
  void slotResetAllViews();
/** connected to KScanDialog image aquired signal
  */
  void fileImageScanned(const QImage&, int);

private:
  bool                      m_modified;
  KURL                      m_docURL;
  PatternPropertiesDialog*  m_propertiesDialog;
  KScanDialog*              m_scanDialog;
  bool                      m_waitingImage;
  QPixmap*                  m_preview;
  FlossPalette*             m_palette;
  PatternCanvas*            m_canvas;
};

#endif // XSTITCHDOC_H
