%option backup nostdinit noyywrap never-interactive full ecs
%option 8bit backup nodefault perf-report perf-report
%x COMMAND HELP STRING PARAM
%{
/*
 * Copyright (C) 2002 Roman Zippel <zippel@linux-m68k.org>
 * Released under the terms of the GNU GPL v2.0.
 */

#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <qstring.h>
#include <klocale.h>

#define YY_DECL int linuz25lex(YYSTYPE *lvalp)
#include "25grammar.h"

#define LKC_DIRECT_LINK
#include "lkc.h"

#define START_STRSIZE	16

QString text;

struct buffer {
        struct buffer *parent;
        YY_BUFFER_STATE state;
};

struct buffer *current_buf;

static int last_ts, first_ts;

static void linuz25_endhelp(YYSTYPE *lvalp);
static struct buffer *linuz25_endfile(void);

void new_string(void)
{
	text = QString::null;
}

void append_string(const char *str, int size)
{
	Q_UNUSED(size);
	text += str;
}

void alloc_string(const char *str, int size)
{
	Q_UNUSED(size);
	text = QString::fromLatin1(str);
}
%}

ws	[ \n\t]
n	[A-Za-z0-9_]

%%
	int str = 0;
	int ts, i;

[ \t]*#.*\n	current_file->lineno++;
[ \t]*#.*

[ \t]*\n	current_file->lineno++; return T_EOL;

[ \t]+	{
	BEGIN(COMMAND);
}

.	{
	unput(yytext[0]);
	BEGIN(COMMAND);
}


<COMMAND>{
	"mainmenu"		BEGIN(PARAM); return T_MAINMENU;
	"menu"			BEGIN(PARAM); return T_MENU;
	"endmenu"		BEGIN(PARAM); return T_ENDMENU;
	"source"		BEGIN(PARAM); return T_SOURCE;
	"choice"		BEGIN(PARAM); return T_CHOICE;
	"endchoice"		BEGIN(PARAM); return T_ENDCHOICE;
	"comment"		BEGIN(PARAM); return T_COMMENT;
	"config"		BEGIN(PARAM); return T_CONFIG;
	"help"			BEGIN(PARAM); return T_HELP;
	"if"			BEGIN(PARAM); return T_IF;
	"endif"			BEGIN(PARAM); return T_ENDIF;
	"depends"		BEGIN(PARAM); return T_DEPENDS;
	"requires"		BEGIN(PARAM); return T_REQUIRES;
	"optional"		BEGIN(PARAM); return T_OPTIONAL;
	"default"		BEGIN(PARAM); return T_DEFAULT;
	"prompt"		BEGIN(PARAM); return T_PROMPT;
	"tristate"		BEGIN(PARAM); return T_TRISTATE;
	"bool"			BEGIN(PARAM); return T_BOOLEAN;
	"boolean"		BEGIN(PARAM); return T_BOOLEAN;
	"int"			BEGIN(PARAM); return T_INT;
	"hex"			BEGIN(PARAM); return T_HEX;
	"string"		BEGIN(PARAM); return T_STRING;
	{n}+	{
		alloc_string(yytext, yyleng);
		lvalp->string = text.latin1();
		return T_WORD;
	}
	.
	\n	current_file->lineno++; BEGIN(INITIAL);
}

<PARAM>{
	"&&"	return T_AND;
	"||"	return T_OR;
	"("	return T_OPEN_PAREN;
	")"	return T_CLOSE_PAREN;
	"!"	return T_NOT;
	"="	return T_EQUAL;
	"!="	return T_UNEQUAL;
	"if"	return T_IF;
	"on"	return T_ON;
	\"|\'	{
		str = yytext[0];
		new_string();
		BEGIN(STRING);
	}
	\n	BEGIN(INITIAL); current_file->lineno++; return T_EOL;
	---	/* ignore */
	({n}|[-/.])+	{
		alloc_string(yytext, yyleng);
		lvalp->string = text.latin1();
		return T_WORD;
	}
	\\\n	current_file->lineno++;
	.
	<<EOF>> {
		BEGIN(INITIAL);
	}
}

<STRING>{
	[^'"\\\n]+/\n	{
		append_string(yytext, yyleng);
		lvalp->string = text.latin1();
		return T_WORD_QUOTE;
	}
	[^'"\\\n]+	{
		append_string(yytext, yyleng);
	}
	\\.?/\n	{
		append_string(yytext + 1, yyleng - 1);
		lvalp->string = text.latin1();
		return T_WORD_QUOTE;
	}
	\\.?	{
		append_string(yytext + 1, yyleng - 1);
	}
	\'|\"	{
		if (str == yytext[0]) {
			BEGIN(PARAM);
			lvalp->string = text.latin1();
			return T_WORD_QUOTE;
		} else
			append_string(yytext, 1);
	}
	\n	{
		printf("%s:%d:warning: multi-line strings not supported\n", linuz25_curname().latin1(), linuz25_lineno());
		current_file->lineno++;
		BEGIN(INITIAL);
		return T_EOL;
	}
	<<EOF>>	{
		BEGIN(INITIAL);
	}
}

<HELP>{
	[ \t]+	{
		ts = 0;
		for (i = 0; i < yyleng; i++) {
			if (yytext[i] == '\t')
				ts = (ts & ~7) + 8;
			else
				ts++;
		}
		last_ts = ts;
		if (first_ts) {
			if (ts < first_ts) {
				linuz25_endhelp(lvalp);
				return T_HELPTEXT;
			}
			ts -= first_ts;
			while (ts > 8) {
				append_string("        ", 8);
				ts -= 8;
			}
			append_string("        ", ts);
		}
		
	}
	\n/[^ \t\n] {
		current_file->lineno++;
		linuz25_endhelp(lvalp);
		return T_HELPTEXT;
	}
	[ \t]*\n	{
		current_file->lineno++;
		append_string("\n", 1);
	}
	[^ \t\n].* {
		append_string(yytext, yyleng);
		if (!first_ts)
			first_ts = last_ts;
	}
	<<EOF>>	{
		linuz25_endhelp(lvalp);
		return T_HELPTEXT;
	}
}

<<EOF>>	{
	if (current_buf) {
		linuz25_endfile();
		return T_EOF;
	}
	fclose(yyin);
	yyterminate();
}

%%
void linuz25_starthelp(void)
{
	new_string();
	last_ts = first_ts = 0;
	BEGIN(HELP);
}

static void linuz25_endhelp(YYSTYPE *lvalp)
{
	lvalp->string = text.latin1();
	BEGIN(INITIAL); 
}


/* 
 * Try to open specified file with following names:
 * ./name
 * $(srctree)/name
 * The latter is used when srctree is separate from objtree
 * when compiling the kernel.
 * Return NULL if file is not found.
 */
FILE *linuz25_fopen(const char *name)
{
	return fopen(name, "r");
}

void linuz25_initscan(const char *name)
{
	yyin = linuz25_fopen(name);
	if (!yyin) {
		printf("can't find file %s\n", name);
		exit(1);
	}

	current_buf = new buffer;
	memset(current_buf, 0, sizeof(*current_buf));

	current_file = file_lookup(name);
	current_file->lineno = 1;
	current_file->flags = FILE_BUSY;
}

void linuz25_nextfile(const char *name)
{
	struct file *file = file_lookup(name);
	struct buffer *buf = new buffer;
	memset(buf, 0, sizeof(*buf));

	current_buf->state = YY_CURRENT_BUFFER;
	yyin = linuz25_fopen(name);
	if (!yyin) {
		printf("%s:%d: can't open file \"%s\"\n", linuz25_curname().latin1(), linuz25_lineno(), name);
		exit(1);
	}
	yy_switch_to_buffer(yy_create_buffer(yyin, YY_BUF_SIZE));
	buf->parent = current_buf;
	current_buf = buf;

	if (file->flags & FILE_BUSY) {
		printf("recursive scan (%s)?\n", name);
		exit(1);
	}
	if (file->flags & FILE_SCANNED) {
		printf("file %s already scanned?\n", name);
		exit(1);
	}
	file->flags |= FILE_BUSY;
	file->lineno = 1;
	file->parent = current_file;
	current_file = file;
}

static struct buffer *linuz25_endfile(void)
{
	struct buffer *parent;

	current_file->flags |= FILE_SCANNED;
	current_file->flags &= ~FILE_BUSY;
	current_file = current_file->parent;

	parent = current_buf->parent;
	if (parent) {
		fclose(yyin);
		yy_delete_buffer(YY_CURRENT_BUFFER);
		yy_switch_to_buffer(parent->state);
	}
	delete current_buf;
	current_buf = parent;

	return parent;
}

int linuz25_lineno(void)
{
	if (current_buf)
		return current_file->lineno;
	else
		return 0;
}

QString linuz25_curname(void)
{
	if (current_buf)
		return current_file->name;
	else
		return i18n("(none)");
}

