/*  ISDN button V2.6 970413
 *  Copyright (C) 1996, 97  M. Gutschke
 *
 *  At the time of writing my e-mail address is:
 *	Internet: gutschk@uni-muenster.de
 *  My snail mail address is:
 *	Markus Gutschke
 *	Schlage 5a
 *	48268 Greven-Gimbte
 *	Germany
 *  If you like this software, I would appreciate if you sent me a postcard
 *  from your hometown. Under the terms of the GNU general public license
 *  you are free to include this program into (commercial) software
 *  distributions (e.g. putting it onto CD-ROM); nonetheless, I would really
 *  appreciate if you dropped me a short note (sending me a sample copy of
 *  your distribution would be even more appreciated!)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/SmeLine.h>
#include <X11/extensions/shape.h>
#include <X11/xpm.h>
#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <linux/isdn.h>
#include <linux/isdnif.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <sys/resource.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <syslog.h>
#include <unistd.h>

#ifndef ISDNPATH
#define ISDNPATH "/usr/sbin/isdn"
/* #define ISDNPATH "/usr/local/sbin/isdn" */
#endif
   
/* redefine isdn_net_ioctl_cfg to include the definition of chargeint */
typedef struct {
  char name[10];     /* Name of interface                     */
  char master[10];   /* Name of Master for Bundling           */
  char slave[10];    /* Name of Slave for Bundling            */
  char eaz[256];     /* EAZ/MSN                               */
  char drvid[25];    /* DriverId for Bindings                 */
  int  onhtime;      /* Hangup-Timeout                        */
  int  charge;       /* Charge-Units                          */
  int  l2_proto;     /* Layer-2 protocol                      */
  int  l3_proto;     /* Layer-3 protocol                      */
  int  p_encap;      /* Encapsulation                         */
  int  exclusive;    /* Channel, if bound exclusive           */
  int  dialmax;      /* Dial Retry-Counter                    */
  int  slavedelay;   /* Delay until slave starts up           */
  int  cbdelay;      /* Delay before Callback                 */
  int  chargehup;    /* Flag: Charge-Hangup                   */
  int  ihup;         /* Flag: Hangup-Timeout on incoming line */
  int  secure;       /* Flag: Secure                          */
  int  callback;     /* Flag: Callback                        */
  int  cbhup;        /* Flag: Reject Call before Callback     */
  int  pppbind;      /* ippp device for bindings              */
  int  chargeint;    /* Use fixed charge interval length      */
} chargeint_isdn_net_ioctl_cfg;

typedef struct menuRec {
  struct menuRec *next;
  char          *fname;
  char          *description;
  char          *interface;
  int           red:1;
  int           yellow:1;
  int           green:1;
} MenuRec;

typedef struct {
  Cardinal      width,height;
  Pixmap        data;
  Pixmap        mask;
} XpmData;

typedef struct {
  Boolean       autoUpdate;
  long          updateInterval;
  Boolean       showChargehup;
  Boolean       shape;
  Boolean       launchAsDaemon;
  XpmData       *curr_xpm;
  XpmData       red_xpm;
  XpmData       yellow_xpm;
  XpmData       green_xpm;
} AppData;

static Widget	    toplevel,form,icon,chargehup;
static XtAppContext app_context;
static enum status {isRed,isYellow,isGreen} curStatus = isRed;
static int          curChargehup = -1;
static XtIntervalId timer;
static Atom         wmDeleteWindow;
static int          rootUID,rootGID;
static AppData      app_data;

static void setShape(Widget widget)
{
  /* Handle shaped windows */
  static Widget lastWidget = NULL;
  Position  x,y,xw,yw;
  Dimension ww,hw,ih,iw;
  int       w,h;

  if (widget)
    lastWidget = widget;
  else if ((widget = lastWidget) == NULL)
    return;
  /* Icon mask */
  XtVaGetValues(form,XtNx,&x,XtNy,&y,NULL);
  XtVaGetValues(icon,XtNx,&xw,XtNy,&yw,XtNwidth,&ww,XtNheight,&hw,
		XtNinternalHeight,&ih,XtNinternalWidth,&iw,NULL);
  if ((w = ((int)ww - (int)app_data.curr_xpm->width)/2)  < (int)iw) w = iw;
       h = ((int)hw - (int)app_data.curr_xpm->height)/2;
  XShapeCombineMask(XtDisplay(widget),XtWindow(widget),ShapeBounding,
		    x+xw+w,y+yw+h,app_data.curr_xpm->mask,ShapeSet);
  /* Chargehuplabel mask */
  if (app_data.showChargehup) {
    XRectangle rect;
    
    XtVaGetValues(chargehup,XtNx,&xw,XtNy,&yw,XtNwidth,&ww,XtNheight,&hw,
		  NULL);
    rect.x = xw; rect.y = yw; rect.width = ww; rect.height = hw;
    XShapeCombineRectangles(XtDisplay(widget),XtWindow(widget),
			    ShapeBounding,x,y,&rect,1,ShapeUnion,0); }
  return;
}

static void setPixmap(enum status status)
{
  app_data.curr_xpm = status == isRed    ? &app_data.red_xpm :
                      status == isYellow ? &app_data.yellow_xpm :
                                           &app_data.green_xpm;
  XtVaSetValues(icon,XtNbitmap,app_data.curr_xpm->data,
		NULL);
  setShape(NULL);
  return;
}

static void setChargehup(int val)
{
  if (app_data.showChargehup && val != curChargehup) {
    char buf[80];
    sprintf(buf,"%ds",curChargehup = val);
    XtVaSetValues(chargehup,XtNlabel,buf,NULL); }
  return;
}

static void testStatus(XtPointer closure,XtIntervalId *id)
{
  char                 buf[8192+1],*ptr;
  enum status          status = isRed;
  int                  fd,len;
  FILE                 *fp;

  seteuid(rootUID); setegid(rootGID);
  /* Check if any of the isdn network devices currently has an outgoing
     telephone number assigned to its interface */
  if ((fp = fopen("/proc/net/dev","r")) != NULL) {
    setvbuf(fp,buf+sizeof(buf)/2,_IOFBF,sizeof(buf)/2);
    while (!feof(fp) && fgets(buf,sizeof(buf),fp)) {
      if ((ptr = strchr(buf,':')) != NULL) {
	*ptr = '\000';
	for (ptr = buf; *ptr == ' '; ptr++);
	if (*ptr && (fd = open("/dev/isdnctrl",O_RDONLY)) >= 0) {
	  isdn_net_ioctl_phone phone;
	  chargeint_isdn_net_ioctl_cfg   cfg;
	  
	  strcpy(phone.name,ptr);
	  phone.outgoing = 1;
	  if (ioctl(fd,IIOCNETGNM,&phone) >= 0 &&
	      *phone.name && strcmp(phone.name,ptr))
	    status = isYellow;
	  strcpy(cfg.name,ptr);
	  if (ioctl(fd,IIOCNETGCF,&cfg) >= 0)
	    setChargehup(cfg.chargeint);
	  close(fd); } } }
    fclose(fp); }
  /* Check whether any of the isdn network devices is currently active */
  if ((fd = open("/dev/isdninfo",O_RDONLY|O_NDELAY)) >= 0) {
    if ((len = read(fd,buf,sizeof(buf)-1)) >= 0) {
      buf[len] = '\000';
      for (ptr = buf; ptr && *ptr; (ptr = strchr(ptr,'\n')) ? ptr++ : 0) {
	if (!strncmp(ptr,"usage:\t",7))
	  for (ptr += 7; *ptr && *ptr != '\n'; ptr++)
	    if (*ptr != '0' && *ptr != ' ') {
	      status = isGreen; break; } } }
    close(fd); }
  seteuid(getuid()); setegid(getgid());
  /* Update the status icon and reschedule the next test in 3 seconds time */
  if (status != curStatus)
    setPixmap(curStatus = status);
  if (id && app_data.autoUpdate)
    timer = XtAppAddTimeOut(app_context,app_data.updateInterval,testStatus,0);
  return;
}

static int open_max(void)
{
  #ifdef OPEN_MAX
  static int openmax = OPEN_MAX;
  #else
  static int openmax = 0;
  #endif

  if (openmax == 0) {
    errno = 0;
    if ((openmax = sysconf(_SC_OPEN_MAX)) < 0) {
      if (errno == 0)
	openmax = 256;
      else {
	syslog(LOG_ERR,"Sysconf error for _SC_OPEN_MAX");
	return(-1); } } }
  return(openmax);
}

static int initDaemon(void)
{
  static int maxfd = -1;
  pid_t      pid;
  int        fd,stat;

  /* Put process into background */
  if ((pid = fork()) < 0) {
    syslog(LOG_ERR,"Fork error: %s\n",strerror(errno));
    return(-1); }
  else if (pid != 0) {
    /* reap exit code to avoid zombies */
    while (waitpid(pid,&stat,0) < 0 && errno == EINTR);
    return(pid); }
  /* Get rid of parent process, so we will be adopted from init */
  if ((pid = fork()) < 0) {
    syslog(LOG_ERR,"Fork error: %s\n",strerror(errno));
    return(-1); }
  else if (pid != 0)
    exit(0);
  /* Now, we are a true daemon process; make sure that all of the
     process environment is configured properly */
  setsid();
  chdir("/");
  umask(0);
  if (maxfd < 0 && (maxfd = open_max()) < 0)
    return(-1);
  for (fd = maxfd; fd--; )
    close(fd);
  openlog("isdnbutton",LOG_PID,LOG_USER);
  return(0);
}

static void launch(const char *prgname)
{
  static int maxfd = -1;
  char       buf[2048];
  const char *basename;
  int        pfd[2],stat;
  pid_t      pid;
  FILE       *fp;

  /* Avoid using "system" or "popen" as they introduce possible security
     holes */
  if (maxfd <  0) maxfd = open_max();
  if ((basename = strrchr(prgname,'/')) != NULL) basename++;
  else basename = prgname;
  if (app_data.launchAsDaemon) {
    if ((pid = initDaemon()) < 0)
      return;
    else if (pid != 0) /* parent */
      return; }
  if (pipe(pfd) < 0 || (pid = fork()) < 0)
    return;
  else if (pid == 0) { /* child */
    char *env[] = { NULL };
    int  i;
    /* Open /dev/null on stdin and connect both stdout and stderr to the
       pipe */
    close(pfd[0]); i = open("/dev/null",O_RDONLY);
    if (i != 0) { dup2(i,0); close(i); }/* stdin  */
    if (pfd[1] != 1) dup2(pfd[1],1);    /* stdout */
    if (pfd[1] != 2) dup2(pfd[1],2);    /* stderr */
    if (pfd[1] != 1 && pfd[1] != 2) close(pfd[1]);
    /* Close all other file handles, including the syslog */
    closelog();
    for (i = 3; i < maxfd; i++)
      close(i);
    /* From here on, run as priviledged user */
    setuid(rootUID); setgid(rootGID);
    setuid(rootUID); setgid(rootGID);
    /* Launch the shell script */
    execle(prgname,basename,NULL,env);
    _exit(127); }
  /* Connect to the child and read all of its output, piping it to syslogd */
  close(pfd[1]);
  if ((fp = fdopen(pfd[0],"r")) == NULL) return;
  while (!feof(fp) && fgets(buf,sizeof(buf)-1,fp) != NULL)
    syslog(LOG_WARNING,"%s: %s",basename,buf);
  fclose(fp);
  /* Report exit status */
  while (waitpid(pid,&stat,0) < 0 && errno == EINTR);
  if ((WIFEXITED(stat) && WEXITSTATUS(stat)) ||
      WIFSIGNALED(stat))
    syslog(LOG_ERR,WIFSIGNALED(stat) ?
	   "%s terminated by signal %d" :
	   WEXITSTATUS(stat) == 127 ?
	   "Could not execute %s" :
	   "%s returns %d",
	   prgname,
	   WIFSIGNALED(stat) ? WTERMSIG(stat) : WEXITSTATUS(stat));
  if (app_data.launchAsDaemon)
    exit(0);
  return;
}

static int readscripts(const char *path,MenuRec **menuList)
{
  DIR           *dir;
  struct dirent *dirent;
  int           fd,num = 0;
  struct stat   stat;
  char          *map,*msg,*p1,*p2;
  MenuRec       *menuRec,**ptr;

  /* Scan the directory for suitable script files */
  *menuList = NULL;
  if ((dir = opendir(path)) == NULL)
    return(0);
  while ((dirent = readdir(dir)) != NULL) {
    char *fname = XtMalloc(strlen(path)+strlen(dirent->d_name)+1);
    strcat(strcpy(fname,path),dirent->d_name);
    /* Open file for reading */
    if ((fd = open(fname,O_RDONLY)) >= 0) {
      /* File has to be a "regular" file and has to be mmap()-able */
      if (!lstat(fname,&stat) && (stat.st_mode & S_IFMT) == S_IFREG &&
	  (map = mmap(NULL,stat.st_size,PROT_READ,MAP_FILE|MAP_PRIVATE,
		      fd,0)) != (char *)-1L) {
	/* Scan for magic cookie */
	if ((msg = memmem(map,stat.st_size,"ISDNBUTTON:",11)) != NULL) {
	  /* Create new record describing this menu entry */
	  menuRec = (MenuRec *)XtMalloc(sizeof(MenuRec)+1);
	  memset(menuRec,0,sizeof(MenuRec)+1); /* interface */
	  /* Parse flags (optional) */
	  for (msg += 11; msg < map+stat.st_size &&
		 *msg > ' ' && *(unsigned char *)msg <= '\x7F'; msg++)
	    switch (*msg) {
	    case 'r': case 'R': menuRec->red    = 1; break;
	    case 'y': case 'Y': menuRec->yellow = 1; break;
	    case 'g': case 'G': menuRec->green  = 1; break;
	    case '<':
	      /* interface name */
	      for (p1 = ++msg; msg < map+stat.st_size && *msg != '>'; msg++);
	      menuRec = (MenuRec *)XtRealloc((XtPointer)menuRec,
					     sizeof(MenuRec)+(msg-p1)+1);
	      ((char *)memcpy(menuRec+1,p1,msg-p1))[msg-p1] = '\000';
	      break;
	    default:                                 break; }
	  while (msg < map+stat.st_size && (*msg == ' ' || *msg == '\t'))msg++;
	  /* Extract description; has to be terminated by newline or NUL */
	  p1 = memchr(msg,'\n',  stat.st_size-(msg-map));
	  p2 = memchr(msg,'\000',stat.st_size-(msg-map));
	  if (p1 == NULL)                 p1 = p2;
	  if (p1 == NULL)                 p1 = msg+stat.st_size;
	  else if (p2 != NULL && p2 < p1) p1 = p2;
	  if (msg == p1)
	    /* Discard empty entries */
	    XtFree((XtPointer)menuRec);
	  else {
	    /* Copy new entry to list */
	    menuRec = (MenuRec *)XtRealloc((XtPointer)menuRec,sizeof(MenuRec)+
					   strlen(fname) +
					   strlen((char *)(menuRec+1)) +
					   (p1-msg) + 3);
	    (menuRec->description =
	     (char *)memcpy(strrchr(menuRec->fname =
				    strcpy(strrchr(menuRec->interface =
						   (char *)(menuRec + 1),
						   '\000')+1,
					   fname),'\000')+1,msg,p1-msg))
	      [p1-msg] = '\000';
	    /* Enter in linear list; sort by filename */
	    for (ptr = menuList; *ptr != NULL &&
		   strcmp((*ptr)->fname,menuRec->fname) < 0;
		 ptr = &((*ptr)->next));
	    menuRec->next = *ptr;
	    *ptr = menuRec;
	    num++; } }
	munmap(map,stat.st_size); }
      close(fd); }
    XtFree(fname); }
  closedir(dir);
  /* Done */
  return(num);
}

static void destroyCb(Widget widget,XtPointer client_data,XtPointer call_data)
{
  /* Cleanup resource of menu entries */
  XtFree(client_data);
  XtRemoveCallback(widget,XtNdestroyCallback,destroyCb,client_data);
  return;
}

static void selectCb(Widget widget,XtPointer client_data,XtPointer call_data)
{
  MenuRec *menuRec = (MenuRec *)client_data;

  launch(menuRec->fname);
  /* Reschedule next timer event immediately */
  XtRemoveTimeOut(timer);
  timer = XtAppAddTimeOut(app_context,200,testStatus,0);
  return;
}

static void popdownCb(Widget widget,XtPointer client_data,XtPointer call_data)
{
  Widget  *children = (Widget *)client_data;

  /* Destroy all menu entries */
  while (*children)
    XtDestroyWidget(*children++);
  XtFree(client_data);
  XtRemoveCallback(widget,XtNpopdownCallback,popdownCb,client_data);
  return;
}

static Widget findWidget(Widget widget,String name)
{
  Widget w,menu;
    
  /* Find widget by name */
  for (w = widget; w != NULL; w = XtParent(w))
    if ((menu = XtNameToWidget(w,name)) != NULL)
      return(menu);
  if (widget == toplevel && !strcmp(name,"isdnbutton"))
    return(widget);
  return(NULL);
}

static void actSetShape(Widget w,XEvent *event,String *params,
		      Cardinal *num_params)
{
  char   *name = "isdnbutton";
  Widget widget;

  if (app_data.shape && event->type == ConfigureNotify) {
    if (*num_params > 1) {
      syslog(LOG_ERR,"setShape() expects one parameter");
      return; }
    else if (*num_params == 1)
      name = params[0];
    if ((widget = findWidget(w,name)) == NULL) {
      syslog(LOG_ERR,"could not find window named \"%s\" for setShape()",name);
      return; }
    setShape(widget); }
  return;
}

static void makeMenu(Widget w,XEvent *event,String *params,
		     Cardinal *num_params)
{
  static MenuRec menuConnect = {NULL,ISDNPATH"/connect"};
  static MenuRec menuDisconnect = {NULL,ISDNPATH"/disconnect"};
  char    *name = "menu";
  Widget  menu;
  Widget  *children;
  int     num_children,i;
  MenuRec *menuRec,*ptr;

  /* Reschedule next timer event immediately (this is useful if
     autoUpdate has been disabled) */
  XtRemoveTimeOut(timer);
  timer = XtAppAddTimeOut(app_context,200,testStatus,0);
  if (*num_params > 1) {
    syslog(LOG_ERR,"makeMenu() expects one parameter");
    return; }
  else if (*num_params == 1)
    name = params[0];
  if ((menu = findWidget(w,name)) == NULL) {
    syslog(LOG_ERR,"could not find menu named \"%s\" for makeMenu()",name);
    return; }
  num_children = readscripts(ISDNPATH"/",&menuRec);
  children = (Widget *)XtMalloc((num_children ? num_children+1 : 3)
				*sizeof(Widget));
  if (num_children) {
    for (i = 0, ptr = menuRec; i < num_children; i++, ptr = ptr->next) {
      char buf[80];
      sprintf(buf,"item%d",i+1);
      XtAddCallback(children[i] =
		    XtVaCreateManagedWidget(buf,smeBSBObjectClass,menu,
					    XtNlabel,ptr->description,NULL),
		    XtNdestroyCallback,destroyCb,ptr);
      XtAddCallback(children[i],XtNcallback,selectCb,ptr); } }
  else {
    /* Fall back to standard menu, if user did not provide customized scripts*/
    num_children = 2;
    XtAddCallback(children[0] =
		  XtVaCreateManagedWidget("item1",smeBSBObjectClass,menu,
					  XtNlabel,"Connect",NULL),
		  XtNcallback,selectCb,&menuConnect);
    XtAddCallback(children[1] =
		  XtVaCreateManagedWidget("item2",smeBSBObjectClass,menu,
					  XtNlabel,"Disconnect",NULL),
		  XtNcallback,selectCb,&menuDisconnect); }
  children[num_children] = NULL;
  XtAddCallback(menu,XtNpopdownCallback,popdownCb,children);
  return;
}  

static void connectSkript(Widget w,XEvent *event,String *params,
			  Cardinal *num_params)
{
  testStatus(NULL,NULL);
  launch(curStatus == isRed ?
	 ISDNPATH"/connect" : ISDNPATH"/disconnect");
  /* Reschedule next timer event immediately */
  XtRemoveTimeOut(timer);
  timer = XtAppAddTimeOut(app_context,200,testStatus,0);
  return;
}

static void quit(Widget w,XEvent *event,String *params,Cardinal *num_params)
{
  if (event->type == ClientMessage &&
      event->xclient.data.l[0] == wmDeleteWindow) {
    XCloseDisplay(XtDisplay(w));
    exit (0); }
  return;
}

static void zapEnvironment(char *keep,...)
{
  extern char **environ;
  char        **env = calloc(sizeof(char *),1);
  char        *val;
  int         count = 0;
  va_list     arg;

  /* Zap all of the environment except for a few selected variables */
  if (!keep) {
    environ = (char *[]){ NULL };
    return; }
  for (va_start(arg,keep); keep != NULL; keep = va_arg(arg,char *))
    if ((val = getenv(keep)) != NULL) {
      env = realloc(env,sizeof(char *)*(count+2));
      env[count++] = strcat(strcat(strcpy(malloc(strlen(keep)+strlen(val)+2),
					  keep),"="),val);
      env[count]   = NULL; }
  va_end(arg);
  environ = env;
  return;
}

static Widget myXtVaOpenApplication(XtAppContext *app_context_return,
				    String application_class,
				    XrmOptionDescRec *options,
				    Cardinal num_options,Cardinal *argc_in_out,
				    String *argv_in_out,
				    String *fallback_resources,
				    WidgetClass widget_class,
				    XtActionsRec *actions,
				    Cardinal num_actions,...)
{
  Display      *disp;
  Widget       applShell;
  Cardinal     the_argc;
  String       *the_argv;
  XrmDatabase  database;
  ArgList      the_args;
  va_list      arg;
  String       name;
  Cardinal     n;

  XtToolkitInitialize();
  *app_context_return = XtCreateApplicationContext();
  the_argv = (String *)XtMalloc((the_argc=*argc_in_out)*sizeof(String));
  for (n = the_argc; n--; ) the_argv[n] = XtNewString(argv_in_out[n]);
  disp = XtOpenDisplay(*app_context_return,NULL,NULL,application_class,
			  options,num_options,argc_in_out,argv_in_out);
  if (disp == NULL) {
    fprintf(stderr,"%s: Cannot not open display\n",the_argv[0]);
    exit(EXIT_FAILURE); }
  XtAddConverter(XtRString,XtRLong,XmuCvtStringToLong,NULL,0);
  XawSimpleMenuAddGlobalActions(*app_context_return);
  XtAppAddActions(*app_context_return,actions,num_actions);
  if (fallback_resources) {
    database = NULL;
    while (*fallback_resources)
      XrmPutLineResource(&database,*fallback_resources++);
    if (database) {
      XrmDatabase display_database = XtDatabase(disp);
      XrmCombineDatabase(database,&display_database,False); } }
  the_args = (ArgList)XtMalloc(3*sizeof(Arg));
  XtSetArg(the_args[0],XtNargc,the_argc);
  XtSetArg(the_args[1],XtNargv,the_argv);
  XtSetArg(the_args[2],XtNscreen,DefaultScreenOfDisplay(disp));
  va_start(arg,num_actions);
  for (n = 3; (name = va_arg(arg,String)) != NULL; n++) {
    the_args = (ArgList)XtRealloc((XtPointer)the_args,(n+1)*sizeof(Arg));
    XtSetArg(the_args[n],name,va_arg(arg,XtArgVal)); }
  va_end(arg);
  applShell = XtAppCreateShell(NULL,application_class,widget_class,disp,
			       the_args,n);
  XtFree((XtPointer)the_args);
  return(applShell);
}

Boolean CvtStringToXpm(Display *display,XrmValue *args,Cardinal *num_args,
		       XrmValue *fromVal,XrmValue *toVal,XtPointer *data)
{
#define XtRXPM  "PixmapXPM"
  char          **xpm,*buf,*ptr;
  int           i;

  if (*num_args != 0)
    XtWarningMsg("wrongParameters","cvtStringToXpm","XtToolkitError",
		 "String to XPM conversion needs no extra arguments",
		 (String *)NULL,(Cardinal *)NULL);
  /* Count the number of lines, that are separated by newline characters */
  for (ptr = (char *)fromVal->addr-1, i = 0; ptr != NULL && ptr[1] != '\000';
       ptr = strchr(ptr+1,'\n'), i++);
  if (i) {
    static XpmData xpmdata;
    XpmAttributes  attr;
    XpmColorSymbol xpmbackground;
    int            dummy;

    /* We have to evalute the icon size and we want to set the background
       color, because the menuButtonWidget does not support a shape mask */
    memset(&attr,0,sizeof(attr));
    /* Query for availibility of the shape extension */
    if (app_data.shape && !XShapeQueryExtension(XtDisplay(toplevel),
						&dummy,&dummy))
      app_data.shape = FALSE;
    if (app_data.shape)
      attr.valuemask    = XpmSize;
    else {
      attr.valuemask    = XpmSize|XpmColorSymbols;
      attr.numsymbols   = 1;
      attr.colorsymbols = &xpmbackground;
      /* Determine the background color of the toplevel shell and use this
	 as background for the pixmap (poor man's shape extension) */
      XtVaGetValues(toplevel,XtNbackground,&xpmbackground.pixel,NULL);
      xpmbackground.name  = "None";
      xpmbackground.value = NULL; }
    /* Create a local copy of the XPM data, so that we can convert the
       newline characters */
    buf = strcpy((char *)XtMalloc(strlen((char *)fromVal->addr)+1),
		 (char *)fromVal->addr);
    xpm = (char **)XtMalloc(sizeof(char *)*i);
    for (ptr = buf-1, i = 0; ptr != NULL && ptr[1] != '\000';
	 ((ptr = strchr(ptr+1,'\n'))?(*ptr='\000'):0), i++)
      xpm[i] = ptr+1;
    /* libXpm performs the conversion for us */
    if (XpmCreatePixmapFromData(display,XtWindow(toplevel),xpm,&xpmdata.data,
				&xpmdata.mask,&attr) == XpmSuccess) {
      xpmdata.width  = attr.width;
      xpmdata.height = attr.height;
      /* Release temporary storage */
      XtFree((XtPointer)buf);
      XtFree((XtPointer)xpm);
      XpmFreeAttributes(&attr);
      /* Return the pixmap ID to Xt */
      if (toVal->addr != NULL) {
	if (toVal->size < sizeof(XpmData)) {
	  toVal->size = sizeof(XpmData);
	  return(FALSE); }
	*(XpmData *)toVal->addr = xpmdata; }
      else
	toVal->addr = (XPointer)&xpmdata;
      toVal->size = sizeof(XpmData);		    
      return(TRUE); }
    /* Release temporary storage */
    XtFree((XtPointer)buf);
    XtFree((XtPointer)xpm); }
  XtDisplayStringConversionWarning(display,(char *)fromVal->addr,XtRXPM);
  return(FALSE);
}

static void DestroyCvtXPM(XtAppContext app,XrmValue *toVal,XtPointer closure,
			  XrmValue *args,Cardinal *num_args)
{
  XFreePixmap(XtDisplay(*(Widget *)args[0].addr),
	      ((XpmData *)toVal->addr)->data);
  XFreePixmap(XtDisplay(*(Widget *)args[0].addr),
	      ((XpmData *)toVal->addr)->mask);
}

int main (int argc, char *argv[])
{
#define XtNautoUpdate     "autoUpdate"
#define XtCAutoUpdate     "AutoUpdate"
#define XtNupdateInterval "updateInterval"
#define XtCUpdateInterval "UpdateInterval"
#define XtNshowChargehup  "showChargehup"
#define XtCShowChargehup  "ShowChagehup"
#define XtNshape          "shape"
#define XtCShape          "Shape"
#define XtNlaunchAsDaemon "launchAsDaemon"
#define XtCLaunchAsDaemon "LaunchAsDaemon"
#define XtNredIcon        "redIcon"
#define XtCRedIcon        "RedIcon"
#define XtNyellowIcon     "yellowIcon"
#define XtCYellowIcon     "YellowIcon"
#define XtNgreenIcon      "greenIcon"
#define XtCGreenIcon      "GreenIcon"
  static XtResource   resources[]= {{XtNautoUpdate,XtCAutoUpdate,XtRBoolean,
				     sizeof(Boolean),XtOffsetOf(AppData,
				     autoUpdate),XtRImmediate,NULL},
				    {XtNupdateInterval,XtCUpdateInterval,
				     XtRLong,sizeof(long),
				     XtOffsetOf(AppData,updateInterval),
				     XtRImmediate,NULL},
				    {XtNshowChargehup,XtCShowChargehup,
				     XtRBoolean,sizeof(Boolean),
				     XtOffsetOf(AppData,showChargehup),
				     XtRImmediate,NULL},
				    /* shape must be defined before defining
				       any pixmaps! */
				    {XtNshape,XtCShape,XtRBoolean,
				     sizeof(Boolean),XtOffsetOf(AppData,shape),
				     XtRImmediate,NULL},
				    {XtNlaunchAsDaemon,XtCLaunchAsDaemon,
				     XtRBoolean,sizeof(Boolean),
				     XtOffsetOf(AppData,launchAsDaemon),
				     XtRImmediate,NULL},
				    {XtNredIcon,XtCRedIcon,XtRXPM,
				     sizeof(XpmData),XtOffsetOf(AppData,
				     red_xpm),XtRImmediate,NULL},
				    {XtNyellowIcon,XtCYellowIcon,XtRXPM,
				     sizeof(XpmData),XtOffsetOf(AppData,
				     yellow_xpm),XtRImmediate,NULL},
				    {XtNgreenIcon,XtCGreenIcon,XtRXPM,
				     sizeof(XpmData),XtOffsetOf(AppData,
				     green_xpm),XtRImmediate,NULL}};
  static XtActionsRec actions[]  = {{"quit",quit},
				    {"setShape",actSetShape},
				    {"connectSkript",connectSkript},
				    {"makeMenu",makeMenu}};
  static String       fallback[] = {
				    #include "red.dat"
				    #include "yellow.dat"
				    #include "green.dat"
				    "*autoUpdate:         True",
				    "*updateInterval:     3000",
				    "*showChargehup:      False",
				    "*shape:              False",
				    "*launchAsDaemon:     False",
				    "*background:         MidnightBlue",
				    "*foreground:         lightblue",
				    "*borderWidth:        0",
				    "*defaultDistance:    0",
				    "*isdnbutton.baseTranslations:"
				    " <Message>WM_PROTOCOLS: quit()\\n"
				    " <Configure>:       setShape(isdnbutton)",
				    "*form.baseTranslations:"
				    " <Btn1Down>:         connectSkript()\\n"
				    " <Btn3Down>:         makeMenu(menu)"
				    "              XawPositionSimpleMenu(menu)"
				    "                     XtMenuPopup(menu)",
				    "*icon.baseTranslations:",
				    "*icon.top:           chainTop",
				    "*icon.bottom:        chainBottom",
				    "*icon.right:         chainRight",
				    "*icon.left:          chainLeft",
				    "*chargehup.fromVert: icon",
				    "*chargehup.top:      chainBottom",
				    "*chargehup.font:     *fixed*60*iso8859*",
				    "*menu*initialResourcesPersistent: False",
				    "*menu.label:         ISDN",
				    "*menu*background:    grey75",
				    "*menu*foreground:    black",
				    "*menu*font:         *times*medium*i*140*",
				    NULL};
  Display             *display;

  /* Close some of the more obvious security holes related to running setuid-
     root programs */
  zapEnvironment("DISPLAY","HOME","XAUTHORITY",NULL);
  setrlimit(RLIMIT_CORE,&((struct rlimit){0,0}));
  rootUID = geteuid(); rootGID = getegid();
  seteuid(getuid()); setegid(getgid());
  openlog("isdnbutton",LOG_PID,LOG_USER);
  /* Initialize the X toolkit and create a toplevel shell */
  if (!(toplevel = myXtVaOpenApplication(&app_context,"isdnbutton",NULL,0,
					 &argc,argv,fallback,
					 applicationShellWidgetClass,
					 actions,XtNumber(actions),
					 XtNmappedWhenManaged,False,
					 XtNwidth,1,
					 XtNheight,1,
					 XtNinput,True,NULL))) {
    perror ("Can't open display or init Xtoolkit");
    exit(1); }
  XtRealizeWidget(toplevel);
  /* Register XPM type converter */
  XtSetTypeConverter(XtRString,XtRXPM,CvtStringToXpm,NULL,0,
		     XtCacheNone|XtCacheRefCount,DestroyCvtXPM);
  /* Get Application resources */
  XtVaGetApplicationResources(toplevel,&app_data,resources,XtNumber(resources),
			      NULL);
  /* Add support for the WM_DELETE_WINDOW protocol */
  display  = XtDisplay(toplevel);
  wmDeleteWindow = XInternAtom(display,"WM_DELETE_WINDOW",False);
  XSetWMProtocols(display,XtWindow(toplevel),&wmDeleteWindow,1);
  /* Create a constraint widget for positioning the visual elements */
  form = XtVaCreateWidget("form",formWidgetClass,toplevel,
			  NULL);
  /* Create the widget that displays the icon */
  icon = XtVaCreateManagedWidget("icon",menuButtonWidgetClass,form,
				 XtNbitmap,
				 (app_data.curr_xpm = &app_data.red_xpm)->data,
				 NULL);
  /* Create the widget that displays the chargehup */
  if (app_data.showChargehup) {
    Dimension width;
    XtVaGetValues(icon,XtNwidth,&width,NULL);
    chargehup = XtVaCreateManagedWidget("chargehup",labelWidgetClass,form,
					XtNlabel,"",
					XtNwidth,width,
					NULL); }
  XtManageChild(form);
  /* Create the popup menu */
  XtVaCreatePopupShell("menu",simpleMenuWidgetClass,toplevel,NULL);
  /* Now lets get started */
  XMapWindow(display,XtWindow(toplevel));
  timer = XtAppAddTimeOut(app_context,200,testStatus,0);
  XtAppMainLoop(app_context);
  exit(0);
}
