/*
 *  Copyright (C) 1999, 2000 Bruno Pires Marinho
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <gnome.h>
#include <bonobo-activation/bonobo-activation.h>
#include <libgnomeui/gnome-window-icon.h>
#include <signal.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <sys/utsname.h>
#include <sys/socket.h>
#include <net/route.h>
#include <net/if.h>
#include <sys/ioctl.h> 
#include "gtm.h"
#include "file-data.h"
#include "file-list.h"
#include "main-window.h"
#include "Gtm.h"

/* The application */
GtkWidget *gtm_app = NULL;

/* Program preferences */
Preferences gtm_pref;

/* The program download list */
GtkWidget *dl_file_list = NULL;

/* Number of active automatic download */
gint num_of_download = 0;

/* ID used on the next file data created */
guint next_file_id = 0;

/* Variable used to disable events during startup */
static gboolean startup;

/* Variable holding the timeout id of the check network function */
static gint check_network_timeout_id = -1;

/* CORBA environment to store CORBA exceptions */
CORBA_Environment ev;


/* Session management */
static gint 
die (GnomeClient *client, gpointer client_data)
{
    /* When we receive this signal we leave the gtk_main */
    gtk_main_quit ();

    return FALSE;
}


/* Functions to load program preferences and save program geometry */
void
save_geometry (void)
{
    gchar *geometry;
    gint width, height, xpos, ypos;
    GtkWindow *window = GTK_WINDOW (gtm_app);

    /* Save the geometry to the configuration file */
    gnome_config_push_prefix ("/gtm/Display/");
    gtk_window_get_size (window, &width, &height);
    gtk_window_get_position (window, &xpos, &ypos);
    geometry = g_strdup_printf ("%dx%d%+d%+d", width, height, xpos, ypos);
    gnome_config_set_string ("geometry", geometry);
    g_free (geometry);
    gnome_config_pop_prefix ();
    gnome_config_sync ();
}

static gchar *
load_preferences (void)
{
    gboolean def;
    gchar *geometry;

    /* Read the geometry from the configuration file */
    gnome_config_push_prefix ("/gtm/Display/");
    geometry = gnome_config_get_string ("geometry=500x300");
    /* This geometry is g_free further ahead on init_main_window */
    gnome_config_pop_prefix ();

    /* Read global program preferences */
    gnome_config_push_prefix ("/gtm/Preferences/");

    gtm_pref.use_proxy =
        gnome_config_get_bool_with_default ("use_proxy=FALSE", &def);
    if (!def) {
        /* Remove old proxy user, pass, http and ftp from the visible config
         * since now I correctly use the private */
        gtm_pref.proxy_user = gnome_config_get_string ("proxy_user=");
        gtm_pref.proxy_pass = gnome_config_get_string ("proxy_pass=");
        gtm_pref.proxy_http = gnome_config_get_string ("proxy_http=");
        gtm_pref.proxy_ftp = gnome_config_get_string ("proxy_ftp=");
	gtm_pref.proxy_https = gnome_config_get_string ("proxy_https=");
        gnome_config_clean_key ("use_proxy");
        gnome_config_clean_key ("proxy_user");
        gnome_config_clean_key ("proxy_pass");
        gnome_config_clean_key ("proxy_http");
        gnome_config_clean_key ("proxy_ftp");
	/* note proxy_https was introduced later, so not needed here */
        gnome_config_private_set_bool ("use_proxy", gtm_pref.use_proxy);
        gnome_config_private_set_string ("proxy_user", gtm_pref.proxy_user);
        gnome_config_private_set_string ("proxy_pass", gtm_pref.proxy_pass);
        gnome_config_private_set_string ("proxy_http", gtm_pref.proxy_http);
        gnome_config_private_set_string ("proxy_ftp", gtm_pref.proxy_ftp);
    } else {
        gtm_pref.use_proxy = gnome_config_private_get_bool ("use_proxy=FALSE");
        gtm_pref.same_proxy = gnome_config_private_get_bool ("same_proxy=FALSE");
        gtm_pref.proxy_user = gnome_config_private_get_string ("proxy_user=");
        gtm_pref.proxy_pass = gnome_config_private_get_string ("proxy_pass=");
        gtm_pref.proxy_http = gnome_config_private_get_string ("proxy_http=");
        gtm_pref.proxy_ftp = gnome_config_private_get_string ("proxy_ftp=");
	gtm_pref.proxy_https = gnome_config_private_get_string ("proxy_https=");
    }
    gtm_pref.toolbar_style = 
	gnome_config_get_int_with_default ("toolbar_style", &def);
    if (def)
	gtm_pref.toolbar_style = GTK_TOOLBAR_ICONS;
    gtm_pref.def_download_dir = 
	gnome_config_get_string_with_default ("def_download_dir", &def);
    if (def)
        gtm_pref.def_download_dir = g_strdup (gnome_util_user_home ());
    gtm_pref.file_list_refresh = 
	gnome_config_get_int ("file_list_refresh=1000");
    gtm_pref.read_timeout = gnome_config_get_int ("read_timeout=30");
    gtm_pref.num_retries = gnome_config_get_int ("num_retries=0");
    gtm_pref.interface = gnome_config_get_string ("interface=ppp0");
    gtm_pref.num_of_auto_download = 
        gnome_config_get_int ("num_of_autodownload=1");
    gtm_pref.chk_interf_time = 
	gnome_config_get_int ("chk_interf_time=1000");
    gtm_pref.use_passive_ftp =
        gnome_config_get_bool ("use_passive_ftp=FALSE");
    gtm_pref.addnew_changes_def_dl_dir =
        gnome_config_get_bool ("addnew_changes_def_dl_dir=FALSE");
    gtm_pref.show_list_stat =
        gnome_config_get_bool ("show_list_stat=TRUE");
    gtm_pref.show_page_stat =
        gnome_config_get_bool ("show_page_stat=TRUE");
    gtm_pref.confirm_remove_incomplete =
        gnome_config_get_bool_with_default ("confirm_remove=TRUE", &def);
    gtm_pref.confirm_remove_completed = FALSE;
    if (!def) {
        gnome_config_clean_key ("confirm_remove");
        gnome_config_set_bool ("confirm_remove_incomplete",
                               gtm_pref.confirm_remove_incomplete);
        gnome_config_set_bool ("confirm_remove_completed", 
                               gtm_pref.confirm_remove_completed);
        gnome_config_sync ();
    } else {
        gtm_pref.confirm_remove_incomplete =
            gnome_config_get_bool ("confirm_remove_incomplete=TRUE");
        gtm_pref.confirm_remove_completed =
            gnome_config_get_bool ("confirm_remove_completed=FALSE");
    }
    gtm_pref.confirm_restart =
        gnome_config_get_bool ("confirm_restart=TRUE");
    gtm_pref.manual_stop_disables_auto_dl =
        gnome_config_get_bool ("manual_stop_disables_auto_dl=FALSE");
    gtm_pref.show_dl_info_page =
        gnome_config_get_bool ("show_dl_info_page=TRUE");
    gtm_pref.confirm_dnd_url =
        gnome_config_get_bool ("confirm_dnd_url=FALSE");
    gtm_pref.use_sound =
        gnome_config_get_bool ("use_sound=FALSE");
    gtm_pref.use_beep =
        gnome_config_get_bool ("use_beep=FALSE");
    gtm_pref.stop_dl_on_link_loss = 
        gnome_config_get_bool ("stop_dl_on_link_loss=TRUE");
    gtm_pref.presume_protocol = 
	gnome_config_get_int_with_default ("presume_protocol", &def);
    if (def)
	gtm_pref.presume_protocol = PROTOCOL_HTTP;
    gtm_pref.command_all_complete =
	gnome_config_get_string ("command_all_complete=");
    gtm_pref.load_pause_auto_dl =
        gnome_config_get_bool ("load_pause_auto_dl=FALSE");
    if (gtm_pref.load_pause_auto_dl)
        gtm_pref.pause_auto_dl =
            gnome_config_get_bool ("pause_auto_dl=FALSE");
    else
        gtm_pref.pause_auto_dl = FALSE;
    gtm_pref.auto_hide_dl_info_page =
        gnome_config_get_bool ("auto_hide_dl_info_page=TRUE");
    gtm_pref.auto_hide_dl_info_page_timeout = 
	gnome_config_get_int ("auto_hide_dl_info_page_timeout=5000");
    gnome_config_pop_prefix ();

    return geometry;
}


/* CORBA functions */

GTM_Download
impl_GTM_Download__create (PortableServer_POA poa, CORBA_Environment *ev);

static void
Exception (CORBA_Environment *ev)
{
    switch (ev->_major) {
	case CORBA_SYSTEM_EXCEPTION:
	    g_log ("GNOME Transfer Manager", G_LOG_LEVEL_DEBUG, 
		   "CORBA system exception %s.\n", CORBA_exception_id (ev));
	    exit (1);
	case CORBA_USER_EXCEPTION:
	    g_log ("GNOME Transfer Manager", G_LOG_LEVEL_DEBUG, 
		   "CORBA user exception: %s.\n",
		   CORBA_exception_id (ev));
	    exit (1);
	default:
	    break;
    }
}

/* Test if gtm is already running */
static gboolean 
corba_server_is_running()
{
    CORBA_Object gtm_servant;

    gtm_servant =
        bonobo_activation_activate ("repo_ids.has('IDL:gtm/download:1.0')",
	                            NULL, Bonobo_ACTIVATION_FLAG_EXISTING_ONLY,
				    NULL, &ev);
    Exception(&ev);
    return (!CORBA_Object_is_nil (gtm_servant, &ev));
}

/* Register GTM with OAF */
static void 
register_corba_server(CORBA_ORB orb) 
{
    CORBA_Object gtm_servant;
    GSList *reg_env;
    const char *display_name;
    PortableServer_POA root_poa;
    /* Register the gtm CORBA object */
    root_poa = (PortableServer_POA) 
	CORBA_ORB_resolve_initial_references (orb, "RootPOA", &ev);
    gtm_servant = impl_GTM_Download__create (root_poa, &ev);
    display_name = gdk_display_get_name (gdk_display_get_default ());
    reg_env =
        bonobo_activation_registration_env_set (NULL,
                                                "DISPLAY", display_name);
    bonobo_activation_register_active_server ("OAFIID:GNOME_GTM", gtm_servant,
					      reg_env);
    PortableServer_POAManager_activate
		(PortableServer_POA__get_the_POAManager (root_poa, &ev), &ev);
    g_slist_free (reg_env);
}


/* Functions to handle the signals */
static void
signal_handler (int signal_num)
{
    /* Save program geometry */
    save_geometry ();

    /* Print error message and exit */
    g_printerr ("GTM: Signal %s caught - saving and exiting\n", 
		g_strsignal (signal_num));
    exit (1);
}

/* Function to check network and automaticaly start downloads if possible */
#if defined (__NetBSD__) || defined (linux)
static gint
check_network (gpointer data)
{
    struct ifreq ifr;
    int fd;
    
    fd = socket (PF_INET, SOCK_DGRAM, 0);
    if (fd == -1) {
        perror ("Couldn't open a socket");
        exit (1);
    }
    
    memset (&ifr, 0, sizeof (ifr)); 
    strcpy (ifr.ifr_name, gtm_pref.interface);
    if (ioctl (fd, SIOCGIFFLAGS, &ifr) < 0) {
       close (fd);
       return TRUE;
    }
    close (fd);

    if (ifr.ifr_flags & IFF_UP)
        /* We are connected to the network so we need to add more
        * downloads if possible */
        file_list_start_auto_downloads (GTK_CLIST (dl_file_list));
    else
        /* The interface is down, so stop downloads */
       file_list_stop_auto_downloads (GTK_CLIST (dl_file_list));

    return TRUE;
}

#else

static gboolean
check_network (gpointer data)
{
    /* We don't know if we are connected on the interface so we don't start
     * anything automatically */

    return TRUE;
}

#endif

/* Fork and run arbitrary shell command */
void
shell_command (const gchar *command)
{
    pid_t pid;

    pid = fork();
    if (pid == 0) {
	execlp ("sh", "sh", "-c", command, NULL);
	perror ("Failed to execute command");
	exit (1);
    }
    if (pid < 0) {
	perror ("Fork failed");
    }
}

/* Report an event (play sound, beep, run command) */
void
report_event (gchar *which)
{
    if (!startup) {
        if (gtm_pref.use_sound)
            gnome_triggers_do (NULL, NULL, "gtm", which, NULL);

        if (gtm_pref.use_beep)
            gdk_beep ();
	
	if (strcmp (which, "all_complete") == 0)
	    shell_command (gtm_pref.command_all_complete);
    }
}


/* Update the check interface timeout */
void
update_check_interface_timeout (guint timeout)
{
    /* Set the new value */
    gtm_pref.chk_interf_time = timeout;
    gnome_config_set_int ("chk_interf_time", 
                          gtm_pref.chk_interf_time);

    /* Update the timeout */
    if (check_network_timeout_id != -1)
        gtk_timeout_remove (check_network_timeout_id);
    check_network_timeout_id =
        gtk_timeout_add (gtm_pref.chk_interf_time, check_network, NULL);
}


/* The main program where all begins */
int 
main (int argc, char *argv[])
{
    CORBA_ORB orb;
    GnomeClient *client;
    gchar *geometry;
    gchar *url_to_get = NULL;
    struct poptOption gtm_popt_options[] = {
	{ NULL, '\0', POPT_ARG_INCLUDE_TABLE, &bonobo_activation_popt_options,
	  0, NULL, NULL },
        { "get-url", 'g', POPT_ARG_STRING, &url_to_get, 0,
          N_ ("url to fetch"), N_ ("URL")},
        { NULL, '\0', 0, NULL, 0 } /* end the list */
    };
    
    startup = TRUE;

    /* Initialize CORBA and oaf */
    CORBA_exception_init (&ev);

    /* Initialize the application and CORBA stuff */
    gnome_init_with_popt_table ("gtm", VERSION, argc, argv,
				gtm_popt_options, 0, NULL);

    orb = bonobo_activation_orb_get ();

    /* Initialize the i18n stuff */
    bindtextdomain (PACKAGE, GNOMELOCALEDIR);
    bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
    textdomain (PACKAGE);
   

    gnome_window_icon_set_default_from_file (GNOME_ICONDIR "/gtm.png");

    /* If we have a URL from the command line connect to the running gtm
     * object and send him the URL to fetch */
    if (url_to_get != NULL) {
        GTM_Download gtm_download_client;
        gboolean result;

        gtm_download_client = 
            bonobo_activation_activate_from_id ("OAFIID:GNOME_GTM", 
						0, NULL, &ev);
	Exception(&ev);
        if (gtm_download_client != NULL) {
	  result = GTM_Download_add_url (gtm_download_client,
	                                 url_to_get, "", FALSE, FALSE, &ev);
            if (ev._major != CORBA_NO_EXCEPTION)
                g_printerr ("Exception thrown: %s\n", 
                            CORBA_exception_id (&ev));
            else if (result == FALSE) {
                g_print (_ ("URL:\n"));
                g_print ("%s\n", url_to_get);
                g_print (_ ("already in the list"));
                g_print ("\n");
            }
        } else {
            /* GTM Server couldn't be started */
            g_print (_ ("Couldn't start GTM"));
            g_print ("\n");
        }
        
        return 0;
    }

    /* Load program preferences */
    geometry = load_preferences ();

    /* Get the master client, that was hopefully connected to the
     * session manager int the 'gnome_init' call.  All communication to
     * the session manager will be done with this master client. */
    client = gnome_master_client ();
  
    /* Arrange to be told when something interesting happens. */
    gtk_signal_connect (GTK_OBJECT (client), "die",
			GTK_SIGNAL_FUNC (die), NULL);

    /* Initialize the main window */
    dl_file_list = main_window_create (geometry);
    g_free (geometry);

    /* Load file list */
    file_list_load_data (GTK_CLIST (dl_file_list));

    /* Install the timer */
    check_network_timeout_id = 
        gtk_timeout_add (gtm_pref.chk_interf_time, check_network, NULL);

    /* Install a signal handler that saves the state of the application */
    signal (SIGINT, signal_handler);
    signal (SIGTERM, signal_handler);
    signal (SIGHUP, signal_handler);
    signal (SIGPIPE, signal_handler);

    /* Test if another GTM is already running */
    if (corba_server_is_running()) {
        GtkWidget *error;
        g_print("GTM already running.\n");
        error = gnome_error_dialog_parented (_ (
		  "There is Another GNOME Transfer Manager running.\n"
	          "Only one GNOME Transfer Manager can be running."), 
	          GTK_WINDOW (gtm_app) );
        gnome_dialog_run_and_close (GNOME_DIALOG (error));
        exit (1);
    }


    /* Register gtm with OAF */
    register_corba_server(orb);

    /* Finished the startup */
    startup = FALSE;

    /* Enter the application main loop */
    gtk_main ();

    /* Destroy the timer */
    gtk_timeout_remove (check_network_timeout_id);
    
    return 0;
}
