/*  GTKtalog.
 *  Copyright (C) 1999  Mathieu VILLEGAS
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <config.h>
#if defined(OS_FREEBSD) || defined(OS_NETBSD) || defined(OS_OPENBSD)
#    include <sys/param.h>
#    include <sys/mount.h>
#endif
#include <gnome.h>
#include <fnmatch.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>

#include "search.h"
#include "categories.h"
#include "date.h"
#include "config_common.h"
#include "compare.h"
#include "edit.h"
#include "folder.h"
#include "io.h"
#include "interface.h"
#include "progressbar.h"
#include "thread_utils.h"

static gchar STRING_ALL[] = N_("All");
static gchar STRING_NONE[] = N_("None");
static gchar *STRING_SM[] = {
  N_("N/A"),
  N_("Exact match"),
  N_("Contains"),
  N_("Shell glob"),
  N_("Regular expression")
};

/* Search dialog */
static GtkWidget *search_dialog_window;
static GtkWidget *simple_search_entry;
static GtkWidget *less_button;
static GtkWidget *es_gnome_entry;
static GtkWidget *cat_combo;
static GtkWidget *information_entry;
static GtkWidget *information_gnome_entry;
static GtkWidget *desc_entry;
static GtkWidget *desc_gnome_entry;
static GtkWidget *disk_combo, *disk_combo_entry;
static GtkWidget *es_tmenu;
static GtkWidget *es_cs;
static GtkWidget *s_path_tmenu;
static GtkWidget *desc_tmenu;
static GtkWidget *desc_cs;
static GtkWidget *info_tmenu;
static GtkWidget *info_cs;
static GtkWidget *cat_replace_combo, *cat_replace_button;
static GtkWidget *desc_replace_entry, *desc_replace_button;
static GtkWidget *s_path_gnome_entry;
static GtkWidget *s_path_entry;
static GtkWidget *s_path_cs;
static GtkWidget *cat_combo_entry;
static GtkWidget *cat_replace_combo_entry;
static GtkWidget *checkdate1;
static GtkWidget *checkdate2;
static GtkWidget *dateedit1;
static GtkWidget *dateedit2;
static GtkWidget *checksize1;
static GtkWidget *checksize2;
static GtkWidget *size1;
static GtkWidget *size1_optionmenu;
static GtkWidget *size2;
static GtkWidget *size2_optionmenu;
static GtkWidget *scrolled_window;
static GtkWidget *search_list;
static GtkWidget *checkvfs;
static GtkWidget *info2desc_button;
static GtkWidget *info2cat_button;
static GtkWidget *display_dirs;
static GtkWidget *start_button;
static GtkWidget *stop_button;

static gboolean gc_descriptions_flag;

static FOLDER *local_racine;
static gboolean stop_search;

GNode *
find_node_with_parent (GNode * parent, gchar * name,
		       gboolean case_sensitive, FOLDER * racine)
{
  GNode *gntmp;
  GNode *result;
  FILE_DATA *fd;

  gntmp = g_node_first_child (parent);
  result = NULL;
  while ((result == NULL) && (gntmp))
    {
      fd = get_file_data_from_gnode (gntmp);
      if (is_dir (gntmp) || is_disk (gntmp))
	{
	  if (compare (name, fd->name->str, COMPARE_REGEX, case_sensitive) ==
	      0)
	    {
	      result = gntmp;
	    }
	  else
	    {
	      gntmp = g_node_next_sibling (gntmp);
	    }
	}
      else
	{
	  gntmp = g_node_next_sibling (gntmp);
	}
    }
  return (result);
}

gboolean
find_files_copy (GNode * gn, gpointer data)
{
  GPtrArray *search_racine = data;
  FILE_DATA *fd;

  progress_addCurrent (1, GNOMEAPPBAR_SEARCH);
  while (gtk_events_pending ())
    gtk_main_iteration ();

  if (g_node_depth (gn) <= 2)
    return (FALSE);

  if (is_unreadable (gn) == TRUE)
    return (FALSE);

  fd = get_file_data_from_gnode (gn);
  if ((!my_config->SEARCH_OPTIONS_display_dirs) && (is_dir (gn)))
    return (FALSE);
  if (((fd->a_parent_node_is_vfs != IS_VFS)
       && (fd->a_parent_node_is_vfs != IS_VFS_EXTENDED))
      || my_config->SEARCH_OPTIONS_search_in_vfs == TRUE)
    {
      /* Compare names */
      if (compare
	  (my_config->SEARCH_OPTIONS_filename, fd->name->str,
	   my_config->SEARCH_OPTIONS_file_regex,
	   my_config->SEARCH_OPTIONS_file_casesensitive))
	return (FALSE);
      g_ptr_array_add (search_racine, gn);

    }
  if (stop_search)
    progress_setCurrent (0, g_node_n_nodes
			 (my_config->SEARCH_OPTIONS_search_root,
			  G_TRAVERSE_ALL), GNOMEAPPBAR_SEARCH);

  return (stop_search);
}

gint
search_mode_from_text (gchar * text)
{
  gint i;
  i = 0;
  while (i < STRING_SM_NB)
    {
      if (strcmp (text, _(STRING_SM[i])) == 0)
	break;
      i++;
    }
  g_assert (i < STRING_SM_NB);
  return (i);
}

gchar *
make_string_escaped (gchar * s)
{
  gchar *tmp, *ptmp1, *ptmp2;
  gint n;
  tmp = s;
  n = 1;
  while (tmp[0])
    {
      if (tmp[0] < '0')
	n++;
      if ((tmp[0] > '9') && (tmp[0] < 'A'))
	n++;
      if ((tmp[0] > 'Z') && (tmp[0] < 'a'))
	n++;
      if (tmp[0] > 'z')
	n++;
      n++;
      tmp++;
    }
  tmp = (gchar *) g_malloc (sizeof (gchar) * (n + 4));
  ptmp1 = s;
  ptmp2 = tmp;
  ptmp2[0] = '.';
  ptmp2++;
  ptmp2[0] = '*';
  ptmp2++;
  while (ptmp1[0])
    {
      if (ptmp1[0] < '0')
	ptmp2[0] = '\\';
      if ((ptmp1[0] > '9') && (ptmp1[0] < 'A'))
	ptmp2[0] = '\\';
      if ((ptmp1[0] > 'Z') && (ptmp1[0] < 'a'))
	ptmp2[0] = '\\';
      if (ptmp1[0] > 'z')
	ptmp2[0] = '\\';
      ptmp2[0] = ptmp1[0];
      ptmp1++;
      ptmp2++;
    }
  ptmp2[0] = '.';
  ptmp2++;
  ptmp2[0] = '*';
  ptmp2++;
  ptmp2[0] = 0;
  return (tmp);
}

static void
cancel_clicked (GtkWidget * w, gpointer data)
{
  if (get_thread_status () != THREAD_SCAN_RUNNING_NO)
    return;

  progress_setGnomeAppBar (NULL, GNOMEAPPBAR_SEARCH, TRUE);
  gtk_widget_destroy (search_dialog_window);
  set_search_running (FALSE);
  menus_set_all_sensitive (TRUE);
  return;
}

static void
stop_clicked (GtkWidget * w, gpointer data)
{
  stop_search = TRUE;
}

static void
switch_startstop_search (gboolean start)
{
  if (start)
    {
      gtk_widget_show (start_button);
      gtk_widget_hide (stop_button);
    }
  else
    {
      gtk_widget_show (stop_button);
      gtk_widget_hide (start_button);
    }
}

gboolean
get_extended_search_options (FOLDER * racine)
{
  G_CONST_RETURN gchar *disk;
  gchar *tmp;
  gint mode_;
  GtkWidget *tmenu_widget;
  gboolean nothing_to_search = TRUE;

  /* filename *********** */
  my_config->SEARCH_OPTIONS_filename =
    g_strdup (gtk_entry_get_text (GTK_ENTRY (simple_search_entry)));


  tmenu_widget =
    gtk_menu_get_active (GTK_MENU
			 (gtk_option_menu_get_menu
			  (GTK_OPTION_MENU (es_tmenu))));
  mode_ =
    GPOINTER_TO_INT (gtk_object_get_user_data (GTK_OBJECT (tmenu_widget)));

  my_config->SEARCH_OPTIONS_file_casesensitive =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (es_cs));
  my_config->SEARCH_OPTIONS_file_regex = COMPARE_EXACT;

  my_config->previous_search_mode = mode_;
  switch (mode_)
    {
    case EVERYTHING_MATCHES:
      g_free (my_config->SEARCH_OPTIONS_filename);
      my_config->SEARCH_OPTIONS_filename = g_strdup (".*");
      my_config->SEARCH_OPTIONS_file_regex = COMPARE_REGEX;
      break;
    case CONTAINS:
      tmp = make_string_escaped (my_config->SEARCH_OPTIONS_filename);
      g_free (my_config->SEARCH_OPTIONS_filename);
      my_config->SEARCH_OPTIONS_filename = tmp;
      my_config->SEARCH_OPTIONS_file_regex = COMPARE_REGEX;
      nothing_to_search = FALSE;
      break;
    case GLOB:
      my_config->SEARCH_OPTIONS_file_regex = COMPARE_GLOB;
      nothing_to_search = FALSE;
      break;
    case COMPARE_EXACT:
      nothing_to_search = FALSE;
      break;
    case REGULAR_EXPRESSION:
      my_config->SEARCH_OPTIONS_file_regex = COMPARE_REGEX;
      nothing_to_search = FALSE;
      break;
    }

  /* filepath *********** */
  my_config->SEARCH_OPTIONS_filepath =
    g_strdup (gtk_entry_get_text (GTK_ENTRY (s_path_entry)));

  tmenu_widget =
    gtk_menu_get_active (GTK_MENU
			 (gtk_option_menu_get_menu
			  (GTK_OPTION_MENU (s_path_tmenu))));
  mode_ =
    GPOINTER_TO_INT (gtk_object_get_user_data (GTK_OBJECT (tmenu_widget)));

  my_config->SEARCH_OPTIONS_path_casesensitive =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (s_path_cs));;
  my_config->SEARCH_OPTIONS_path_regex = COMPARE_EXACT;

  switch (mode_)
    {
    case EVERYTHING_MATCHES:
      g_free (my_config->SEARCH_OPTIONS_filepath);
      my_config->SEARCH_OPTIONS_filepath = g_strdup (".*");
      my_config->SEARCH_OPTIONS_path_regex = COMPARE_REGEX;
      break;
    case CONTAINS:
      tmp = make_string_escaped (my_config->SEARCH_OPTIONS_filepath);
      g_free (my_config->SEARCH_OPTIONS_filepath);
      my_config->SEARCH_OPTIONS_filepath = tmp;
      my_config->SEARCH_OPTIONS_path_regex = COMPARE_REGEX;
      nothing_to_search = FALSE;
      break;
    case GLOB:
      my_config->SEARCH_OPTIONS_path_regex = COMPARE_GLOB;
      nothing_to_search = FALSE;
      break;
    case COMPARE_EXACT:
      nothing_to_search = FALSE;
      if (my_config->SEARCH_OPTIONS_filepath[0] != '/')
	{
	  tmp = g_strconcat ("/", my_config->SEARCH_OPTIONS_filepath, NULL);
	  g_free (my_config->SEARCH_OPTIONS_filepath);
	  my_config->SEARCH_OPTIONS_filepath = tmp;
	}
      tmp = my_config->SEARCH_OPTIONS_filepath;
      while (tmp[0])
	tmp++;
      tmp--;
      if (tmp[0] != '/')
	{
	  tmp = g_strconcat (my_config->SEARCH_OPTIONS_filepath, "/", NULL);
	  g_free (my_config->SEARCH_OPTIONS_filepath);
	  my_config->SEARCH_OPTIONS_filepath = tmp;

	}
      break;
    case REGULAR_EXPRESSION:
      my_config->SEARCH_OPTIONS_path_regex = COMPARE_REGEX;
      nothing_to_search = FALSE;
      break;
    }

  /* dirs *********** */
  my_config->SEARCH_OPTIONS_display_dirs =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (display_dirs));

  /* size *********** */
  my_config->SEARCH_OPTIONS_size1 =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (checksize1));
  my_config->SEARCH_OPTIONS_size2 =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (checksize2));
  if (my_config->SEARCH_OPTIONS_size1)
    {
      my_config->SEARCH_OPTIONS_size_val1 =
	gtk_spin_button_get_value_as_int ((GtkSpinButton *) size1);
      tmenu_widget =
	gtk_menu_get_active (GTK_MENU
			     (gtk_option_menu_get_menu
			      (GTK_OPTION_MENU (size1_optionmenu))));
      my_config->SEARCH_OPTIONS_size_val1 *=
	GPOINTER_TO_INT (gtk_object_get_user_data
			 (GTK_OBJECT (tmenu_widget)));
      nothing_to_search = FALSE;
    }
  if (my_config->SEARCH_OPTIONS_size2)
    {
      my_config->SEARCH_OPTIONS_size_val2 =
	gtk_spin_button_get_value_as_int ((GtkSpinButton *) size2);
      tmenu_widget =
	gtk_menu_get_active (GTK_MENU
			     (gtk_option_menu_get_menu
			      (GTK_OPTION_MENU (size2_optionmenu))));
      my_config->SEARCH_OPTIONS_size_val2 *=
	GPOINTER_TO_INT (gtk_object_get_user_data
			 (GTK_OBJECT (tmenu_widget)));
      nothing_to_search = FALSE;
    }

  /* date *********** */
  my_config->SEARCH_OPTIONS_date1 =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (checkdate1));
  if (my_config->SEARCH_OPTIONS_date1)
    {
      nothing_to_search = FALSE;
    }
  my_config->SEARCH_OPTIONS_date2 =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (checkdate2));
  if (my_config->SEARCH_OPTIONS_date2)
    {
      nothing_to_search = FALSE;
    }

  my_config->SEARCH_OPTIONS_date_val1 =
    gnome_date_edit_get_date (GNOME_DATE_EDIT (dateedit1));
  my_config->SEARCH_OPTIONS_date_val2 =
    gnome_date_edit_get_date (GNOME_DATE_EDIT (dateedit2));

  /* Category *********** */
  my_config->SEARCH_OPTIONS_category =
    g_strdup (gtk_entry_get_text (GTK_ENTRY (cat_combo_entry)));
  if (strcmp (my_config->SEARCH_OPTIONS_category, STRING_ALL))
    {
      nothing_to_search = FALSE;
    }

  /* Description *********** */
  my_config->SEARCH_OPTIONS_description =
    g_strdup (gtk_entry_get_text (GTK_ENTRY (desc_entry)));

  tmenu_widget =
    gtk_menu_get_active (GTK_MENU
			 (gtk_option_menu_get_menu
			  (GTK_OPTION_MENU (desc_tmenu))));
  mode_ =
    GPOINTER_TO_INT (gtk_object_get_user_data (GTK_OBJECT (tmenu_widget)));

  my_config->SEARCH_OPTIONS_desc_casesensitive =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (desc_cs));
  my_config->SEARCH_OPTIONS_desc_regex = COMPARE_EXACT;

  switch (mode_)
    {
    case EVERYTHING_MATCHES:
      g_free (my_config->SEARCH_OPTIONS_description);
      my_config->SEARCH_OPTIONS_description = g_strdup (".*");
      my_config->SEARCH_OPTIONS_desc_regex = COMPARE_REGEX;
      break;
    case CONTAINS:
      tmp = make_string_escaped (my_config->SEARCH_OPTIONS_description);
      g_free (my_config->SEARCH_OPTIONS_description);
      my_config->SEARCH_OPTIONS_description = tmp;
      my_config->SEARCH_OPTIONS_desc_regex = COMPARE_REGEX;
      nothing_to_search = FALSE;
      break;
    case GLOB:
      my_config->SEARCH_OPTIONS_desc_regex = COMPARE_GLOB;
      nothing_to_search = FALSE;
      break;
    case COMPARE_EXACT:
      nothing_to_search = FALSE;
      break;
    case REGULAR_EXPRESSION:
      my_config->SEARCH_OPTIONS_desc_regex = COMPARE_REGEX;
      nothing_to_search = FALSE;
      break;
    }

  /* Information *********** */
  my_config->SEARCH_OPTIONS_information =
    g_strdup (gtk_entry_get_text (GTK_ENTRY (information_entry)));

  tmenu_widget =
    gtk_menu_get_active (GTK_MENU
			 (gtk_option_menu_get_menu
			  (GTK_OPTION_MENU (info_tmenu))));
  mode_ =
    GPOINTER_TO_INT (gtk_object_get_user_data (GTK_OBJECT (tmenu_widget)));

  my_config->SEARCH_OPTIONS_info_casesensitive =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (info_cs));
  my_config->SEARCH_OPTIONS_info_regex = COMPARE_EXACT;

  switch (mode_)
    {
    case EVERYTHING_MATCHES:
      g_free (my_config->SEARCH_OPTIONS_information);
      my_config->SEARCH_OPTIONS_information = g_strdup (".*");
      my_config->SEARCH_OPTIONS_info_regex = COMPARE_REGEX;
      break;
    case CONTAINS:
      tmp = make_string_escaped (my_config->SEARCH_OPTIONS_information);
      g_free (my_config->SEARCH_OPTIONS_information);
      my_config->SEARCH_OPTIONS_information = tmp;
      my_config->SEARCH_OPTIONS_info_regex = COMPARE_REGEX;
      nothing_to_search = FALSE;
      break;
    case GLOB:
      my_config->SEARCH_OPTIONS_info_regex = COMPARE_GLOB;
      nothing_to_search = FALSE;
      break;
    case COMPARE_EXACT:
      nothing_to_search = FALSE;
      break;
    case REGULAR_EXPRESSION:
      my_config->SEARCH_OPTIONS_info_regex = COMPARE_REGEX;
      nothing_to_search = FALSE;
      break;
    }

  /* find root */
  disk = gtk_entry_get_text (GTK_ENTRY (disk_combo_entry));

  my_config->SEARCH_OPTIONS_search_root = NULL;
  if (strcmp (disk, STRING_ALL))
    {
      my_config->SEARCH_OPTIONS_search_root =
	find_node_with_parent (racine->tree, (gchar *) disk, FALSE, racine);
    }
  if (my_config->SEARCH_OPTIONS_search_root == NULL)
    {
      my_config->SEARCH_OPTIONS_search_root = racine->tree;
    }
  else
    {
      nothing_to_search = FALSE;
    }

  my_config->SEARCH_OPTIONS_search_in_vfs =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (checkvfs));
  return (nothing_to_search);
}

void
get_simple_search_options (FOLDER * racine)
{
  /* read what the user entered */
  my_config->SEARCH_OPTIONS_filename =
    g_strdup (gtk_entry_get_text (GTK_ENTRY (simple_search_entry)));
  my_config->SEARCH_OPTIONS_file_casesensitive = TRUE;
  my_config->SEARCH_OPTIONS_file_regex = COMPARE_GLOB;

  my_config->SEARCH_OPTIONS_filepath = g_strdup (".*");
  my_config->SEARCH_OPTIONS_path_casesensitive = TRUE;
  my_config->SEARCH_OPTIONS_path_regex = COMPARE_REGEX;

  my_config->SEARCH_OPTIONS_display_dirs = TRUE;

  my_config->SEARCH_OPTIONS_size1 = FALSE;
  my_config->SEARCH_OPTIONS_size2 = FALSE;
  my_config->SEARCH_OPTIONS_size_val1 = 0;
  my_config->SEARCH_OPTIONS_size_val2 = 0;

  my_config->SEARCH_OPTIONS_date1 = FALSE;
  my_config->SEARCH_OPTIONS_date2 = FALSE;

  my_config->SEARCH_OPTIONS_search_in_vfs = FALSE;

  my_config->SEARCH_OPTIONS_category = g_strdup (STRING_ALL);

  my_config->SEARCH_OPTIONS_description = g_strdup (".*");
  my_config->SEARCH_OPTIONS_desc_casesensitive = TRUE;
  my_config->SEARCH_OPTIONS_desc_regex = COMPARE_REGEX;

  my_config->SEARCH_OPTIONS_information = g_strdup (".*");
  my_config->SEARCH_OPTIONS_info_casesensitive = TRUE;
  my_config->SEARCH_OPTIONS_info_regex = COMPARE_REGEX;

  /* find root */
  my_config->SEARCH_OPTIONS_search_root = racine->tree;
}

void
start_search (GtkWidget * w, gpointer data)
{

  GPtrArray *search_racine;
  gint i, n;
  gint nc;
  GNode *gn;
  FILE_DATA *fd;
  GString *current_filepath;
  const gchar *tmpstr;
  gchar *gs_cat_replace, *gs_desc_replace;
  const gchar *result[8];
  gint row;
  PIX *pix;
  gchar *nothing = " ";
  gboolean is_search_simple;
  FOLDER *racine = data;
  gboolean file_path_exists = FALSE;

  if (get_thread_status () != THREAD_SCAN_RUNNING_NO)
    return;

  is_search_simple =
    GPOINTER_TO_INT (gtk_object_get_user_data (GTK_OBJECT (less_button)));
  if (!is_search_simple)
    {
      if (get_extended_search_options (racine))
	{


	  g_free (my_config->SEARCH_OPTIONS_filename);
	  g_free (my_config->SEARCH_OPTIONS_filepath);
	  g_free (my_config->SEARCH_OPTIONS_information);
	  g_free (my_config->SEARCH_OPTIONS_category);
	  g_free (my_config->SEARCH_OPTIONS_description);

	  reinit_thread_status ();
	  ERROR_DIALOG (_("Please specify something to search\n"
			  "At least check that the file name match mode is not N/A!"),
			search_dialog_window);
	  return;
	}
    }
  else
    {
      get_simple_search_options (racine);
    }

  search_racine = g_ptr_array_new ();
  set_thread_status_to_running ();

  gtk_clist_clear (GTK_CLIST (search_list));
  gtk_clist_freeze (GTK_CLIST (search_list));

  gc_descriptions_flag = FALSE;

  progress_reset ();
  progress_setTotal (0, g_node_n_nodes
		     (my_config->SEARCH_OPTIONS_search_root, G_TRAVERSE_ALL),
		     GNOMEAPPBAR_SEARCH);
  progress_setCurrent (0, 0, GNOMEAPPBAR_SEARCH);
  progress_setStatus (_("Searching..."), GNOMEAPPBAR_SEARCH);
  progress_createTimeout (GINT_TO_POINTER (GNOMEAPPBAR_SEARCH));
  stop_search = FALSE;
  switch_startstop_search (FALSE);
  local_racine = racine;
  g_node_traverse (my_config->SEARCH_OPTIONS_search_root, G_PRE_ORDER,
		   G_TRAVERSE_ALL, -1, find_files_copy, search_racine);

  n = search_racine->len;
  if (is_search_simple)
    {
      file_path_exists = TRUE;
    }
  else
    {
  for (i = 0; ((i < n) && !stop_search); i++)
    {
      gn = g_ptr_array_index (search_racine, i);
      fd = get_file_data_from_gnode (gn);
      /* Compare paths */
      current_filepath = get_path_from_node (gn);
      if (compare
	  (my_config->SEARCH_OPTIONS_filepath, current_filepath->str,
	   my_config->SEARCH_OPTIONS_path_regex,
	   my_config->SEARCH_OPTIONS_path_casesensitive))
	{
	  g_ptr_array_index (search_racine, i) = NULL;
	}
      else
	file_path_exists = TRUE;
      g_string_free (current_filepath, TRUE);
    }
    }
  if ((!is_search_simple) && !stop_search)
    {
      /* Compare categories */
      if (strcmp (my_config->SEARCH_OPTIONS_category, STRING_ALL))
	{
	  for (i = 0; ((i < n) && !stop_search); i++)
	    {
	      gn = g_ptr_array_index (search_racine, i);
	      if (gn)
		{
		  fd = get_file_data_from_gnode (gn);
		  tmpstr = get_categorie (racine, fd->categorie);
		  if (tmpstr)
		    {
		      if (compare
			  (my_config->SEARCH_OPTIONS_category, tmpstr,
			   COMPARE_EXACT, FALSE))
			g_ptr_array_index (search_racine, i) = NULL;
		    }
		  else
		    {
		      g_ptr_array_index (search_racine, i) = NULL;
		    }
		}
	    }
	}
      /* Compare descriptions */
      if (strcmp (my_config->SEARCH_OPTIONS_description, ".*"))
	{
	  for (i = 0; ((i < n) && !stop_search); i++)
	    {
	      gn = g_ptr_array_index (search_racine, i);
	      if (gn)
		{
		  fd = get_file_data_from_gnode (gn);
		  tmpstr = get_description (racine, fd->description);
		  if (compare
		      (my_config->SEARCH_OPTIONS_description, tmpstr,
		       my_config->SEARCH_OPTIONS_desc_regex,
		       my_config->SEARCH_OPTIONS_desc_casesensitive))
		    g_ptr_array_index (search_racine, i) = NULL;
		}
	    }
	}

      /* Compare information */
      if (strcmp (my_config->SEARCH_OPTIONS_information, ".*"))
	{
	  for (i = 0; ((i < n) && !stop_search); i++)
	    {
	      gn = g_ptr_array_index (search_racine, i);
	      if (gn)
		{
		  fd = get_file_data_from_gnode (gn);
		  tmpstr = folder_get_information_from_file_data (fd);
		  if (compare
		      (my_config->SEARCH_OPTIONS_information, tmpstr,
		       my_config->SEARCH_OPTIONS_info_regex,
		       my_config->SEARCH_OPTIONS_info_casesensitive))
		    g_ptr_array_index (search_racine, i) = NULL;
		}
	    }
	}
      /* Compare size */
      if (my_config->SEARCH_OPTIONS_size1)
	{
	  for (i = 0; ((i < n) && !stop_search); i++)
	    {
	      gn = g_ptr_array_index (search_racine, i);
	      if (gn)
		{
		  fd = get_file_data_from_gnode (gn);
		  if (fd->taille < my_config->SEARCH_OPTIONS_size_val1)
		    g_ptr_array_index (search_racine, i) = NULL;
		  {
		  }
		}
	    }
	}
      if (my_config->SEARCH_OPTIONS_size2)
	{
	  for (i = 0; ((i < n) && !stop_search); i++)
	    {
	      gn = g_ptr_array_index (search_racine, i);
	      if (gn)
		{
		  fd = get_file_data_from_gnode (gn);
		  if (fd->taille > my_config->SEARCH_OPTIONS_size_val2)
		    g_ptr_array_index (search_racine, i) = NULL;
		  {
		  }
		}
	    }
	}

      /* Compare date */
      if (my_config->SEARCH_OPTIONS_date1)
	{
	  for (i = 0; ((i < n) && !stop_search); i++)
	    {
	      gn = g_ptr_array_index (search_racine, i);
	      if (gn)
		{
		  fd = get_file_data_from_gnode (gn);
		  if (fd->date < my_config->SEARCH_OPTIONS_date_val1)
		    {
		      g_ptr_array_index (search_racine, i) = NULL;
		    }
		}
	    }
	}
      if (my_config->SEARCH_OPTIONS_date2)
	{
	  for (i = 0; ((i < n) && !stop_search); i++)
	    {
	      gn = g_ptr_array_index (search_racine, i);
	      if (gn)
		{
		  fd = get_file_data_from_gnode (gn);
		  if (fd->date > my_config->SEARCH_OPTIONS_date_val2)
		    g_ptr_array_index (search_racine, i) = NULL;
		}
	    }
	}


      /* Replace ? */

      if (gtk_toggle_button_get_active
	  (GTK_TOGGLE_BUTTON (cat_replace_button)))
	{
	  gs_cat_replace =
	    g_strdup (gtk_entry_get_text
		      (GTK_ENTRY (cat_replace_combo_entry)));
	  nc = string_to_index (racine->categories, gs_cat_replace);
	  for (i = 0; ((i < n) && !stop_search); i++)
	    {
	      gn = g_ptr_array_index (search_racine, i);
	      if (gn)
		{
		  fd = get_file_data_from_gnode (gn);
		  if (strcmp (gs_cat_replace, _(STRING_NONE)))
		    fd->categorie = nc;
		  else
		    fd->categorie = 0;
		}
	    }
	}
      if (gtk_toggle_button_get_active
	  (GTK_TOGGLE_BUTTON (desc_replace_button)))
	{
	  gs_desc_replace =
	    g_strdup (gtk_entry_get_text (GTK_ENTRY (desc_replace_entry)));
	  nc = description_add (racine->descriptions, gs_desc_replace);
	  if (strlen (gs_desc_replace))
	    {
	      for (i = 0; ((i < n) && !stop_search); i++)
		{
		  gn = g_ptr_array_index (search_racine, i);
		  if (gn)
		    {
		      fd = get_file_data_from_gnode (gn);
		      fd->description = nc;
		      racine->is_modified = TRUE;
		    }
		}
	    }
	  else
	    {
	      for (i = 0; ((i < n) && !stop_search); i++)
		{
		  gn = g_ptr_array_index (search_racine, i);
		  if (gn)
		    {
		      fd = get_file_data_from_gnode (gn);
		      fd->description = 0;
		      racine->is_modified = TRUE;
		    }
		}
	      gc_descriptions_flag = TRUE;
	    }
	}

      if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (info2cat_button)))
	{
	  for (i = 0; ((i < n) && !stop_search); i++)
	    {
	      gn = g_ptr_array_index (search_racine, i);
	      if (gn)
		{
		  fd = get_file_data_from_gnode (gn);
		  if (folder_get_informations (fd->node))
		    {
		      gchar *newcat =
			g_strdup (folder_get_informations (fd->node));
		      if (newcat)
			{
			  if (compare
			      ("^[\t ]*$", newcat, COMPARE_REGEX, FALSE))
			    {
			      nc =
				string_to_index (racine->categories, newcat);
			      if (!nc)
				{
				  g_ptr_array_add (racine->categories,
						   g_string_new (newcat));
				  nc =
				    string_to_index (racine->categories,
						     newcat);


				}
			      if (nc)
				{
				  fd = get_file_data_from_gnode (gn);
				  fd->categorie = nc;
				  racine->is_modified = TRUE;
				}
			      g_free (newcat);
			    }
			}
		    }
		}
	    }
	}

      if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (info2desc_button)))
	{
	  for (i = 0; ((i < n) && !stop_search); i++)
	    {
	      gn = g_ptr_array_index (search_racine, i);
	      if (gn)
		{
		  fd = get_file_data_from_gnode (gn);
		  if (folder_get_informations (fd->node))
		    {
		      nc =
			description_add (racine->descriptions,
					 g_strdup (folder_get_informations
						   (fd->node)));
		      fd->description = nc;
		      racine->is_modified = TRUE;
		    }
		}
	    }
	}
    }

  for (i = 0; i < n; i++)
    {
      gn = g_ptr_array_index (search_racine, i);
      if (gn)
	{
	  fd = get_file_data_from_gnode (gn);
	  result[0] = fd->name->str;
	  result[1] = get_disk_from_node (gn);
	  result[2] = get_path_from_node (gn)->str;
	  if (is_dir (gn) == TRUE)
	    result[3] = g_strdup ("0");
	  else
	    result[3] = get_size (fd->taille);
	  result[4] =
	    get_time (fd->date, my_config->date_format_type,
		      my_config->date_format);
	  result[5] = get_categorie (racine, fd->categorie);
	  if (!result[5])
	    result[5] = nothing;
	  result[6] = get_description (racine, fd->description);
	  if (!result[6])
	    result[6] = nothing;
	  result[7] = folder_get_informations (fd->node);
	  if ((!result[7]) || (fd->a_parent_node_is_vfs == IS_VFS))
	    result[7] = nothing;
	  row = gtk_clist_append (GTK_CLIST (search_list), (gchar **) result);
	  /* pixmap depending on the extension */

	  pix = get_pix_from_gnode (gn);

	  gtk_clist_set_pixtext (GTK_CLIST (search_list), row, 0,
				 fd->name->str, 3, pix->pixmap, *(pix->mask));

	  gtk_clist_set_row_data (GTK_CLIST (search_list), row, gn);

	}
    }
  for (i = 0; i < n; i++)
    {
      g_ptr_array_index (search_racine, i) = NULL;
    }
  g_ptr_array_free (search_racine, FALSE);

  if (gc_descriptions_flag == TRUE)
    description_garbage_collection ();

  gtk_clist_sort (GTK_CLIST (search_list));
  gtk_clist_columns_autosize (GTK_CLIST (search_list));
  gtk_clist_thaw (GTK_CLIST (search_list));
  switch_startstop_search (TRUE);

  g_free (my_config->SEARCH_OPTIONS_filename);
  g_free (my_config->SEARCH_OPTIONS_filepath);
  g_free (my_config->SEARCH_OPTIONS_information);
  g_free (my_config->SEARCH_OPTIONS_category);
  g_free (my_config->SEARCH_OPTIONS_description);
  if (!file_path_exists)
    {
      if (my_config->SEARCH_OPTIONS_search_root == racine->tree)
	{
	  WARNING_DIALOG (_("No file was found."), search_dialog_window);
	}
      else
	{
	  WARNING_DIALOG (_
			  ("No file was found.\nMaybe the path you specified was wrong?"),
		      search_dialog_window);
	}
    }
  reinit_thread_status ();
}

static void
destroy_window (GtkWidget * w, GdkEventKey * event, gpointer data)
{
  if (get_thread_status () != THREAD_SCAN_RUNNING_NO)
    return;
  cancel_clicked (w, data);
}

static void
selection_made (GtkWidget * clist, gint row, gint column,
		GdkEventButton * event, gpointer data)
{
  FOLDER *racine = data;
  GNode *gn;
  gchar *rep;
  gchar *path;
  gchar *datas;
  gchar *name;
  gint8 pouet;
  GdkPixmap *da_pix;
  GdkBitmap *da_mask;

  if (get_thread_status () != THREAD_SCAN_RUNNING_NO)
    return;

  if (event != NULL && event->button == 1 && event->type == GDK_2BUTTON_PRESS)
    {
      gtk_clist_get_pixtext (GTK_CLIST (clist), row, 0, &name, &pouet,
			     &da_pix, &da_mask);
      gn = gtk_clist_get_row_data ((GtkCList *) clist, row);

      gtk_clist_get_text (GTK_CLIST (search_list), row, 1, &rep);
      gtk_clist_get_text (GTK_CLIST (search_list), row, 2, &path);
      if (is_dir (gn) == TRUE)
	{
	  int l = strlen (rep) + strlen (path) + 6 + strlen (name);
	  datas = (char *) g_malloc (l * sizeof (char));
	  snprintf (datas, l, "[%s]->%s%s/", rep, path, name);
	}
      else
	{
	  int l = strlen (rep) + strlen (path) + 5;
	  datas = (char *) g_malloc (l * sizeof (char));
	  snprintf (datas, l, "[%s]->%s", rep, path);
	}
      gtk_entry_set_text (GTK_ENTRY (path_entry), datas);
      g_free (datas);
      if ((is_file (gn) == FALSE) && (is_unreadable (gn) == FALSE))
	{
	  list_folder (gn, TRUE);
	}
      else
	{
	  list_folder (gn->parent, TRUE);
	}
    }
  else
    {
      gn = gtk_clist_get_row_data ((GtkCList *) clist, row);
      racine->selected_folder = gn;
    }
}

static void
unselection_made (GtkWidget * clist, gint row, gint column,
		  GdkEventButton * event, gpointer data)
{
  FOLDER *racine = data;

  if (get_thread_status () != THREAD_SCAN_RUNNING_NO)
    return;

  racine->selected_folder = NULL;
}

static void
sort_list (GtkWidget * wid, gint column, gpointer data)
{
  FOLDER *racine = data;

  if (get_thread_status () != THREAD_SCAN_RUNNING_NO)
    return;

  racine->selected_folder = NULL;
  gtk_clist_set_sort_column (GTK_CLIST (search_list), (gint) column);
  if (column == 3)
    {
      gtk_clist_set_compare_func (GTK_CLIST (search_list), clist_cmp_2gint);
    }
  else if (column == 4)
    {
      gtk_clist_set_compare_func (GTK_CLIST (search_list), clist_cmp_2dates);
    }
  else
    {
      gtk_clist_set_compare_func (GTK_CLIST (search_list), NULL);
    }
  gtk_clist_sort (GTK_CLIST (search_list));
}

static void
cb_entry_NA1 (GtkWidget * w, gpointer data)
{
  GtkWidget *tmenu_widget;
  GtkWidget *tmenu = NULL, *entry = NULL, *cs = NULL;

  if (data == s_path_tmenu)
    {
      tmenu = s_path_tmenu;
      entry = s_path_gnome_entry;
      cs = s_path_cs;
    }
  else if (data == desc_tmenu)
    {
      tmenu = desc_tmenu;
      entry = desc_gnome_entry;
      cs = desc_cs;
    }
  else if (data == info_tmenu)
    {
      tmenu = info_tmenu;
      entry = information_gnome_entry;
      cs = info_cs;
    }
  if (tmenu)
    {
      tmenu_widget =
	gtk_menu_get_active (GTK_MENU
			     (gtk_option_menu_get_menu
			      (GTK_OPTION_MENU (tmenu))));
      if (GPOINTER_TO_INT
	  (gtk_object_get_user_data (GTK_OBJECT (tmenu_widget))) !=
	  EVERYTHING_MATCHES)
	{
	  gtk_widget_set_sensitive (GTK_WIDGET (entry), TRUE);
	  gtk_widget_set_sensitive (GTK_WIDGET (cs), TRUE);
	}
      else
	{
	  gtk_widget_set_sensitive (GTK_WIDGET (entry), FALSE);
	  gtk_widget_set_sensitive (GTK_WIDGET (cs), FALSE);
	}
    }
}

static void
cb_set_sensitive (GtkWidget * w, gpointer data)
{
  GtkWidget *w2 = data;
  gtk_widget_set_sensitive (GTK_WIDGET (w2),
			    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
							  (w)));
}

static void
cb_set_visible (GtkWidget * w, gpointer data)
{
  GtkWidget *w2 = data;
  gboolean mode = GPOINTER_TO_INT
    (gtk_object_get_user_data (GTK_OBJECT (less_button)));

  if (mode)
    {
      gtk_widget_show (w2);
      gtk_label_set_text (GTK_LABEL ((GTK_BIN (w))->child), _("Less <<"));
      gtk_object_set_user_data (GTK_OBJECT (less_button),
				GINT_TO_POINTER (FALSE));
    }
  else
    {
      gtk_widget_hide (w2);
      gtk_label_set_text (GTK_LABEL ((GTK_BIN (w))->child), _("More >>"));
      gtk_object_set_user_data (GTK_OBJECT (less_button),
				GINT_TO_POINTER (TRUE));
    }
  if ((my_config->new_search_type == SEARCH_TYPE_LAST_SIMPLE)
      || (my_config->new_search_type == SEARCH_TYPE_LAST_SIMPLE))
    {
      if (mode)
	my_config->new_search_type = SEARCH_TYPE_LAST_EXTENDED;
      else
	my_config->new_search_type = SEARCH_TYPE_LAST_SIMPLE;
    }
}

void
create_search_dialog (FOLDER * racine)
{
  GtkWidget *appbar1;
  GtkWidget *frame, *extended_search_frame;
  GtkWidget *cancel_button;
  GtkWidget *hbuttonbox;
  GtkWidget *main_box;
  GtkWidget *vbox1;
  GtkWidget *hbox2, *hbox_copy;
  GtkObject *adj;
  GtkWidget *menuitem;
  GtkWidget *label;
  GtkWidget *table, *ds_table;
  GtkWidget *ext_search_notebook;
  GtkWidget *es_tmenu_menu, *es_tmenu_menuitem;
  GtkWidget *s_path_tmenu_menu, *s_path_tmenu_menuitem;
  GtkWidget *size1_optionmenu_menu;
  GtkWidget *size2_optionmenu_menu;
  GtkWidget *info_tmenu_menu, *info_tmenu_menuitem;
  GtkWidget *desc_tmenu_menu, *desc_tmenu_menuitem;
  GtkAccelGroup *accel_group;
  GList *tcombo_items = NULL;
  gint tmenu_items[] =
    { EVERYTHING_MATCHES, EXACT_MATCH, CONTAINS, GLOB, REGULAR_EXPRESSION, -1
  };
  GList *disk_list, *cat_list;
  gchar *titles[8] =
    { _("File Name"), _("Disk"), _("Path"), _("Size"), _("Date"),
    _("Category"), _("Description"), _("Information")
  };
  gint i, prevmode;

  reinit_thread_status ();

  accel_group = gtk_accel_group_new ();

  tcombo_items =
    g_list_append (tcombo_items, _(STRING_SM[EVERYTHING_MATCHES]));
  tcombo_items = g_list_append (tcombo_items, _(STRING_SM[EXACT_MATCH]));
  tcombo_items = g_list_append (tcombo_items, _(STRING_SM[CONTAINS]));
  tcombo_items = g_list_append (tcombo_items, _(STRING_SM[GLOB]));
  tcombo_items =
    g_list_append (tcombo_items, _(STRING_SM[REGULAR_EXPRESSION]));

/* Make the window */
  search_dialog_window = gnome_app_new (PACKAGE, _("Search"));

  gtk_window_set_transient_for (GTK_WINDOW (search_dialog_window),
				GTK_WINDOW (main_window));

  gtk_window_set_position (GTK_WINDOW (search_dialog_window),
			   GTK_WIN_POS_MOUSE);
  gtk_window_set_policy (GTK_WINDOW (search_dialog_window), FALSE, TRUE,
			 FALSE);
  gtk_signal_connect (GTK_OBJECT (search_dialog_window), "delete_event",
		      G_CALLBACK (destroy_window), NULL);

/* Status and progress bar */
  appbar1 = gnome_appbar_new (TRUE, TRUE, GNOME_PREFERENCES_NEVER);
  gtk_widget_ref (appbar1);
  gtk_object_set_data_full (GTK_OBJECT (search_dialog_window),
			    "appbar_search", appbar1,
			    (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (appbar1);
  gnome_app_set_statusbar (GNOME_APP (search_dialog_window), appbar1);
  progress_setGnomeAppBar (appbar1, GNOMEAPPBAR_SEARCH, TRUE);



  main_box = gtk_vbox_new (FALSE, 4);
  gnome_app_set_contents (GNOME_APP (search_dialog_window), main_box);

/*********************************************************************/
/* 1st frame *********************************************************/
/*********************************************************************/

  frame = gtk_frame_new (_("Query"));
  gtk_frame_set_label_align (GTK_FRAME (frame), 0.0, 0.0);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_OUT);
  gtk_box_pack_start (GTK_BOX (main_box), frame, FALSE, FALSE, 0);
  gtk_widget_show (frame);

  hbox2 = gtk_hbox_new (FALSE, 4);
  gtk_container_add (GTK_CONTAINER (frame), hbox2);
  gtk_widget_show (hbox2);

  label = gtk_label_new (_("File name:"));
  gtk_widget_show (label);
  gtk_box_pack_start (GTK_BOX (hbox2), label, FALSE, FALSE, 2);

  es_gnome_entry = gnome_entry_new ("gtktalog_es_gnome_entry");
  gtk_widget_show (es_gnome_entry);
  gtk_box_pack_start (GTK_BOX (hbox2), es_gnome_entry, TRUE, TRUE, 0);


  simple_search_entry = gnome_entry_gtk_entry (GNOME_ENTRY (es_gnome_entry));

  gtk_signal_connect (GTK_OBJECT (simple_search_entry), "activate",
		      G_CALLBACK (start_search), racine);
  gtk_entry_set_text (GTK_ENTRY (simple_search_entry), "");
  gtk_widget_show (simple_search_entry);

  hbuttonbox = gtk_hbutton_box_new ();
  gtk_widget_show (hbuttonbox);
  gtk_box_pack_start (GTK_BOX (hbox2), hbuttonbox, FALSE, FALSE, 4);
  gtk_button_box_set_spacing (GTK_BUTTON_BOX (hbuttonbox), 15);

  less_button = gtk_button_new_with_label (_("More >>"));
  {
    gboolean tmp = FALSE;
    switch (my_config->new_search_type)
      {
      case SEARCH_TYPE_SIMPLE:
	tmp = FALSE;
	break;

      case SEARCH_TYPE_EXTENDED:
	tmp = TRUE;
	break;

      case SEARCH_TYPE_LAST_SIMPLE:
	tmp = FALSE;
	break;

      case SEARCH_TYPE_LAST_EXTENDED:
	tmp = TRUE;
	break;

      default:
	g_assert_not_reached ();
	break;
      }
    gtk_object_set_user_data (GTK_OBJECT (less_button),
			      GINT_TO_POINTER (tmp));
  }
  gtk_widget_show (less_button);
  gtk_container_add (GTK_CONTAINER (hbuttonbox), less_button);
  GTK_WIDGET_SET_FLAGS (less_button, GTK_CAN_DEFAULT);

/************** Extended search **************************************/

  extended_search_frame = gtk_frame_new (_("Options"));
  gtk_box_pack_start (GTK_BOX (main_box), extended_search_frame, FALSE, FALSE,
		      2);
  gtk_container_set_border_width (GTK_CONTAINER (extended_search_frame), 4);
  gtk_widget_show (extended_search_frame);

  ext_search_notebook = gtk_notebook_new ();
  gtk_widget_show (ext_search_notebook);
  gtk_container_add (GTK_CONTAINER (extended_search_frame),
		     ext_search_notebook);
  gtk_container_set_border_width (GTK_CONTAINER (ext_search_notebook), 4);

/**** Name and location ***************/
  vbox1 = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (vbox1);
  gtk_container_add (GTK_CONTAINER (ext_search_notebook), vbox1);
  gtk_container_set_border_width (GTK_CONTAINER (vbox1), 4);

  frame = gtk_frame_new (_("Name"));
  gtk_widget_show (frame);
  gtk_box_pack_start (GTK_BOX (vbox1), frame, FALSE, FALSE, 2);

  table = gtk_table_new (1, 3, FALSE);
  gtk_widget_show (table);
  gtk_container_add (GTK_CONTAINER (frame), table);
  gtk_container_set_border_width (GTK_CONTAINER (table), 3);
  gtk_table_set_col_spacings (GTK_TABLE (table), 6);

  es_tmenu = gtk_option_menu_new ();
  gtk_widget_show (es_tmenu);
  gtk_table_attach (GTK_TABLE (table), es_tmenu, 1, 2, 0, 1,
		    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);
  es_tmenu_menu = gtk_menu_new ();
  gtk_widget_show (es_tmenu_menu);
  i = 0;
  prevmode = 0;
  while (tmenu_items[i] >= 0)
    {
      es_tmenu_menuitem =
	gtk_menu_item_new_with_label (_(STRING_SM[tmenu_items[i]]));
      gtk_object_set_user_data (GTK_OBJECT (es_tmenu_menuitem),
				GINT_TO_POINTER (tmenu_items[i]));
      gtk_widget_show (es_tmenu_menuitem);
      gtk_menu_append (GTK_MENU (es_tmenu_menu), es_tmenu_menuitem);
      if (my_config->previous_search_mode == tmenu_items[i])
	prevmode = i;
      i++;
    }
  gtk_option_menu_set_menu (GTK_OPTION_MENU (es_tmenu), es_tmenu_menu);
  gtk_option_menu_set_history (GTK_OPTION_MENU (es_tmenu), prevmode);

  es_cs = gtk_check_button_new_with_label (_("Case sensitive"));
  gtk_widget_show (es_cs);
  gtk_table_attach (GTK_TABLE (table), es_cs, 2, 3, 0, 1,
		    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);

  label = gtk_label_new (_("File name match:"));
  gtk_widget_show (label);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);

  frame = gtk_frame_new (_("Location"));
  gtk_widget_show (frame);
  gtk_box_pack_start (GTK_BOX (vbox1), frame, FALSE, FALSE, 2);

  table = gtk_table_new (2, 4, FALSE);
  gtk_widget_show (table);
  gtk_container_add (GTK_CONTAINER (frame), table);
  gtk_table_set_row_spacings (GTK_TABLE (table), 2);
  gtk_table_set_col_spacings (GTK_TABLE (table), 6);

  label = gtk_label_new (_("Path:"));
  gtk_widget_show (label);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label), 1, 0.5);

  label = gtk_label_new (_("Disk:"));
  gtk_widget_show (label);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label), 1, 0.5);

  disk_combo = gtk_combo_new ();
  gtk_widget_show (disk_combo);
  disk_list = make_disk_list (racine);
  disk_list = g_list_prepend (disk_list, STRING_ALL);
  gtk_combo_set_popdown_strings (GTK_COMBO (disk_combo), disk_list);
  g_list_free (disk_list);
  gtk_table_attach (GTK_TABLE (table), disk_combo, 1, 2, 0, 1,
		    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);

  disk_combo_entry = GTK_COMBO (disk_combo)->entry;
  gtk_editable_set_editable (GTK_EDITABLE (disk_combo_entry), FALSE);
  gtk_widget_show (disk_combo_entry);

  s_path_gnome_entry = gnome_entry_new ("gtktalog_s_path_gnome_entry");
  gtk_widget_show (s_path_gnome_entry);
  gtk_table_attach (GTK_TABLE (table), s_path_gnome_entry, 1, 2, 1, 2,
		    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);

  s_path_entry = gnome_entry_gtk_entry (GNOME_ENTRY (s_path_gnome_entry));

  s_path_tmenu = gtk_option_menu_new ();
  gtk_widget_show (s_path_tmenu);
  s_path_tmenu_menu = gtk_menu_new ();
  gtk_widget_show (s_path_tmenu_menu);
  i = 0;
  while (tmenu_items[i] >= 0)
    {
      s_path_tmenu_menuitem =
	gtk_menu_item_new_with_label (_(STRING_SM[tmenu_items[i]]));
      gtk_object_set_user_data (GTK_OBJECT (s_path_tmenu_menuitem),
				GINT_TO_POINTER (tmenu_items[i]));
      gtk_widget_show (s_path_tmenu_menuitem);
      gtk_menu_append (GTK_MENU (s_path_tmenu_menu), s_path_tmenu_menuitem);
      i++;
    }
  gtk_signal_connect (GTK_OBJECT (s_path_tmenu_menu), "selection-done",
		      G_CALLBACK (cb_entry_NA1), s_path_tmenu);

  gtk_option_menu_set_menu (GTK_OPTION_MENU (s_path_tmenu),
			    s_path_tmenu_menu);
  gtk_option_menu_set_history (GTK_OPTION_MENU (s_path_tmenu), 0);
  gtk_table_attach (GTK_TABLE (table), s_path_tmenu, 2, 3, 1, 2,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);

  gtk_editable_set_editable (GTK_EDITABLE (s_path_entry), TRUE);
  gtk_widget_set_sensitive (GTK_WIDGET (s_path_gnome_entry), FALSE);

  s_path_cs = gtk_check_button_new_with_label (_("Case sensitive"));
  gtk_widget_show (s_path_cs);
  gtk_table_attach (GTK_TABLE (table), s_path_cs, 3, 4, 1, 2,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);
  gtk_widget_set_sensitive (GTK_WIDGET (s_path_cs), FALSE);

  label = gtk_label_new (_("Name and location"));
  gtk_widget_show (label);
  gtk_notebook_set_tab_label (GTK_NOTEBOOK (ext_search_notebook),
			      gtk_notebook_get_nth_page (GTK_NOTEBOOK
							 (ext_search_notebook),
							 0), label);

/**** Date and size ***************/
  ds_table = gtk_table_new (2, 2, FALSE);
  gtk_widget_show (ds_table);
  gtk_container_add (GTK_CONTAINER (ext_search_notebook), ds_table);
  gtk_container_set_border_width (GTK_CONTAINER (ds_table), 4);
  gtk_table_set_row_spacings (GTK_TABLE (ds_table), 2);
  gtk_table_set_col_spacings (GTK_TABLE (ds_table), 6);

  frame = gtk_frame_new (_("Options"));
  gtk_widget_show (frame);
  gtk_table_attach (GTK_TABLE (ds_table), frame, 0, 2, 1, 2,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);

  table = gtk_table_new (1, 2, FALSE);
  gtk_widget_show (table);
  gtk_container_add (GTK_CONTAINER (frame), table);
  gtk_table_set_row_spacings (GTK_TABLE (table), 6);

  display_dirs = gtk_check_button_new_with_label (_("Display directories"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (display_dirs), TRUE);
  gtk_widget_show (display_dirs);
  gtk_table_attach (GTK_TABLE (table), display_dirs, 1, 2, 0, 1,
		    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);

  checkvfs = gtk_check_button_new_with_label (_("Search in VFS too"));
  gtk_widget_show (checkvfs);
  gtk_table_attach (GTK_TABLE (table), checkvfs, 0, 1, 0, 1,
		    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);

  frame = gtk_frame_new (_("Date"));
  gtk_widget_show (frame);
  gtk_table_attach (GTK_TABLE (ds_table), frame, 1, 2, 0, 1,
		    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);

  table = gtk_table_new (2, 2, FALSE);
  gtk_widget_show (table);
  gtk_container_add (GTK_CONTAINER (frame), table);
  gtk_container_set_border_width (GTK_CONTAINER (table), 3);
  gtk_table_set_row_spacings (GTK_TABLE (table), 4);
  gtk_table_set_col_spacings (GTK_TABLE (table), 6);

  dateedit2 = gnome_date_edit_new ((time_t) 0, FALSE, TRUE);
  gtk_widget_show (dateedit2);
  gtk_table_attach (GTK_TABLE (table), dateedit2, 1, 2, 1, 2,
		    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);

  checkdate2 = gtk_check_button_new_with_label (_("Older than:"));
  gtk_widget_show (checkdate2);
  gtk_table_attach (GTK_TABLE (table), checkdate2, 0, 1, 1, 2,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);
  gtk_widget_set_sensitive (GTK_WIDGET (dateedit2),
			    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
							  (checkdate2)));

  checkdate1 = gtk_check_button_new_with_label (_("More recent than:"));
  gtk_widget_show (checkdate1);
  gtk_table_attach (GTK_TABLE (table), checkdate1, 0, 1, 0, 1,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);

  dateedit1 = gnome_date_edit_new ((time_t) 0, FALSE, TRUE);
  gtk_widget_show (dateedit1);
  gtk_table_attach (GTK_TABLE (table), dateedit1, 1, 2, 0, 1,
		    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);
  gtk_widget_set_sensitive (GTK_WIDGET (dateedit1),
			    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
							  (checkdate1)));

  frame = gtk_frame_new (_("Size"));
  gtk_widget_show (frame);
  gtk_table_attach (GTK_TABLE (ds_table), frame, 0, 1, 0, 1,
		    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);

  table = gtk_table_new (2, 3, FALSE);
  gtk_widget_show (table);
  gtk_container_add (GTK_CONTAINER (frame), table);
  gtk_container_set_border_width (GTK_CONTAINER (table), 3);
  gtk_table_set_col_spacings (GTK_TABLE (table), 6);

  checksize2 = gtk_check_button_new_with_label (_("Smaller than:"));
  gtk_widget_show (checksize2);
  gtk_table_attach (GTK_TABLE (table), checksize2, 0, 1, 1, 2,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);

  checksize1 = gtk_check_button_new_with_label (_("Bigger than:"));
  gtk_widget_show (checksize1);
  gtk_table_attach (GTK_TABLE (table), checksize1, 0, 1, 0, 1,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);

  adj = gtk_adjustment_new (1, 0, 1023, 1, 10, 10);
  size1 = gtk_spin_button_new (GTK_ADJUSTMENT (adj), 1, 0);
  gtk_widget_show (size1);
  gtk_table_attach (GTK_TABLE (table), size1, 1, 2, 0, 1,
		    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (size1), TRUE);
  gtk_widget_set_sensitive (GTK_WIDGET (size1),
			    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
							  (checksize1)));

  adj = gtk_adjustment_new (1, 0, 1023, 1, 10, 10);
  size2 = gtk_spin_button_new (GTK_ADJUSTMENT (adj), 1, 0);
  gtk_widget_show (size2);
  gtk_table_attach (GTK_TABLE (table), size2, 1, 2, 1, 2,
		    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (size2), TRUE);
  gtk_widget_set_sensitive (GTK_WIDGET (size2),
			    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
							  (checksize2)));

  size2_optionmenu = gtk_option_menu_new ();
  gtk_widget_show (size2_optionmenu);
  gtk_table_attach (GTK_TABLE (table), size2_optionmenu, 2, 3, 1, 2,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);
  size2_optionmenu_menu = gtk_menu_new ();
  menuitem = gtk_menu_item_new_with_label (_("Bytes"));
  gtk_object_set_user_data (GTK_OBJECT (menuitem), GINT_TO_POINTER (1));
  gtk_widget_show (menuitem);
  gtk_menu_append (GTK_MENU (size2_optionmenu_menu), menuitem);
  menuitem = gtk_menu_item_new_with_label (_("Kilobytes"));
  gtk_object_set_user_data (GTK_OBJECT (menuitem), GINT_TO_POINTER (1024));
  gtk_widget_show (menuitem);
  gtk_menu_append (GTK_MENU (size2_optionmenu_menu), menuitem);
  menuitem = gtk_menu_item_new_with_label (_("Megabytes"));
  gtk_object_set_user_data (GTK_OBJECT (menuitem),
			    GINT_TO_POINTER (1024 * 1024));
  gtk_widget_show (menuitem);
  gtk_menu_append (GTK_MENU (size2_optionmenu_menu), menuitem);
  gtk_option_menu_set_menu (GTK_OPTION_MENU (size2_optionmenu),
			    size2_optionmenu_menu);
  gtk_widget_set_sensitive (GTK_WIDGET (size2_optionmenu),
			    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
							  (checksize2)));

  size1_optionmenu = gtk_option_menu_new ();
  gtk_widget_show (size1_optionmenu);
  gtk_table_attach (GTK_TABLE (table), size1_optionmenu, 2, 3, 0, 1,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);
  size1_optionmenu_menu = gtk_menu_new ();
  menuitem = gtk_menu_item_new_with_label (_("Bytes"));
  gtk_object_set_user_data (GTK_OBJECT (menuitem), GINT_TO_POINTER (1));
  gtk_widget_show (menuitem);
  gtk_menu_append (GTK_MENU (size1_optionmenu_menu), menuitem);
  menuitem = gtk_menu_item_new_with_label (_("Kilobytes"));
  gtk_object_set_user_data (GTK_OBJECT (menuitem), GINT_TO_POINTER (1024));
  gtk_widget_show (menuitem);
  gtk_menu_append (GTK_MENU (size1_optionmenu_menu), menuitem);
  menuitem = gtk_menu_item_new_with_label (_("Megabytes"));
  gtk_object_set_user_data (GTK_OBJECT (menuitem),
			    GINT_TO_POINTER (1024 * 1024));
  gtk_widget_show (menuitem);
  gtk_menu_append (GTK_MENU (size1_optionmenu_menu), menuitem);
  gtk_option_menu_set_menu (GTK_OPTION_MENU (size1_optionmenu),
			    size1_optionmenu_menu);
  gtk_widget_set_sensitive (GTK_WIDGET (size1_optionmenu),
			    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
							  (checksize1)));

  gtk_signal_connect (GTK_OBJECT (checksize1), "clicked",
		      G_CALLBACK (cb_set_sensitive), size1);
  gtk_signal_connect (GTK_OBJECT (checksize1), "clicked",
		      G_CALLBACK (cb_set_sensitive), size1_optionmenu);
  gtk_signal_connect (GTK_OBJECT (checksize2), "clicked",
		      G_CALLBACK (cb_set_sensitive), size2);
  gtk_signal_connect (GTK_OBJECT (checksize2), "clicked",
		      G_CALLBACK (cb_set_sensitive), size2_optionmenu);
  gtk_signal_connect (GTK_OBJECT (checkdate1), "clicked",
		      G_CALLBACK (cb_set_sensitive), dateedit1);
  gtk_signal_connect (GTK_OBJECT (checkdate2), "clicked",
		      G_CALLBACK (cb_set_sensitive), dateedit2);

  label = gtk_label_new (_("Date and size"));
  gtk_widget_show (label);
  gtk_notebook_set_tab_label (GTK_NOTEBOOK (ext_search_notebook),
			      gtk_notebook_get_nth_page (GTK_NOTEBOOK
							 (ext_search_notebook),
							 1), label);

/**** Desc & Category ***************/
  table = gtk_table_new (3, 4, FALSE);
  gtk_widget_show (table);
  gtk_container_add (GTK_CONTAINER (ext_search_notebook), table);
  gtk_container_set_border_width (GTK_CONTAINER (table), 4);
  gtk_table_set_row_spacings (GTK_TABLE (table), 2);
  gtk_table_set_col_spacings (GTK_TABLE (table), 6);

  label = gtk_label_new (_("Category:"));
  gtk_widget_show (label);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label), 1, 0.5);

  label = gtk_label_new (_("Description:"));
  gtk_widget_show (label);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label), 1, 0.5);

  label = gtk_label_new (_("Information:"));
  gtk_widget_show (label);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 2, 3,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label), 1, 0.5);

  cat_combo = gtk_combo_new ();
  gtk_widget_show (cat_combo);
  cat_list = make_categories_list (racine);
  cat_list = g_list_prepend (cat_list, STRING_ALL);
  gtk_combo_set_popdown_strings (GTK_COMBO (cat_combo), cat_list);
  g_list_free (cat_list);
  gtk_table_attach (GTK_TABLE (table), cat_combo, 1, 2, 0, 1,
		    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);

  cat_combo_entry = GTK_COMBO (cat_combo)->entry;
  gtk_editable_set_editable (GTK_EDITABLE (cat_combo_entry), FALSE);
  gtk_widget_show (cat_combo_entry);

  desc_gnome_entry = gnome_entry_new ("gtktalog_desc_gnome_entry");
  gtk_widget_show (desc_gnome_entry);
  gtk_table_attach (GTK_TABLE (table), desc_gnome_entry, 1, 2, 1, 2,
		    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);

  desc_entry = gnome_entry_gtk_entry (GNOME_ENTRY (desc_gnome_entry));
  gtk_widget_show (desc_entry);

  information_gnome_entry =
    gnome_entry_new ("gtktalog_information_gnome_entry");
  gtk_widget_show (information_gnome_entry);
  gtk_table_attach (GTK_TABLE (table), information_gnome_entry, 1, 2, 2, 3,
		    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);

  information_entry =
    gnome_entry_gtk_entry (GNOME_ENTRY (information_gnome_entry));
  gtk_widget_show (information_entry);

  desc_tmenu = gtk_option_menu_new ();
  gtk_widget_show (desc_tmenu);
  desc_tmenu_menu = gtk_menu_new ();
  gtk_widget_show (desc_tmenu_menu);
  i = 0;
  while (tmenu_items[i] >= 0)
    {
      desc_tmenu_menuitem =
	gtk_menu_item_new_with_label (_(STRING_SM[tmenu_items[i]]));
      gtk_object_set_user_data (GTK_OBJECT (desc_tmenu_menuitem),
				GINT_TO_POINTER (tmenu_items[i]));
      gtk_widget_show (desc_tmenu_menuitem);
      gtk_menu_append (GTK_MENU (desc_tmenu_menu), desc_tmenu_menuitem);
      i++;
    }
  gtk_signal_connect (GTK_OBJECT (desc_tmenu_menu), "selection-done",
		      G_CALLBACK (cb_entry_NA1), desc_tmenu);

  gtk_option_menu_set_menu (GTK_OPTION_MENU (desc_tmenu), desc_tmenu_menu);
  gtk_option_menu_set_history (GTK_OPTION_MENU (desc_tmenu), 0);
  gtk_table_attach (GTK_TABLE (table), desc_tmenu, 2, 3, 1, 2,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);

  gtk_editable_set_editable (GTK_EDITABLE (desc_entry), TRUE);
  gtk_widget_set_sensitive (GTK_WIDGET (desc_gnome_entry), FALSE);

  desc_cs = gtk_check_button_new_with_label (_("Case sensitive"));
  gtk_widget_show (desc_cs);
  gtk_table_attach (GTK_TABLE (table), desc_cs, 3, 4, 1, 2,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);

  info_tmenu = gtk_option_menu_new ();
  gtk_widget_show (info_tmenu);
  info_tmenu_menu = gtk_menu_new ();
  gtk_widget_show (info_tmenu_menu);
  i = 0;
  while (tmenu_items[i] >= 0)
    {
      info_tmenu_menuitem =
	gtk_menu_item_new_with_label (_(STRING_SM[tmenu_items[i]]));
      gtk_object_set_user_data (GTK_OBJECT (info_tmenu_menuitem),
				GINT_TO_POINTER (tmenu_items[i]));
      gtk_widget_show (info_tmenu_menuitem);
      gtk_menu_append (GTK_MENU (info_tmenu_menu), info_tmenu_menuitem);
      i++;
    }
  gtk_signal_connect (GTK_OBJECT (info_tmenu_menu), "selection-done",
		      G_CALLBACK (cb_entry_NA1), info_tmenu);

  gtk_option_menu_set_menu (GTK_OPTION_MENU (info_tmenu), info_tmenu_menu);
  gtk_option_menu_set_history (GTK_OPTION_MENU (info_tmenu), 0);
  gtk_table_attach (GTK_TABLE (table), info_tmenu, 2, 3, 2, 3,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);

  gtk_editable_set_editable (GTK_EDITABLE (information_entry), TRUE);
  gtk_widget_set_sensitive (GTK_WIDGET (information_gnome_entry), FALSE);

  info_cs = gtk_check_button_new_with_label (_("Case sensitive"));
  gtk_widget_show (info_cs);
  gtk_table_attach (GTK_TABLE (table), info_cs, 3, 4, 2, 3,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);
  gtk_widget_set_sensitive (GTK_WIDGET (desc_cs), FALSE);
  gtk_widget_set_sensitive (GTK_WIDGET (info_cs), FALSE);

  label = gtk_label_new (_("Description & Category"));
  gtk_widget_show (label);
  gtk_notebook_set_tab_label (GTK_NOTEBOOK (ext_search_notebook),
			      gtk_notebook_get_nth_page (GTK_NOTEBOOK
							 (ext_search_notebook),
							 2), label);

/**** Replace ***************/
  table = gtk_table_new (3, 3, FALSE);
  gtk_widget_show (table);
  gtk_container_add (GTK_CONTAINER (ext_search_notebook), table);

  label = gtk_label_new (_("Category"));
  gtk_widget_show (label);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
		    (GtkAttachOptions) (0), (GtkAttachOptions) (0), 0, 0);

  label = gtk_label_new (_("Description"));
  gtk_widget_show (label);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2,
		    (GtkAttachOptions) (0), (GtkAttachOptions) (0), 0, 0);

  cat_replace_combo = gtk_combo_new ();
  gtk_widget_show (cat_replace_combo);
  cat_list = make_categories_list (racine);
  cat_list = g_list_prepend (cat_list, _(STRING_NONE));
  gtk_combo_set_popdown_strings (GTK_COMBO (cat_replace_combo), cat_list);
  g_list_free (cat_list);
  gtk_table_attach (GTK_TABLE (table), cat_replace_combo, 1, 2, 0, 1,
		    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		    (GtkAttachOptions) (GTK_EXPAND), 0, 0);

  cat_replace_combo_entry = GTK_COMBO (cat_replace_combo)->entry;
  gtk_editable_set_editable (GTK_EDITABLE (cat_replace_combo_entry), FALSE);
  gtk_widget_show (cat_replace_combo_entry);

  cat_replace_button = gtk_check_button_new_with_label (_("Replace"));
  gtk_widget_show (cat_replace_button);
  gtk_table_attach (GTK_TABLE (table), cat_replace_button, 2, 3, 0, 1,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);

  desc_replace_entry = gtk_entry_new ();
  gtk_widget_show (desc_replace_entry);
  gtk_table_attach (GTK_TABLE (table), desc_replace_entry, 1, 2, 1, 2,
		    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		    (GtkAttachOptions) (GTK_EXPAND), 0, 0);

  desc_replace_button = gtk_check_button_new_with_label (_("Replace"));
  gtk_widget_show (desc_replace_button);
  gtk_table_attach (GTK_TABLE (table), desc_replace_button, 2, 3, 1, 2,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);


  hbox_copy = gtk_hbox_new (FALSE, 2);
  gtk_widget_show (hbox_copy);
  gtk_table_attach (GTK_TABLE (table), hbox_copy, 0, 3, 2, 3,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);

  info2desc_button =
    gtk_check_button_new_with_label (_
				     ("Copy the information into the description"));
  gtk_box_pack_start (GTK_BOX (hbox_copy), info2desc_button, TRUE, TRUE, 0);
  gtk_widget_show (info2desc_button);

  info2cat_button =
    gtk_check_button_new_with_label (_
				     ("Copy the information into the category"));
  gtk_box_pack_start (GTK_BOX (hbox_copy), info2cat_button, TRUE, TRUE, 0);
  gtk_widget_show (info2cat_button);




  label = gtk_label_new (_("Replace"));
  gtk_widget_show (label);
  gtk_notebook_set_tab_label (GTK_NOTEBOOK (ext_search_notebook),
			      gtk_notebook_get_nth_page (GTK_NOTEBOOK
							 (ext_search_notebook),
							 3), label);

/* Search results **********************************************************/

  frame = gtk_frame_new (_("Search results"));
  gtk_widget_show (frame);
  gtk_box_pack_start (GTK_BOX (main_box), frame, TRUE, TRUE, 0);
  gtk_container_set_border_width (GTK_CONTAINER (frame), 4);

  scrolled_window = gtk_scrolled_window_new (NULL, NULL);
  gtk_widget_show (scrolled_window);
  gtk_container_add (GTK_CONTAINER (frame), scrolled_window);

  search_list = gtk_clist_new_with_titles (8, titles);
  gtk_signal_connect (GTK_OBJECT (search_list), "select_row",
		      G_CALLBACK (selection_made), racine);
  gtk_signal_connect (GTK_OBJECT (search_list), "unselect_row",
		      G_CALLBACK (unselection_made), racine);
  gtk_widget_show (search_list);
  gtk_container_add (GTK_CONTAINER (scrolled_window), search_list);

  gtk_clist_column_titles_show (GTK_CLIST (search_list));
  gtk_widget_set_usize (GTK_WIDGET (search_list), 600, 200);

  gtk_signal_connect (GTK_OBJECT (search_list), "button_press_event",
		      G_CALLBACK (clist_pressed), racine);
  gtk_signal_connect (GTK_OBJECT (search_list), "click_column",
		      G_CALLBACK (sort_list), racine);

  hbuttonbox = gtk_hbutton_box_new ();
  gtk_widget_show (hbuttonbox);
  gtk_box_pack_start (GTK_BOX (main_box), hbuttonbox, FALSE, FALSE, 0);
  gtk_button_box_set_layout (GTK_BUTTON_BOX (hbuttonbox), GTK_BUTTONBOX_END);
  gtk_button_box_set_spacing (GTK_BUTTON_BOX (hbuttonbox), 15);

#ifdef GNOME1
  stop_button =
    gnome_pixmap_button (gnome_stock_new_with_icon (GNOME_STOCK_PIXMAP_STOP),
			 _("Stop Search"));
#else
  stop_button = gtk_button_new_with_mnemonic (_("_Stop Search"));
#endif
  gtk_object_set_user_data (GTK_OBJECT (stop_button), NULL);
  gtk_container_add (GTK_CONTAINER (hbuttonbox), stop_button);
  gtk_signal_connect (GTK_OBJECT (stop_button), "clicked",
		      G_CALLBACK (stop_clicked), racine);

#ifdef GNOME1
  start_button =
    gnome_pixmap_button (gnome_stock_new_with_icon
			 (GNOME_STOCK_PIXMAP_SEARCH), _("Start Search"));
#else
  start_button = gtk_button_new_with_mnemonic (_("_Start Search"));
#endif
  gtk_widget_show (start_button);
  gtk_object_set_user_data (GTK_OBJECT (start_button), NULL);
  gtk_container_add (GTK_CONTAINER (hbuttonbox), start_button);
  gtk_signal_connect (GTK_OBJECT (start_button), "clicked",
		      G_CALLBACK (start_search), racine);

#ifdef GNOME1
  cancel_button = gnome_stock_button (GNOME_STOCK_BUTTON_CLOSE);
#else
  cancel_button = gtk_button_new_from_stock (GNOME_STOCK_BUTTON_CLOSE);
#endif
  gtk_widget_show (cancel_button);
  gtk_signal_connect (GTK_OBJECT (cancel_button), "clicked",
		      G_CALLBACK (cancel_clicked), NULL);
  gtk_container_add (GTK_CONTAINER (hbuttonbox), cancel_button);
  GTK_WIDGET_SET_FLAGS (cancel_button, GTK_CAN_DEFAULT);

  gtk_widget_add_accelerator (cancel_button, "clicked", accel_group,
			      GDK_Escape, 0, GTK_ACCEL_VISIBLE);

  g_list_free (tcombo_items);
  gtk_signal_connect (GTK_OBJECT (less_button), "clicked",
		      G_CALLBACK (cb_set_visible), extended_search_frame);
  cb_set_visible (less_button, extended_search_frame);
  switch_startstop_search (TRUE);

  gtk_widget_grab_focus (simple_search_entry);
  gtk_widget_show (search_dialog_window);
  gtk_window_add_accel_group (GTK_WINDOW (search_dialog_window), accel_group);
}
