/***************************************************************************
                          pg_query.c  -  description
                             -------------------
    begin                : Mon May 22 2000
    copyright            : (C) 2000 by Thierry Florac
    email                : tflorac@free.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>

#include "libpq-fe.h"
#include "pg_connection.h"
#include "pg_query.h"


/** This 'abstract' structure is used to store internal query parameters   */
struct _pgQuery {

  pgConnection 	*_connection;  /* pgConnection reference pointer           */
  PGresult      *_handle;      /* PostgreSQL query result handle           */

  gchar         *_sql;         /* SQL text                                 */

  int		 _index;       /* Index of cursor in current recordset     */
  gboolean	 _BOF;         /* Flag : is cursor at the beginning ?      */
  gboolean	 _EOF;         /* Flag : is cursor at the end ?            */

};


/** Allocate a new pgQuery structure using a given pgConnection
    - @connection: reference of the connection to use
    Result: NULL if connection is not defined or memory can't be allocated
            A valid pointer otherwise                                      */
pgQuery* 
pg_query_new (pgConnection *connection)
{
  pgQuery *query;

  if (!connection)
    return NULL;
  query = g_new0 (pgQuery, 1);
  query->_connection = connection;
  query->_handle = NULL;
  query->_sql = NULL;
  query->_index = -1;
  query->_BOF = TRUE;
  query->_EOF = TRUE;
  return query;
}


/** Allocate a new pgQuery structure, using a given pgConnection
    - @connection: reference of the connection to use
    - @sql: SQL code of the query ; may be 'SELECT' or not
    Result: NULL if connection is not defined or memory can't be allocated
            A valid pointer otherwise                                      */
pgQuery*
pg_query_new_with_sql (pgConnection *connection, 
                       gchar *sql)
{
  pgQuery *query;

  if (!connection)
    return NULL;
  query = pg_query_new (connection);
  pg_query_set_sql (query, sql);
  return query;
}


/** Update SQL code of a given pgQuery
    - @query: reference of the pgQuery to update
    - @sql: new SQL code of the query ; may be 'SELECT' or not
    Result: TRUE is SQL code was updated successfully, 
            FALSE otherwise                                                */
gboolean 
pg_query_set_sql (pgQuery *query, 
                  gchar *sql)
{
  if (!query)
    return FALSE;
  if (pg_query_is_active (query))
    pg_query_close (query);
  if (query->_sql)
    g_free (query->_sql);
  query->_sql = g_strdup (sql);
  return TRUE;
}


/** Execute command specified in a given query
    - @query: reference of the pgQuery to execute
    Result: TRUE if SQL command was executed successfully
            FALSE otherwise
            Query output messages or values are not stored
            Result is FALSE if query is already active                     */
gboolean 
pg_query_execute (pgQuery *query)
{
  if (!query || !query->_connection || !query->_sql)
    return FALSE;
  if (pg_query_is_active (query))
    return FALSE;
  query->_handle = PQexec (pg_connection_get_handle (query->_connection), query->_sql);
  if (!query->_handle) {
    PQclear (query->_handle);
    return FALSE;
  }
  if (PQresultStatus (query->_handle) != PGRES_COMMAND_OK) {
    PQclear (query->_handle);
    return FALSE;
  }
  PQclear (query->_handle);
  return TRUE;
}


/** Open specified query containing an SQL 'SELECT' statement
    - @query: reference of the pgQuery to execute
    Result: number of records fetched by the query,
            or -1 if an error was encountered                              */
gint 
pg_query_open (pgQuery *query)
{
  gint nbTuples;

  if (!query || !query->_connection || !query->_sql)
    return -1;
  if (pg_query_is_active (query))
    return pg_query_get_recordcount (query);
  query->_handle = PQexec (pg_connection_get_handle (query->_connection), query->_sql);
  if (!query->_handle) {
    PQclear (query->_handle);
    return -1;
  }
  nbTuples = pg_query_get_recordcount (query);
  if ((PQresultStatus (query->_handle) != PGRES_COMMAND_OK) &&
      (PQresultStatus (query->_handle) != PGRES_TUPLES_OK)) {
    PQclear (query->_handle);
    return -1;
  }
  if (nbTuples > 0)
    pg_query_move_first (query);
  return nbTuples;
}


/** Close specified query containing an SQL 'SELECT' statement
    - @query: reference of the pgQuery to execute
    Result: TRUE if query was closed successfully
            FALSE if an error was encountered                              */
gboolean 
pg_query_close (pgQuery *query)
{
  if (!query || !pg_query_is_active (query))
    return FALSE;
  query->_index = -1;
  query->_BOF = TRUE;
  query->_EOF = TRUE;
  PQclear (query->_handle);
  query->_handle = NULL;
  return TRUE;
}


/** Check to see if a specified query is already active
    - @query: reference of the pgQuery to check
    Result: TRUE if query is already active,
            FALSE otherwise                                                */
gboolean 
pg_query_is_active (pgQuery *query)
{
  if (!query)
    return FALSE;
  return (PQresultStatus (query->_handle) == PGRES_TUPLES_OK);
}


/** Get the number of records fetched by an SQL query
    - @query: reference of the pgQuery to check
    Result: number of records if query is active,
            -1 if an error was encountered                                 */
gint   
pg_query_get_recordcount (pgQuery *query)
{
  if (!query || !pg_query_is_active (query))
    return -1;
  return PQntuples(query->_handle);
}


/** Get the number of fields fetched by an SQL query
    - @query: reference of the pgQuery to check
    Result: number of fields if query is active,
            -1 if an error was encountered                                 */
gint   
pg_query_get_fieldcount (pgQuery *query)
{
  if (!query || !pg_query_is_active (query))
    return -1;
  return PQnfields (query->_handle);
}


/** Get the name of the field in a query result, specified by index
    - @query: reference of the pgQuery to check
    - @index: position of the field name to retrieve
    Result: name of the specified field,
            NULL if an error was encountered                               */
gchar* 
pg_query_get_fieldname (pgQuery *query, 
                        int index)
{
  if (!query || !pg_query_is_active (query))
    return NULL;
  return PQfname (query->_handle, index);
}


/** Get the index of a field in a query result, specified by name
    - @query: reference of the pgQuery to check
    - @fieldname: name of the field to get position
    Result: position of the specified field,
            -1 if an error was encountered                                 */
gint
pg_query_get_fieldindex (pgQuery *query,
                         gchar *fieldname)
{
  if (!query || !pg_query_is_active (query))
    return -1;
  return PQfnumber (query->_handle, fieldname);
}


/** Get the value of a field in a query result, specified by index
    - @query: reference of the pgQuery to check
    - @index: position of the field to get value
    Result: value of the specified field (given as string),
            NULL if an error was encountered                               */
gchar* 
pg_query_get_fieldvalue (pgQuery *query, 
                         int index)
{
  if (!query || !pg_query_is_active (query))
    return NULL;
  return PQgetvalue (query->_handle, query->_index, index);
}


/** Get the value of a field in a query result, specified by name
    - @query: reference of the pgQuery to check
    - @fieldname: name of the field to get value
    Result: value of the specified field (given as string),
            NULL if an error was encountered                               */
gchar* 
pg_query_get_fieldvalue_byname (pgQuery *query, 
                                gchar *fieldname)
{
  if (!query || !pg_query_is_active (query))
    return NULL;
  return PQgetvalue (query->_handle, query->_index, pg_query_get_fieldindex (query, fieldname));
}


/** Move a query cursor to the specified position
    - @query: reference of the pgQuery to check
    - @index: new position where to move the cursor
    Result: TRUE if the cursor was moved successfully,
            FALSE if an error was encountered                              */
gboolean 
pg_query_jumpto (pgQuery *query, 
                 int index)
{
  if (!query || !pg_query_is_active (query))
    return FALSE;
  if ((query->_index >= 0) && (query->_index < pg_query_get_recordcount (query))) {
    query->_index = index;
    query->_BOF = FALSE;
    query->_EOF = FALSE;
  }
  return TRUE;
}


/** Move a query cursor to the first record
    - @query: reference of the pgQuery to check
    Result: TRUE if the cursor was moved successfully,
            FALSE if an error was encountered                              */
gboolean 
pg_query_move_first (pgQuery *query)
{
  if (!query || !pg_query_is_active (query))
    return FALSE;
  query->_index = 0;
  query->_BOF = TRUE;
  query->_EOF = pg_query_get_recordcount (query) <= 0;
  return TRUE;
}


/** Move a query cursor to the previous record
    - @query: reference of the pgQuery to check
    Result: TRUE if the cursor was moved successfully,
            FALSE if an error was encountered                              */
gboolean 
pg_query_move_prev (pgQuery *query)
{
  if (!query || !pg_query_is_active (query) || pg_query_at_BOF (query))
    return FALSE;
  if (query->_index == 0)
    query->_BOF = TRUE;
  else
    query->_index--;
  return TRUE;
}


/** Move a query cursor to the newt record
    - @query: reference of the pgQuery to check
    Result: TRUE if the cursor was moved successfully,
            FALSE if an error was encountered                              */
gboolean 
pg_query_move_next (pgQuery *query)
{
  if (!query || !pg_query_is_active (query) || pg_query_at_EOF (query))
    return FALSE;
  if (query->_index == pg_query_get_recordcount (query)-1)
    query->_EOF = TRUE;
  else
    query->_index++;
  return TRUE;
}


/** Move a query cursor to the last record
    - @query: reference of the pgQuery to check
    Result: TRUE if the cursor was moved successfully,
            FALSE if an error was encountered                              */
gboolean 
pg_query_move_last (pgQuery *query)
{
  if (!query || !pg_query_is_active (query))
    return FALSE;
  query->_index = pg_query_get_recordcount (query)-1;
  query->_BOF = pg_query_get_recordcount (query) <= 0;
  query->_EOF = TRUE;
  return TRUE;
}


/** Check to know if a cursor is on the beginning of the resultset
    - @query: reference of the pgQuery to check
    Result: TRUE if the cursor is on the beginning or if query is inactive */
gboolean 
pg_query_at_BOF (pgQuery *query)
{
  if (!query || !pg_query_is_active (query))
    return TRUE;
  return query->_BOF;
}


/** Check to know if a cursor is on the end of the resultset
    - @query: reference of the pgQuery to check
    Result: TRUE if the cursor is on the end or if query is inactive       */
gboolean 
pg_query_at_EOF (pgQuery *query)
{
  if (!query || !pg_query_is_active (query))
    return TRUE;
  return query->_EOF;
}


/** Display a dialog box showing the last PostgreSQL error
    - @query: reference of the pgQuery to check
    - @start: message to display before PostgreSQL internal error message
    - @end: message to display after PostgreSQL internal error message     */
void 
pg_query_show_error (pgQuery *query, 
                     gchar *start, 
                     gchar *end)
{
  gchar *message;

  if (!query)
    return;
  message = g_strdup_printf ("%s\n\n  %s\n\n%s", start, PQresultErrorMessage (query->_handle), end);
  gnome_dialog_run_and_close (GNOME_DIALOG (gnome_error_dialog (message)));
  g_free (message);
}


/** Free resources and memory allocated to the given pgQuery
    - @query: reference of the pgQuery to check	                           */
void 
pg_query_free (pgQuery *query)
{
  if (!query)
    return;
  if (pg_query_is_active (query))
    pg_query_close (query);
  if (query->_sql)
    g_free (query->_sql);
  g_free (query);
  query = NULL;
}
