//  Gnomoradio - gnomoradio/criteria-input.cc
//  Copyright (C) 2004  Matt Gerginski
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <libintl.h>
#include <libxml++/libxml++.h>
#include <string>
#include <stdlib.h>
#include <sstream>
#include <iostream>
#include "criteria-input.h"

#define _(String) gettext (String)
#define gettext_noop(String) String
#define N_(String) gettext_noop (String)

using namespace std;
using namespace Glib;
using namespace Gtk;

Gnomoradio::CriteriaInput::CriteriaInput ()
	: remove(_("Remove")),
	  any_field(_("Any Field")),
	  artist(_("Artist")),
	  title(_("Title")),
	  album(_("Album")),
	  length(_("Length")),
	  rating(_("Rating")),
	  last_played(_("Last Played")),
	  contains(_("contains")),
	  higher_than(_("higher than")),
	  equal_to(_("equal to")),
	  lower_than(_("lower than")),
	  more_than(_("more than")),
	  less_than(_("less than")),
	  longer_than(_("longer than")),
	  shorter_than(_("shorter than")),
	  minutes(_("minutes")),
	  hours(_("hours")),
	  days(_("days")),
	  weeks(_("weeks")),
	  ending_label(_("minutes")),
	  rating_adj(0, -3, 3),
	  length_adj(5, 0, 60),
	  last_played_adj(5, 0, 50)
{
	/* possible things to set criteria for */
	category_menu.append(any_field);
	category_menu.append(artist);
	category_menu.append(title);
	category_menu.append(album);
	category_menu.append(rating);
	category_menu.append(length);
	category_menu.append(last_played);

	category.set_menu(category_menu);

	category.signal_changed().connect(sigc::mem_fun(*this, &Gnomoradio::CriteriaInput::update));

	/* make the constraint menus */
	info_menu.append(contains);

	rating_menu.append(higher_than);
	rating_menu.append(equal_to);
	rating_menu.append(lower_than);

	length_menu.append(longer_than);
	length_menu.append(shorter_than);

	last_played_menu.append(less_than);
	last_played_menu.append(more_than);

	constraint.set_menu(info_menu);

	constraint.signal_changed().connect(sigc::mem_fun(*this, &Gnomoradio::CriteriaInput::emit_changed_signal));

	/* make time units menu */
	time_units_menu.append(minutes);
	time_units_menu.append(hours);
	time_units_menu.append(days);
	time_units_menu.append(weeks);

	time_units.set_menu(time_units_menu);
	time_units.signal_changed().connect(sigc::mem_fun(*this, &Gnomoradio::CriteriaInput::emit_changed_signal));

	remove.signal_clicked().connect(sigc::mem_fun(*this, &Gnomoradio::CriteriaInput::on_remove_clicked));

	spin.set_adjustment(rating_adj);
	spin.signal_value_changed().connect(sigc::mem_fun(*this, &Gnomoradio::CriteriaInput::emit_changed_signal));

	pack_start(remove, PACK_SHRINK, 8);
	pack_start(category, PACK_SHRINK, 0);
	pack_start(constraint, PACK_SHRINK, 4);
	pack_start(entry, PACK_EXPAND_WIDGET, 4);
	pack_start(spin, PACK_SHRINK, 4);
	pack_start(time_units, PACK_SHRINK, 4);
	pack_start(ending_label, PACK_SHRINK, 4);
}

Gnomoradio::CriteriaInput::~CriteriaInput ()
{

}

Glib::ustring Gnomoradio::CriteriaInput::get_criteria ()
{
	xmlpp::Document query;
	xmlpp::Element *element;

	Gtk::MenuItem *cat = category_menu.get_active();

	if (cat == &any_field)
		query.create_root_node("info")->set_child_text(entry.get_text());
	else if (cat == &title) {
		element = query.create_root_node("info");
		element->set_child_text(entry.get_text());
		element->set_attribute("key", "title");
	} else if (cat == &artist) {
		element = query.create_root_node("info");
		element->set_child_text(entry.get_text());
		element->set_attribute("key", "artist");
	} else if (cat == &album) {
		element = query.create_root_node("info");
		element->set_child_text(entry.get_text());
		element->set_attribute("key", "album");
	} else if (cat == &rating) {
		element = query.create_root_node("rating");

		std::stringstream ss;
		ss << spin.get_value_as_int();

		element->set_child_text(ss.str());
		
		Gtk::MenuItem *con = rating_menu.get_active();
		
		if (con == &higher_than)
			element->set_attribute("compare", "greater");
		else if (con == &equal_to)
			element->set_attribute("compare", "equal");
		else
			element->set_attribute("compare", "less");
	} else if (cat == &length) {
		element = query.create_root_node("length");

		std::stringstream ss;
		ss << spin.get_value_as_int() * 60;

		element->set_child_text(ss.str());
		
		Gtk::MenuItem *con = length_menu.get_active();
		
		if (con == &longer_than)
			element->set_attribute("compare", "greater");
		else
			element->set_attribute("compare", "less");
	} else if (cat == &last_played) {
		element = query.create_root_node("last");

		std::stringstream ss;
		ss << spin.get_value_as_int();

		element->set_child_text(ss.str());
		
		Gtk::MenuItem *con = last_played_menu.get_active();
		
		if (con == &more_than)
			element->set_attribute("compare", "greater");
		else
			element->set_attribute("compare", "less");

		Gtk::MenuItem *unit = time_units_menu.get_active();

		if (unit == &minutes)
			element->set_attribute("unit", "minutes");
		else if (unit == &hours)
			element->set_attribute("unit", "hours");
		else if (unit == &days)
			element->set_attribute("unit", "days");
		else if (unit == &weeks)
			element->set_attribute("unit", "weeks");
	}

	return query.write_to_string();
}

void Gnomoradio::CriteriaInput::add_criteria (xmlpp::Node *parent)
{
	Gtk::MenuItem *cat = category_menu.get_active();
	xmlpp::Element *element;

	if (cat == &any_field)
		parent->add_child("info")->set_child_text(entry.get_text());
	else if (cat == &title) {
		element = parent->add_child("info");
		element->set_child_text(entry.get_text());
		element->set_attribute("key", "title");
	} else if (cat == &artist) {
		element = parent->add_child("info");
		element->set_child_text(entry.get_text());
		element->set_attribute("key", "artist");
	} else if (cat == &album) {
		element = parent->add_child("info");
		element->set_child_text(entry.get_text());
		element->set_attribute("key", "album");
	} else if (cat == &rating) {
		element = parent->add_child("rating");

		std::stringstream ss;
		ss << spin.get_value_as_int();

		element->set_child_text(ss.str());
		
		Gtk::MenuItem *con = rating_menu.get_active();
		
		if (con == &higher_than)
			element->set_attribute("compare", "greater");
		else if (con == &equal_to)
			element->set_attribute("compare", "equal");
		else
			element->set_attribute("compare", "less");
	} else if (cat == &length) {
		element = parent->add_child("length");

		std::stringstream ss;
		ss << spin.get_value_as_int() * 60;

		element->set_child_text(ss.str());
		
		Gtk::MenuItem *con = length_menu.get_active();
		
		if (con == &longer_than)
			element->set_attribute("compare", "greater");
		else
			element->set_attribute("compare", "less");
	} else if (cat == &last_played) {
		element = parent->add_child("last");

		std::stringstream ss;
		ss << spin.get_value_as_int();

		element->set_child_text(ss.str());
		
		Gtk::MenuItem *con = last_played_menu.get_active();
		
		if (con == &more_than)
			element->set_attribute("compare", "greater");
		else
			element->set_attribute("compare", "less");

		Gtk::MenuItem *unit = time_units_menu.get_active();

		if (unit == &minutes)
			element->set_attribute("unit", "minutes");
		else if (unit == &hours)
			element->set_attribute("unit", "hours");
		else if (unit == &days)
			element->set_attribute("unit", "days");
		else if (unit == &weeks)
			element->set_attribute("unit", "weeks");
	}

	signal_changed.emit();
}

void Gnomoradio::CriteriaInput::set_criteria (xmlpp::Element *element)
{
	ustring name = element->get_name();

	if (name == "info") {
		xmlpp::Attribute *key = element->get_attribute("key");

		if (key) {
			ustring key_value = key->get_value();

			if (key_value == "artist") {
				category.set_history(1);
			} else if (key_value == "title") {
				category.set_history(2);
			} else if (key_value == "album") {
				category.set_history(3);
			}
		} else {
			category.set_history(0);
		}

		category_menu.signal_selection_done();

		// set value of the entry
		xmlpp::TextNode *txt = element->get_child_text();
		if (txt)
			entry.set_text(txt->get_content());
	} else if (name == "rating") {
		category.set_history(4);

		spin.set_value((double) atoi(element->get_child_text()->get_content().c_str()));

		constraint.set_menu(rating_menu);

		string compare = element->get_attribute("compare")->get_value();

		if (compare == "greater")
			constraint.set_history(0);
		else if (compare == "equal")
			constraint.set_history(1);
		else if (compare == "less")
			constraint.set_history(2);
	} else if (name == "length") {
		category.set_history(5);

		spin.set_value((double) (atoi(element->get_child_text()->get_content().c_str()) / 60));

		constraint.set_menu(length_menu);

		string compare = element->get_attribute("compare")->get_value();

		if (compare == "greater")
			constraint.set_history(0);
		else if (compare == "less")
			constraint.set_history(1);
	} else if (name == "last") {
		category.set_history(6);

		spin.set_value((double) atoi(element->get_child_text()->get_content().c_str()));

		constraint.set_menu(last_played_menu);

		string compare = element->get_attribute("compare")->get_value();

		if (compare == "less")
			constraint.set_history(0);
		else if (compare == "greater")
			constraint.set_history(1);

		string unit = element->get_attribute("unit")->get_value();

		if (unit == "minutes")
			time_units.set_history(0);
		else if (unit == "hours")
			time_units.set_history(1);
		else if (unit == "days")
			time_units.set_history(2);
		else if (unit == "weeks")
			time_units.set_history(3);
	}

	update();
}

void Gnomoradio::CriteriaInput::on_show ()
{
	remove.show();
	category.show_all();

	// force widget to update what it looks like
	update();
	
	// show the criteria-input
	Gtk::HBox::on_show();
}

void Gnomoradio::CriteriaInput::on_remove_clicked ()
{
	signal_remove_clicked.emit();
}

void Gnomoradio::CriteriaInput::update ()
{
	if (category.get_history() == 4) { // rating
		entry.hide();
		time_units.hide();
		ending_label.hide();
		constraint.set_menu(rating_menu);
		constraint.show_all();
		spin.set_adjustment(rating_adj);
		spin.show();
	} else if (category.get_history() == 5) { // length
		entry.hide();
		time_units.hide();
		constraint.set_menu(length_menu);
		constraint.show_all();
		spin.set_adjustment(length_adj);
		spin.show();
		ending_label.set_text(_("minutes"));
		ending_label.show();
		
	} else if (category.get_history() == 6) { // last played
		entry.hide();
		constraint.set_menu(last_played_menu);
		constraint.show_all();
		spin.set_adjustment(last_played_adj);
		spin.show();
		time_units.show_all();
		ending_label.set_text(_("ago"));
		ending_label.show();
		
	} else { // any field, artist, title, album
		spin.hide();
		time_units.hide();
		ending_label.hide();
		constraint.set_menu(info_menu);
		constraint.show_all();
		entry.show();
	}

	signal_changed.emit();
}
