#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <pwd.h>
#include <string.h>
#include <getopt.h>

#include <gdk/gdk.h>
#include <gdk/gdkx.h>
#include <X11/Xlib.h>
#include <gtk/gtk.h>

#include <gksuui.h>
#include <gksu.h>

#include "defines.h"
#include "../config.h"

#include "util.h"

/* GLOBALS */
gboolean print_pass = FALSE;
gboolean force_grab = FALSE;
gboolean grab = TRUE;
gboolean sudo_mode = FALSE;
gboolean message_changed = FALSE;
gboolean prompt = FALSE;

struct option long_opts[] = {
    /*
     * { name  has_arg  *flag  val } 
     */
    {"help", no_argument, NULL, 'h'},
    {"login", no_argument, NULL, 'l'},
    {"preserv-env", no_argument, NULL, 'k'},
    {"preserve-env", no_argument, NULL, 'k'},
    {"user", required_argument, NULL, 'u'},
    {"print-pass", no_argument, NULL, 'p'},
    {"message", required_argument, NULL, 'm'},
    {"title", required_argument, NULL, 't'},
    {"icon", required_argument, NULL, 'i'},
    {"disable-grab", optional_argument, NULL, 'g'},
    {"ssh-fwd", no_argument, NULL, 's'},
    {"debug", no_argument, NULL, 'd'},
    {"sudo-mode", optional_argument, NULL, 'S'},
    {"prompt", optional_argument, NULL, 'P'},
    {0, 0, 0, 0}
};

/* 
 * code 'stolen' from gnome-session's logout.c
 *
 * Written by Owen Taylor <otaylor@redhat.com>
 * Copyright (C) Red Hat
 */
typedef struct {
  GdkScreen    *screen;
  int           monitor;
  GdkRectangle  area;
  int           rowstride;
  GdkWindow    *root_window;
  GdkWindow    *draw_window;
  GdkPixbuf    *start_pb, *end_pb, *frame;
  guchar       *start_p, *end_p, *frame_p;
  GTimeVal      start_time;
  GdkGC        *gc;
} FadeoutData;

static GList *fadeout_windows = NULL;

/* Go for five seconds */
#define FADE_DURATION 100.0

int
gsm_screen_get_width (GdkScreen *screen,
		      int        monitor)
{
	GdkRectangle geometry;

	gdk_screen_get_monitor_geometry (screen, monitor, &geometry);

	return geometry.width;
}

int
gsm_screen_get_height (GdkScreen *screen,
		       int        monitor)
{
	GdkRectangle geometry;

	gdk_screen_get_monitor_geometry (screen, monitor, &geometry);

	return geometry.height;
}

int
gsm_screen_get_x (GdkScreen *screen,
		  int        monitor)
{
	GdkRectangle geometry;

	gdk_screen_get_monitor_geometry (screen, monitor, &geometry);

	return geometry.x;
}

int
gsm_screen_get_y (GdkScreen *screen,
		  int        monitor)
{
	GdkRectangle geometry;

	gdk_screen_get_monitor_geometry (screen, monitor, &geometry);

	return geometry.y;
}

static void
get_current_frame (FadeoutData *fadeout,
		   double    sat)
{
  guchar *sp, *ep, *fp;
  int i, j, width, offset;

  width = fadeout->area.width * 3;
  offset = 0;
  
  for (i = 0; i < fadeout->area.height; i++)
    {
      sp = fadeout->start_p + offset;
      ep = fadeout->end_p   + offset;
      fp = fadeout->frame_p + offset;

      for (j = 0; j < width; j += 3)
	{
	  guchar r = abs (*(sp++) - ep[0]);
	  guchar g = abs (*(sp++) - ep[1]);
	  guchar b = abs (*(sp++) - ep[2]);

	  *(fp++) = *(ep++) + r * sat;
	  *(fp++) = *(ep++) + g * sat;
	  *(fp++) = *(ep++) + b * sat;
	}

      offset += fadeout->rowstride;
    }
}

static void
darken_pixbuf (GdkPixbuf *pb)
{
  int width, height, rowstride;
  int i, j;
  guchar *p, *pixels;
  
  width     = gdk_pixbuf_get_width (pb) * 3;
  height    = gdk_pixbuf_get_height (pb);
  rowstride = gdk_pixbuf_get_rowstride (pb);
  pixels    = gdk_pixbuf_get_pixels (pb);
  
  for (i = 0; i < height; i++)
    {
      p = pixels + (i * rowstride);
      for (j = 0; j < width; j++)
	p [j] >>= 1;
    }
}

static gboolean
fadeout_callback (FadeoutData *fadeout)
{
  GTimeVal current_time;
  double elapsed, percent;

  g_get_current_time (&current_time);
  elapsed = ((((double)current_time.tv_sec - fadeout->start_time.tv_sec) * G_USEC_PER_SEC +
	      (current_time.tv_usec - fadeout->start_time.tv_usec))) / 1000.0;

  if (elapsed < 0)
    {
      g_warning ("System clock seemed to go backwards?");
      elapsed = G_MAXDOUBLE;
    }

  if (elapsed > FADE_DURATION)
    {
      gdk_draw_pixbuf (fadeout->draw_window,
		       fadeout->gc,
		       fadeout->end_pb,
		       0, 0,
		       0, 0,
		       fadeout->area.width,
		       fadeout->area.height,
		       GDK_RGB_DITHER_NONE,
		       0, 0);

      g_object_unref (fadeout->gc);
      g_object_unref (fadeout->start_pb);
      g_object_unref (fadeout->end_pb);
      g_object_unref (fadeout->frame);

      g_free (fadeout);
    
      return FALSE;
    }

  percent = elapsed / FADE_DURATION;

  get_current_frame (fadeout, 1.0 - percent);
  gdk_draw_pixbuf (fadeout->draw_window,
		   fadeout->gc,
		   fadeout->frame,
		   0, 0,
		   0, 0,
		   fadeout->area.width,
		   fadeout->area.height,
		   GDK_RGB_DITHER_NONE,
		   0, 0);

  gdk_flush ();
  
  return TRUE;
}
  
static void
fadeout_screen (GdkScreen *screen,
		int        monitor)
{
  GdkWindowAttr attr;
  int attr_mask;
  GdkGCValues values;
  FadeoutData *fadeout;

  fadeout = g_new (FadeoutData, 1);

  fadeout->screen = screen;
  fadeout->monitor = monitor;

  fadeout->area.x = gsm_screen_get_x (screen, monitor);
  fadeout->area.y = gsm_screen_get_y (screen, monitor);
  fadeout->area.width = gsm_screen_get_width (screen, monitor);
  fadeout->area.height = gsm_screen_get_height (screen, monitor);

  fadeout->root_window = gdk_screen_get_root_window (screen);
  attr.window_type = GDK_WINDOW_CHILD;
  attr.x = fadeout->area.x;
  attr.y = fadeout->area.y;
  attr.width = fadeout->area.width;
  attr.height = fadeout->area.height;
  attr.wclass = GDK_INPUT_OUTPUT;
  attr.visual = gdk_screen_get_system_visual (fadeout->screen);
  attr.colormap = gdk_screen_get_default_colormap (fadeout->screen);
  attr.override_redirect = TRUE;
  attr_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP | GDK_WA_NOREDIR;

  fadeout->draw_window = gdk_window_new (fadeout->root_window, &attr, attr_mask);
  fadeout_windows = g_list_prepend (fadeout_windows, fadeout->draw_window);
  
  fadeout->start_pb = gdk_pixbuf_get_from_drawable (NULL,
						    fadeout->root_window,
						    NULL,
						    fadeout->area.x,
						    fadeout->area.y,
						    0, 0,
						    fadeout->area.width,
						    fadeout->area.height);
  
  fadeout->end_pb = gdk_pixbuf_copy (fadeout->start_pb);
  darken_pixbuf (fadeout->end_pb);
  
  fadeout->frame = gdk_pixbuf_copy (fadeout->start_pb);
  fadeout->rowstride = gdk_pixbuf_get_rowstride (fadeout->start_pb);

  fadeout->start_p = gdk_pixbuf_get_pixels (fadeout->start_pb);
  fadeout->end_p   = gdk_pixbuf_get_pixels (fadeout->end_pb);
  fadeout->frame_p = gdk_pixbuf_get_pixels (fadeout->frame);
  
  values.subwindow_mode = GDK_INCLUDE_INFERIORS;

  fadeout->gc = gdk_gc_new_with_values (fadeout->root_window, &values, GDK_GC_SUBWINDOW);

  gdk_window_set_back_pixmap (fadeout->draw_window, NULL, FALSE);
  gdk_window_show (fadeout->draw_window);
  gdk_draw_pixbuf (fadeout->draw_window,
		   fadeout->gc,
		   fadeout->frame,
		   0, 0,
		   0, 0,
		   fadeout->area.width,
		   fadeout->area.height,
		   GDK_RGB_DITHER_NONE,
		   0, 0);
  
  g_get_current_time (&fadeout->start_time);
  g_idle_add ((GSourceFunc) fadeout_callback, fadeout);
}

static void
hide_fadeout_windows (void)
{
  GList *l;

  for (l = fadeout_windows; l; l = l->next)
    {
      gdk_window_hide (GDK_WINDOW (l->data));
      g_object_unref (l->data);
    }

  g_list_free (fadeout_windows);
  fadeout_windows = NULL;
}
/* End of 'stolen' code */


/**
 * help:
 * @cmdname:  name of the command which was called by the user 
 * (argv[0])
 *
 * This function is a simple 'usage'-style printing function.
 * It is called if the user calls the program with --help or -h
 */
void
help (gchar *cmdname)
{ 
  g_print (_(
"GKsu version %s\n\n"
"Usage: %s [-u <user>] [-k] [-l] <command>\n\n"
"  --debug, -d\n"
"    Print information on the screen that might be\n"
"    useful for diagnosing and/or solving problems.\n"
"  --disable-grab, -g\n"
"    Disable the \"locking\" of the keyboard, mouse,\n"
"    and focus done by the program when asking for\n"
"    password.\n"
"  --icon <icon>, -i <icon>\n"
"    Replace the default window icon with the argument.\n"
"  --message <message>, -m <message>\n"
"    Replace the standard message shown to ask for\n"
"    password for the argument passed to the option.\n"
"  --print-pass, -p\n"
"    Ask gksu to print the password to stdout, just\n"
"    like ssh-askpass. Useful to use in scripts with\n"
"    programs that accept receiving the password on\n"
"    stdin.\n"
"  --prompt, -P\n"
"    Ask the user if they want to have their keyboard\n"
"    and mouse grabbed before doing so.\n"
"  --ssh-fwd, -s\n"
"    Strip the host part of the $DISPLAY variable, so that\n"
"    GKSu will work on SSH X11 Forwarding.\n"
"  --sudo-mode, -S\n"
"    Make GKSu use sudo instead of su, as if it had been\n"
"    run as \"gksudo\".\n"
"  --title <title>, -t <title>\n"
"    Replace the default title with the argument.\n"
"  --user <user>, -u <user>\n"
"    Call <command> as the specified user.\n"
"\n" 
"  --preserve-env, -k\n"
"    Preserve the current environments, does not set $HOME\n"
"    nor $PATH, for example.\n"
"  --login, -l\n"
"    Make this a login shell. Beware this may cause\n"
"    problems with the Xauthority magic. Run xhost\n"
"    to allow the target user to open windows on your\n"
"    display!\n"
"\n"
"\n"), PACKAGE_VERSION, cmdname);
}

/* copied from gnome-ssh-askpass */
#define GRAB_TRIES	16
#define GRAB_WAIT	250 /* milliseconds */

typedef enum
  {
    FAILED_GRAB_MOUSE,
    FAILED_GRAB_KEYBOARD
  } FailedGrabWhat;

void
report_failed_grab (FailedGrabWhat what)
{
  switch (what)
    {
    case FAILED_GRAB_MOUSE:
      gk_dialog (GTK_MESSAGE_WARNING, 
	     _("Could not grab your mouse.\n"
	       "A malicious client may be eavesdropping\n"
	       "on your session."));
      break;
    case FAILED_GRAB_KEYBOARD:
      gk_dialog (GTK_MESSAGE_WARNING, 
	     _("Could not grab your keyboard.\n"
	       "A malicious client may be eavesdropping\n"
	       "on your session."));
      break;
    }
}

pid_t test_lock(const char* fname)
{
   int FD = open(fname, 0);
   if(FD < 0) {
      if(errno == ENOENT) {
	 // File does not exist
	 return 0; 
      } else {
	 perror("open");
	 return(-1);
      }
   }
   struct flock fl;
   fl.l_type = F_WRLCK;
   fl.l_whence = SEEK_SET;
   fl.l_start = 0;
   fl.l_len = 0;
   if (fcntl(FD, F_GETLK, &fl) < 0) {
      g_critical("fcntl error");
      close(FD);
      return(-1);
   }
   close(FD);
   // lock is available
   if(fl.l_type == F_UNLCK)
      return(0);
   // file is locked by another process
   return (fl.l_pid);
}

int get_lock(const char *File)
{
   int FD = open(File,O_RDWR | O_CREAT | O_TRUNC,0640);
   if (FD < 0)
   {
      // Read only .. cant have locking problems there.
      if (errno == EROFS)
      {
	 g_warning(_("Not using locking for read only lock file %s"),File);
	 return dup(0);       // Need something for the caller to close
      }
      
      // Feh.. We do this to distinguish the lock vs open case..
      errno = EPERM;
      return -1;
   }
   fcntl(FD,F_SETFD, FD_CLOEXEC);
      
   // Aquire a write lock
   struct flock fl;
   fl.l_type = F_WRLCK;
   fl.l_whence = SEEK_SET;
   fl.l_start = 0;
   fl.l_len = 0;
   if (fcntl(FD,F_SETLK,&fl) == -1)
   {
      if (errno == ENOLCK)
      {
	 g_warning(_("Not using locking for nfs mounted lock file %s"), File);
	 unlink(File);
	 close(FD);
	 return dup(0);       // Need something for the caller to close	 
      }      
      
      int Tmp = errno;
      close(FD);
      errno = Tmp;
      return -1;
   }

   return FD;
}

int
grab_keyboard_and_mouse (GtkWidget *dialog)
{
  GdkGrabStatus status;
  gint grab_tries = 0;
  gint lock = -1;
  
  gchar *fname = g_strdup_printf ("%s/.gksu.lock", getenv ("HOME"));
  pid_t pid = test_lock (fname);

  if (pid != 0)
    {
      g_warning ("Lock taken by pid: %i. Exiting.", pid);
      exit (0);
    }
  
  lock = get_lock(fname);
  if( lock < 0)
    g_warning ("Unable to create lock file.");
  g_free (fname);
  
  fadeout_screen (gdk_screen_get_default (), 0);

  gtk_widget_show_all (dialog);

  for(;;) 
    {
      status = gdk_pointer_grab ((GTK_WIDGET(dialog))->window, TRUE, 0, NULL,
				 NULL, GDK_CURRENT_TIME);
      if (status == GDK_GRAB_SUCCESS)
	break;
      usleep (GRAB_WAIT * 1000);
      if (++grab_tries > GRAB_TRIES)
	{
	  report_failed_grab (FAILED_GRAB_MOUSE);
	  break;
	}
    }
  
  for(;;) 
    {
      status = gdk_keyboard_grab ((GTK_WIDGET(dialog))->window,
				  FALSE, GDK_CURRENT_TIME);
      if (status == GDK_GRAB_SUCCESS)
	break;
      
      usleep(GRAB_WAIT * 1000);
      
      if (++grab_tries > GRAB_TRIES) 
	{
	  report_failed_grab (FAILED_GRAB_KEYBOARD);
	  break;
	}
    }
  
  return lock;
}

void
ungrab_keyboard_and_mouse (int lock)
{
  /* Ungrab */
  gdk_pointer_ungrab(GDK_CURRENT_TIME);
  gdk_keyboard_ungrab(GDK_CURRENT_TIME);
  gdk_flush();

  hide_fadeout_windows ();

  close(lock);
}

#define GKSU_CONFFILE "/etc/gksu.conf"
void
read_config_file_options (GksuContext *context)
{
  int fconf;
  GIOChannel *channel;
  GError *err = NULL;
  gchar *buffer, **tmp, *key, *value;
  struct stat info;

  if (stat (GKSU_CONFFILE, &info) == -1)
    {
      fprintf (stderr, _("WARNING: Could not stat %s: %s.\n"), GKSU_CONFFILE, strerror (errno));
      return;
    }

  if (info.st_uid != 0)
    {
      fprintf (stderr, _("FATAL: File %s is not owned by root.\n"), GKSU_CONFFILE);
      exit (4);
    }
  else if (info.st_gid != 0)
    {
      fprintf (stderr, _("FATAL: File %s is not owned by group root.\n"), GKSU_CONFFILE);
      exit (4);
    }
  else if (info.st_mode & S_IWGRP || info.st_mode & S_IWOTH || info.st_mode & S_IXUSR
	   || info.st_mode & S_IXGRP || info.st_mode & S_IXOTH)
    {
      fprintf (stderr, _("FATAL: File %s has wrong permissions, should be 0644.\n"), GKSU_CONFFILE);
      exit (4);
    }

  fconf = open (GKSU_CONFFILE, O_RDONLY);
  if (fconf == -1)
    {
      fprintf (stderr, _("FATAL: Could not open %s: %s.\n"), GKSU_CONFFILE, strerror (errno));
      exit (4);
    }

  channel = g_io_channel_unix_new (fconf);
  while (TRUE)
    {
      GIOStatus status;

      status = g_io_channel_read_line (channel, &buffer, NULL, NULL, &err);
      if (status != G_IO_STATUS_NORMAL)
	{
	  if (status != G_IO_STATUS_EOF)
	    close (fconf);
	  break;
	}
      
      if (err)
	g_error_free (err);

      /* remove comments */
      tmp = g_strsplit (buffer, "#", 2);
      g_free (buffer);

      buffer = g_strdup (tmp[0]);
      g_strfreev (tmp);

      /* check if we still have a key/value pair */
      tmp = g_strsplit (buffer, "=", 2);
      g_free (buffer);
      if (tmp[1] == NULL || tmp[0] == NULL)
	{
	  g_strfreev (tmp);
	  continue;
	}

      key = tmp[0];
      value = tmp[1];

      g_strstrip (key);
      g_strstrip (value);

      if (!strcmp ("disable-grab", key))
	{
	  if (!strcasecmp ("yes", value))
	    grab = FALSE;
	}
      else if (!strcmp ("force-grab", key))
	{
	  if (!strcasecmp ("yes", value))
	    force_grab = TRUE;
	}
      else if (!strcmp ("sudo-mode", key))
	{
	  if (!strcasecmp ("yes", value))
	    sudo_mode = TRUE;
	}
      else if (!strcmp ("prompt", key))
	{
	  if (!strcasecmp ("yes", value))
	    prompt = TRUE;
	}

      g_strfreev (tmp);
    }
  g_io_channel_shutdown (channel, TRUE, NULL);
}
	      
int
main (int argc, char **argv)
{
  GtkWidget *dialog;
  GksuContext *context;

  gchar *password = NULL;
  GError *error = NULL;

  gint newargc = 0;
  gchar **newargv = NULL;

  gchar *title = NULL, *message = NULL;
  GdkPixbuf *icon = NULL;

  int retvalue = 0;
  int c = 0;

  setlocale (LC_ALL, "");
  bindtextdomain(PACKAGE_NAME, LOCALEDIR);  
  bind_textdomain_codeset (PACKAGE_NAME, "UTF-8");
  textdomain(PACKAGE_NAME);

  { /* support gksu_sudo_run */
    gchar *myname = g_path_get_basename (argv[0]);
    if (!strcmp(myname, "gksudo"))
      sudo_mode = TRUE;
    g_free (myname);
  }

  /* 
   * bad, bad code... adds a second -- right after the first one,
   * because gtk_init will remove one of them... 
   */
  {
    /* to check whether a -- was already found when parsing arguments */
    gboolean separator_found = 0;

    for (c = 0; c < argc; c++)
      {
	if (!strcmp ("--", argv[c]) && (!separator_found))
	  {
	    newargv = g_realloc (newargv, sizeof(char*) * (newargc + 2));
	    newargv[newargc] = g_strdup (argv[c]);
	    newargv[newargc + 1] = g_strdup (argv[c]);
	    
	    newargc = newargc + 2;
	    separator_found = TRUE;
	  }
	else
	  {
	    newargv = g_realloc (newargv, sizeof(char*) * (newargc + 1));
	    newargv[newargc] = g_strdup (argv[c]);
	    
	    newargc++;
	  }
      }
  }

  gtk_init (&newargc, &newargv);

  context = gksu_context_new ();
  read_config_file_options (context);
  while ((c = getopt_long(newargc, newargv, "?hu:lpm:kt:i:gdsS::P::", long_opts, NULL))
	 != EOF)
    {
      switch (c)
	{
	case 0:
	  break;

	case 'h':
	  help (newargv[0]);
	  exit(0);
	  break;
	case '?':
	  help (newargv[0]);
	  exit(0);
	  break;
	case 'u':
	  gksu_context_set_user (context, optarg);
	  break;
	case 'l':
	  gksu_context_set_login_shell (context, TRUE);
	  break;
	case 'p':
	  print_pass = TRUE;
	  break;
	case 't':
	  title = g_strdup (optarg);
	  break;
	case 'm':
	  message = g_strdup (optarg);
	  message_changed = TRUE;
	  break;
	case 'i':
	  icon = gdk_pixbuf_new_from_file (optarg, NULL);
	  break;
	case 'k':
	  gksu_context_set_keep_env (context, TRUE);
	  break;
	case 'g':
	  grab = FALSE;

	  if (optarg != NULL)
	    {
	      if (!strcasecmp (optarg, "yes")); /* ignore, already set */
	      else if (!strcasecmp (optarg, "no"))
		grab = FALSE;
	      else
		{
		  fprintf (stderr, _("Option not accepted for --disable-grab: %s\n"),
			   optarg);
		  return 1;
		}
	    }

	  break;
	case 'd':
	  gksu_context_set_debug (context, TRUE);
	  break;
	case 's':
	  gksu_context_set_ssh_fwd (context, TRUE);
	  break;
	case 'S':
	  sudo_mode = TRUE;

	  if (optarg != NULL)
	    {
	      if (!strcasecmp (optarg, "yes")); /* ignore, already set */
	      else if (!strcasecmp (optarg, "no"))
		sudo_mode = FALSE;
	      else
		{
		  fprintf (stderr, _("Option not accepted for --sudo-mode: %s\n"),
			   optarg);
		  return 1;
		}
	    }

	  break;
	case 'P':
	  prompt = TRUE;

	  if (optarg != NULL)
	    {
	      if (!strcasecmp (optarg, "yes")); /* ignore, already set */
	      else if (!strcasecmp (optarg, "no"))
		prompt = FALSE;
	      else
		{
		  fprintf (stderr, _("Option not accepted for --prompt: %s\n"),
			   optarg);
		  return 1;
		}
	    }

	  break;
	}
    }

  if (force_grab)
    grab = TRUE;

  if (prompt)
    {
      GtkWidget *d;
      
      d = gtk_message_dialog_new_with_markup (NULL, 0, GTK_MESSAGE_QUESTION,
					      GTK_BUTTONS_YES_NO,
					      _("<b>Would you like your screen to be \"grabbed\"\n"
						"while you enter the password?</b>"
						"\n\n"
						"This means all applications will be paused to avoid\n"
						"the eavesdropping of your password by a a malicious\n"
						"application while you type it."));
      
      if (gtk_dialog_run (GTK_DIALOG(d)) == GTK_RESPONSE_YES)
	grab = TRUE;
      
      gtk_widget_destroy (d);
    }

  if (grab)
    dialog = g_object_new (GKSUUI_TYPE_DIALOG,
			   "type", GTK_WINDOW_POPUP,
			   NULL);
  else
    dialog = gksuui_dialog_new ();

  if (title)
    gtk_window_set_title (GTK_WINDOW(dialog), title);
  if (message)
    gksuui_dialog_set_message (GKSUUI_DIALOG(dialog), message);
  if (icon)
    gksuui_dialog_set_icon (GKSUUI_DIALOG(dialog), icon);

  if (print_pass)
    {
      if (!gksuui_dialog_get_message (GKSUUI_DIALOG(dialog)))
	{
	  gchar *msg = 
	    g_strdup_printf (_("<b>Please enter %s's password</b>"),
			     gksu_context_get_user (context));

	  gksuui_dialog_set_message (GKSUUI_DIALOG(dialog), msg);
	  g_free (msg);
	}

      int lock = 0;
      if (grab)
	lock = grab_keyboard_and_mouse (dialog);
      retvalue = gtk_dialog_run (GTK_DIALOG(dialog));
      gtk_widget_hide (dialog);
      if (grab)
	ungrab_keyboard_and_mouse (lock);

      /* 
	 the user may have pressed cancel or
	 closed the window 
      */
      if (retvalue != GTK_RESPONSE_OK)
	  return 2;

      {
	gchar *tmp;
	tmp = gksuui_dialog_get_password (GKSUUI_DIALOG(dialog));
	password = g_locale_from_utf8 (tmp, strlen (tmp), NULL, NULL, NULL);
	g_free (tmp);
      }

      if (password)
	printf ("%s\n", password);
      return 0;
    }

  /* now we can begin to care about a command */
  if (newargc <= optind)
    {
      gk_dialog (GTK_MESSAGE_ERROR, _("Missing command to run."));
      return 1;
    }

  {
    gchar *command = g_strdup (newargv[optind]);
    gchar *tmp = NULL;
    gint i = 0;

    for (i = optind + 1; i < newargc; i++)
      {
  	tmp = g_strconcat (command, " '", newargv[i], "'", NULL);
	g_free (command);
	command = tmp;
      }
    gksu_context_set_command (context, command);
    g_free (command);
  }

  {
    struct passwd *pwentry;

    pwentry = getpwnam (gksu_context_get_user (context));

    if (pwentry->pw_uid == geteuid ())
      return g_spawn_command_line_sync (gksu_context_get_command (context),
					NULL, NULL, NULL, NULL);
  }


  if (sudo_mode)
    {
      /* test if we can run without a password */
      gksu_context_sudo_run (context, &error);
      if (error)
	{
	  if (!message_changed)
	    {
	      gchar *msg = 
		g_strdup_printf (_("<b>Please enter your password\n"
				   "to run %s as user %s</b>"),
				 gksu_context_get_command (context),
				 gksu_context_get_user (context));
	      
	      gksuui_dialog_set_message (GKSUUI_DIALOG(dialog), msg);
	      g_free (msg);
	    }

	  int lock = 0;
	  if (grab)
	    lock = grab_keyboard_and_mouse (dialog);
	  retvalue = gtk_dialog_run (GTK_DIALOG(dialog));
	  gtk_widget_hide (dialog);
	  if (grab)
	    ungrab_keyboard_and_mouse (lock);

	  while (gtk_events_pending ())
	    gtk_main_iteration ();

	  if (retvalue != GTK_RESPONSE_OK)
	    return 2;

	  {
	    gchar *tmp;
	    tmp = gksuui_dialog_get_password (GKSUUI_DIALOG(dialog));
	    password = g_locale_from_utf8 (tmp, strlen (tmp), NULL, NULL, NULL);
	    g_free (tmp);
	  }

	  gksu_context_set_password (context, password);
      
	  error = NULL;
	  gksu_context_sudo_run (context, &error);
	  if (error)
	    gk_dialog (GTK_MESSAGE_ERROR, 
		       _("Failed to run %s as user %s:\n %s"),
		       gksu_context_get_command (context),
		       gksu_context_get_user (context),
		       error->message);
	  return 3;
	}
    }
  else
    {
      if (!message_changed)
	{
	  gchar *msg = 
	    g_strdup_printf (_("<b>Please enter %s's password\n"
			       "to run %s.</b>"),
			     gksu_context_get_user (context),
			     gksu_context_get_command (context));
	  
	  gksuui_dialog_set_message (GKSUUI_DIALOG(dialog), msg);
	  g_free (msg);
	}

      int lock = 0;
      if (grab)
	 lock = grab_keyboard_and_mouse (dialog);
      retvalue = gtk_dialog_run (GTK_DIALOG(dialog));
      gtk_widget_hide (dialog);
      if (grab)
	ungrab_keyboard_and_mouse (lock);

      while (gtk_events_pending ())
	gtk_main_iteration ();

      if (retvalue != GTK_RESPONSE_OK)
	  return 2;

      {
	gchar *tmp;
	tmp = gksuui_dialog_get_password (GKSUUI_DIALOG(dialog));
	password = g_locale_from_utf8 (tmp, strlen (tmp), NULL, NULL, NULL);
	g_free (tmp);
      }

      gksu_context_set_password (context, password);
      gksu_context_run (context, &error);

      if (error) {
	gk_dialog (GTK_MESSAGE_ERROR, 
		   _("Failed to run %s as user %s:\n %s"),
		   gksu_context_get_command (context),
		   gksu_context_get_user (context),
		   error->message);
	return 3;
      }
    }
  
  return 0;
}
