/*
 * sysinfos.c : some information about the current system status
 *
 * Part of fbgetty 
 * Copyright (C) 1999 2000 2001, Yann Droneaud <ydroneaud@meuh.eu.org>. 
 *
 * fbgetty is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * fbgetty is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
 *
 */

#include <fbgetty/global.h>

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <string.h>

#include <ctype.h>
#include <errno.h>
#include <utmp.h>
#include <sys/param.h>
#include <netdb.h>

#include <fbgetty/errors.h>
#include <fbgetty/options.h>
#include <fbgetty/utmp.h>
#include <fbgetty/sysinfos.h>

/* global symbol (exported) */
sysinfos_t sysinfos;

/*
 * format a struct tm and print it
 */
void
printf_time(FILE *out, const char *fmt, sysinfos_t *sysinfos)
{
  int size;
  char *str;

  /* the longest string converted by strftime is %c -> date+time+zone (about 27 chars) */
  size = strlen(fmt) * 32;

  str = (char *) alloca(size + 1); /* allocate it */

  if (str != NULL)
    { 
      strftime(str, size, fmt, sysinfos->broken_time);
      fputs(str, out);
    }
}

/*
 * get the sysinfos
 */
sysinfos_t *
get_sysinfos(void)
{
  struct hostent *h_ent;
  time_t current_time;

  /* many informations */
  uname (&sysinfos.uts);

  /* Hostname */
  if (gethostname(sysinfos.hostname, SYSINFOS_HOSTNAME_LEN) == -1)
    {
      strncpy(sysinfos.hostname, "(unavailable)", SYSINFOS_HOSTNAME_LEN);
      /* if hostname is unavailable, we won't be able to get the fqdn ! */
      strncpy(sysinfos.fqdn, sysinfos.hostname, SYSINFOS_FQDN_LEN);
    }
  else
    {
      /* get the host entry
       * NOTE: this could take some time, if the system is misconfigured
       */
      h_ent = gethostbyname(sysinfos.hostname);
      if (h_ent == NULL)
	{
	  /* some problem in the configuration
	   * the hostname couldn't be resolved,
	   * so we set the fqdn equal to the hostname
	   */
	  strncpy(sysinfos.fqdn, sysinfos.hostname, SYSINFOS_FQDN_LEN);
	}
      else
	{
	  /* the fqdn is stored here */
	  strncpy(sysinfos.fqdn, h_ent->h_name, SYSINFOS_FQDN_LEN);
	}
    }

  /* Number of user */
  sysinfos.users = utmp_get_users_count();

  /* Time */
  time(&current_time);
  sysinfos.broken_time = localtime(&current_time);

#if 0
  strftime (&si->date, SYSINFOS_DATE_LEN , "%a %b %d %Y", tm);
  /* date in form YYYY-MM-DD */
  strftime (&si->date_iso, SYSINFOS_DATE_LEN , "%F", tm);
  /* date in form dd/mm/yyyy */
  strftime (&si->date_iso, SYSINFOS_DATE_LEN , "%Y", tm);

  strftime (&si->time, SYSINFOS_TIME_LEN , "%X", tm);
#endif 

  return(&sysinfos);
}
