"======================================================================
" opsplorer.vim -- with modifications by Cream for Vim
"
" Cream -- An easy-to-use configuration of the famous Vim text editor
" [ http://cream.sourceforge.net ] Copyright (C) 2002-2004  Steve Hall
" 
" License:
" This program is free software; you can redistribute it and/or modify
" it under the terms of the GNU General Public License as published by
" the Free Software Foundation; either version 2 of the License, or
" (at your option) any later version.
" [ http://www.gnu.org/licenses/gpl.html ]
" 
" This program is distributed in the hope that it will be useful, but
" WITHOUT ANY WARRANTY; without even the implied warranty of
" MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
" General Public License for more details.
" 
" You should have received a copy of the GNU General Public License
" along with this program; if not, write to the Free Software
" Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
" 02111-1307, USA.
"
" Cream {{{1
"
" CHANGES:
" 
" Substantive Changes:
" o Removed redundant GetPathName() at bottom.(!)
" o Revised all function names to begin with "<SID>" so the names
"   aren't exposed as global functions.
" o Added insertmode mappings.
" o Optioned mappings based on Cream presence.
" o Disabled file/folder operations until we can menu them.
" o More refined syntax mappings for better control of individual
"   pieces such as node color.
" o Many other minor tweaks of format and experimentation. ;)
" o Added features to coordinate with Cream's window manager.
"
" Options Added:
" o Use single open buffer (s:bufname).
" o Locate the top of the tree at a user-defined line number for
"   provision of a menu above (s:treepos).
" o Changed root path value to user-defined rather than hard-code "/"
"   (s:myroot).
" o Open folders like clicking on a node, rather than re-pathing the
"   tree (s:click_on_dir_opens_node). This is dependent on using the
"   hardcoded root since it is probably desirable to let the
"   re-pathing happen otherwise.
" o Use a menu above tree (s:menu). ** Unfinished **
" o Display file size and date on startup (s:file_details).
"
" Cosmetic Changes:
" o Changed script format to use Vim command and option long-names
"   rather than abbeviated ones.
" o Re-organized a few function groupings within the script. File
"   operations and special functions were moved to the bottom, and
"   motion and mouse-event handling just above them. This leaves core
"   functions at the top, just below initialization stuff.
" o Revised default highlighting colors.
" o Added folds, marker foldmethod modeline.
"
"
" TODO:
" o Bug killing:
"   * Handle dir/file names with spaces (Linux only?)
"   * s:single_click_to_edit = 1 ruins menu
" o Restore-at-next-session option
" o Detail view
"   * Toggling Details/Simple view wipes out tree
"   * Attributes: Hidden, Read-only, Owner/group
"
" TODO Advanced:
" o Right-click menu for file operations (same interface for
"   directories and files)
"   * Rename
"   * Delete
"   * Edit (Open/See/Execute, too?)
" o Add shortcuts to listings. Verify s/h links.
" o Graphic representation improvements
"   * Current folder
"   * Shorcut/hard/soft links (hard: ">>", soft: ">" following name)
"   * Indicate filter
"   * Tree "branch" indicators
" o Filter dialog
" o Options
"   * Sort case-insensitive
" o File/Folder operations
"   * Sort
"   * Delete
"   * Rename
"   * Search
"   * Move
"   * Cut/Copy/Paste
"   * Make shortcut
" o Virtual nodes
"   * Provide virtual roots 
"     - GNU/Linux: Home, Favorites, Desktop?, Documents?, mount points 
"     - Windows: Drives, Favorites, Desktop
" o Type keyboard letters to go down to alphabetical in directory.
"
"
" end Cream 1}}}

function! Cream_opsplorer()
" wrapper for main function call for window management purposes

	call Opsplore()

	" window management
	" reset window configuration
	call Cream_window_setup()
	"" restore cursor to original current buffer's new window
	"let mywinnr = bufwinnr("_opsplorer")
	"call MoveCursorToWindow(mywinnr)

endfunction

"----------------------------------------------------------------------
" opsplorer - treeview file explorer for vim
"
" Author:  Patrick Schiel
" Date:    2002/08/09
" Email:   schiel@dfki.de
" Version: 1.1
"
" see :help opsplorer.txt for detailed description

" Initialization {{{1

" setup command
command! -nargs=* -complete=dir Opsplore call Opsplore(<f-args>)
"+++ Cream: we have all maps elsewhere
if !exists("$CREAM")
	noremap <silent> <F11> :call <SID>ToggleShowExplorer()<CR>
endif
"+++

function! <SID>InitOptions()
	let s:single_click_to_edit = 0
	let s:file_match_pattern = '*'
	let s:show_hidden_files = 0
	let s:split_vertical = 1
	let s:split_width = 26
	let s:split_minwidth = 1
	let s:use_colors = 1
	let s:close_explorer_after_open = 0
	"+++ Cream:
	" option using only a single buffer named "_opsplorer"
	" (empty to use multiple dynamic)
	let s:bufname = "_opsplorer"

	" option to (double)click on folder to open node/expands tree
    " rather than shift to that folder as window root
	let s:click_on_dir_opens_node = 1

	" * let's not hardcode root (s:myroot)
    " * option static (root) path if double-click on directory opens
    "   node (use variable path if double-click re-paths it)
    if s:click_on_dir_opens_node == 1
		if has("win32")
			" Note: roots larger than 1 char are currently broken!
			let s:myroot = "/"
		else
			let s:myroot = "/"
		endif
	else
		let s:myroot = getcwd()
	endif

    " *** must be coordinated! ***
	" option on which line tree begins (lets us put menu above)
	let s:treepos = 5
	" use menu
	let s:menu = 1

    " show details (file size, date) on startup
	let s:file_details = 0

	"+++
endfunction

function! <SID>InitMappings()
	noremap <silent> <buffer> <LeftRelease> :call <SID>OnClick()<CR>
	noremap <silent> <buffer> <2-LeftMouse> :call <SID>OnDoubleClick(-1)<CR>
	noremap <silent> <buffer> <Space> :call <SID>OnDoubleClick(0)<CR>
	noremap <silent> <buffer> <CR> :call <SID>OnDoubleClick(1)<CR>
	noremap <silent> <buffer> <Down> :call <SID>GotoNextEntry()<CR>
	noremap <silent> <buffer> <Up> :call <SID>GotoPrevEntry()<CR>
	noremap <silent> <buffer> <S-Down> :call <SID>GotoNextNode()<CR>
	noremap <silent> <buffer> <S-Up> :call <SID>GotoPrevNode()<CR>
	noremap <silent> <buffer> <BS> :call <SID>BuildParentTree()<CR>
	"+++ Cream:
	" * Option file-management mappings
	if !exists("$CREAM")
		noremap <silent> <buffer> q :call <SID>CloseExplorer()<CR>
		noremap <silent> <buffer> n :call <SID>InsertFilename()<CR>
		noremap <silent> <buffer> p :call <SID>InsertFileContent()<CR>
		noremap <silent> <buffer> s :call <SID>FileSee()<CR>
		noremap <silent> <buffer> N :call <SID>FileRename()<CR>
		noremap <silent> <buffer> D :call <SID>FileDelete()<CR>
		noremap <silent> <buffer> C :call <SID>FileCopy()<CR>
		noremap <silent> <buffer> O :call <SID>FileMove()<CR>
	endif
	"+++
	noremap <silent> <buffer> H :call <SID>ToggleShowHidden()<CR>
	noremap <silent> <buffer> M :call <SID>SetMatchPattern()<CR>

	"+++ Cream: 
	" * Add insertmode mappings
	" * Option for Cream (our <C-o> is re-mapped to <C-o>)
	if exists("$CREAM")
		imap <silent> <buffer> <LeftRelease> <C-o>:call <SID>OnClick()<CR>
		imap <silent> <buffer> <2-LeftMouse> <C-o>:call <SID>OnDoubleClick(-1)<CR>
		imap <silent> <buffer> <Space> <C-o>:call <SID>OnDoubleClick(0)<CR>
		imap <silent> <buffer> <CR> <C-o>:call <SID>OnDoubleClick(1)<CR>
		" Windows also needs <Return>!
		if has("win32")
			imap <silent> <buffer> <Return> <C-o>:call <SID>OnDoubleClick(1)<CR>
		endif
		imap <silent> <buffer> <Down> <C-o>:call <SID>GotoNextEntry()<CR>
		imap <silent> <buffer> <Up> <C-o>:call <SID>GotoPrevEntry()<CR>
		imap <silent> <buffer> <S-Down> <C-o>:call <SID>GotoNextNode()<CR>
		imap <silent> <buffer> <S-Up> <C-o>:call <SID>GotoPrevNode()<CR>
		imap <silent> <buffer> <BS> <C-o>:call <SID>BuildParentTree()<CR>
		imap <silent> <buffer> H <C-o>:call <SID>ToggleShowHidden()<CR>
		imap <silent> <buffer> M <C-o>:call <SID>SetMatchPattern()<CR>
	else
		imap <silent> <buffer> <LeftRelease> <C-o>:call <SID>OnClick()<CR>
		imap <silent> <buffer> <2-LeftMouse> <C-o>:call <SID>OnDoubleClick(-1)<CR>
		imap <silent> <buffer> <Space> <C-o>:call <SID>OnDoubleClick(0)<CR>
		imap <silent> <buffer> <CR> <C-o>:call <SID>OnDoubleClick(1)<CR>
		" Windows also needs <Return>!
		if has("win32")
			imap <silent> <buffer> <Return> <C-o>:call <SID>OnDoubleClick(1)<CR>
		endif
		imap <silent> <buffer> <Down> <C-o>:call <SID>GotoNextEntry()<CR>
		imap <silent> <buffer> <Up> <C-o>:call <SID>GotoPrevEntry()<CR>
		imap <silent> <buffer> <S-Down> <C-o>:call <SID>GotoNextNode()<CR>
		imap <silent> <buffer> <S-Up> <C-o>:call <SID>GotoPrevNode()<CR>
		imap <silent> <buffer> <BS> <C-o>:call <SID>BuildParentTree()<CR>
		imap <silent> <buffer> H <C-o>:call <SID>ToggleShowHidden()<CR>
		imap <silent> <buffer> M <C-o>:call <SID>SetMatchPattern()<CR>
	endif
	"+++
endfunction

function! <SID>InitCommonOptions()
	setlocal nowrap
	setlocal nonumber
	"+++ Cream:
	setlocal noautoindent
	"+++
endfunction

function! <SID>InitColors()
	syntax clear
	if s:use_colors

		"+++ Cream:
		"" original
		"highlight link OpsPath Label
		"highlight link OpsNode Comment
		"highlight link OpsFile Question

		highlight OpsPath gui=bold guibg=#ffffff guifg=#804040
		highlight OpsSeps gui=bold guibg=#ffffff guifg=#999900
		highlight OpsDir  gui=none guibg=#ffffff guifg=#0000cc
		highlight OpsNode gui=bold guibg=#ffffff guifg=#cc6600
		highlight OpsFile gui=none guibg=#ffffff guifg=#006000
		highlight OpsMenu gui=none guibg=#eeeeee guifg=#0000cc

		"" original
		"syntax match OpsPath "^/.*"
		"syntax match OpsNode "^\s*[+-]"
		"syntax match OpsFile "^\s*\w\w*.*$"

		execute "syntax match OpsPath '\\%" . s:treepos . "l.*$'"
		syntax match OpsNode '^\s*\zs[+-]\{1}'
		syntax match OpsDir  '\(^\s*[+-]\{1}\)\@<=.*$'
		syntax match OpsFile '^\(\s*[+-]\{1}\)\@!\(\a\)\@!\(/\)\@!\s*.*$'
		execute "syntax match OpsMenu '\\%<" . s:treepos . "l.*$'"
		"+++

	endif
endfunction

 " Core {{{1
 
function! Opsplore(...)
" create explorer window

	"--------------------------------------------------------------------
	" setup basic options
	call <SID>InitOptions()

	"+++ Cream: 
    " if single buffer option, test if open, and switch to it if so.
    if s:bufname != ""
		let mybufno = bufnr(s:bufname)
		" if already open
		if bufloaded(mybufno)

			"----------------------------------------------------------------------
			" From:   genutils.vim by Hari Krishna Dara <hari_vim at yahoo.com>
			" Source: http://vim.sourceforge.net/scripts/script.php?script_id=197
			"
			" count windows
			let windowcnt = 1
			while winbufnr(windowcnt) != -1
				let windowcnt = windowcnt + 1
			endwhile
			let windowcnt = windowcnt - 1
			" move to that window
			if windowcnt != 1

				" Find the window number for the buffer. If non-zero,
				" search for the buffer in the unlisted buffers
				" (work-around the vim bug that bufnr() doesn't work
				" for unlisted buffers).
				let bufno = bufnr(s:bufname)
				" bufnr() will not find unlisted buffers.
				if bufno == -1
				" Iterate over all the open windows for 

					" quote to protect other escaping (reverse escape() )
					let bufName = "\"" . a:val . "\""
					let i = 1
					while winbufnr(i) != -1
						if bufname(winbufnr(i)) == bufName
							let let mywindow = i;
							break
						endif
						let i = i + 1
					endwhile
				else
					let mywindow = bufwinnr(bufno)
				endif

				execute mywindow . " wincmd w"
			endif
			"----------------------------------------------------------------------
			return
		endif
	endif
	"+++

	"--------------------------------------------------------------------
	" root path (take argument as path, if given)
	if a:0 > 0
		let path = a:1
	else
		"+++ Cream: use root instead of current dir
		" otherwise current dir
		"let path = getcwd()
		let path = s:myroot
		"+++
	endif

	" substitute leading ~
	" (doesn't work with isdirectory() otherwise!)
	let path = fnamemodify(path,":p")

	" expand, if relative path
	"+++ Cream: add win32 check
	if !has("win32")
		if path[0] != '/'
			let path = getcwd() . '/' . path
		endif
	endif
	"+++

	"--------------------------------------------------------------------
	" create new window
	"+++ Cream:
	" single buffer option
	if s:bufname != ""
		let mybufname = s:bufname
	else
		let mybufname = ""
	endif
	" split window orientation option
	if s:split_vertical
		let splitcmd = 'vnew ' . mybufname
	else
		let splitcmd = 'new ' . mybufname
	endif
	"+++
	let splitcmd = s:split_width . splitcmd
	execute splitcmd
	execute "setlocal winwidth=" . s:split_minwidth

	" remember buffer nr
	let s:window_bufnr = winbufnr(0)

	"--------------------------------------------------------------------
	" setup remaining mappings, options, colors
	call <SID>InitCommonOptions()
	call <SID>InitColors()
	call <SID>InitMappings()

	"--------------------------------------------------------------------
	" draw tree
	call s:BuildTree(path)

	let g:opsplorer_loaded = 1

endfunction

function! s:BuildTree(path)
" call this to re-draw the entire window

	let path = a:path
	" clean up
	setlocal modifiable
	" delete all, open new line
	"+++ Cream: variable position of tree
	"normal ggVGxo
	" delete all
	normal ggVGx
	" clear lines at the top
	let i = 0
	while i < s:treepos
		normal o
		let i = i + 1
	endwhile
	"+++

	"+++ Cream
	if s:menu == 1
		call s:BuildMenu()
	endif
	"+++

	" remove unneeded trailing slash
	if strlen(path) > strlen(s:myroot) && 
	\ path[strlen(path) - 1] == '/'
		let path = strpart(path, 0, strlen(path) - 1)
	endif

	" put full path on first [treepos] line
	call setline(s:treepos, path)
	"+++ Cream: do this later, since we have to re-write the path line
	"setlocal nomodifiable 
	"setlocal nomodified
	"+++

	" draw tree, pass -1 as xpos to start at column 0
	call <SID>TreeExpand(-1, s:treepos, path)

	"+++ Cream: TreeExpand may have reset this
	setlocal modifiable
	"+++

	" move to first entry
	"--- original (end of path)
	"normal ggj
	"normal 1|
	"normal g^
	"---
	"+++ Cream: place cursor on line s:treepos
	execute "normal :" . s:treepos . "\<CR>"
	"" first column
	"normal 0
	" go to first char of last node 
	normal $
	normal F/l
    "+++

	"+++ Cream: re-do it, TreeExpand messes it up
	call setline(s:treepos, path)
	"+++

	"+++ Cream: let's do it at the end
	setlocal nomodifiable 
	setlocal nomodified
	"+++

endfunction

function! <SID>TreeExpand(xpos, ypos, path)
" draw remainder of tree below top path

	let path = a:path
	setlocal modifiable

	" turn + into -
	normal r-

	" remember where to append
	let row = a:ypos

	"+++ Cream: list both directories and files first to calculate longest entry +++
	" directories
	let dirlist = ""
	" hidden directories
	if s:show_hidden_files
		let dirlist = glob(path . '/.*') . "\n"
	endif
	" normal entries
	let dirlist = dirlist . glob(path . '/*') . "\n"

	" files
	let filelist = ""
	" hidden files
	if s:show_hidden_files
		let filelist = glob(path . '/.*' . s:file_match_pattern) . "\n"
	endif
	" normal entries
	let filelist = filelist . glob(path . '/*' . s:file_match_pattern) . "\n"


	"+++ Cream:
	" calculate longest entry (g:op_longestname) and
	" calculate longest filesize (g:op_longestsize)
	if !exists("g:op_longestname")
		let g:op_longestname = 1
	endif
	if !exists("g:op_longestsize")
		let g:op_longestsize = 1
	endif
	"+++

	"======================================================================
	" Option 1
	" * Iterate through list, (get details,) calculate longest
	" * Iterate through list, (get details,) write to screen
	"======================================================================

	" calculate sizes
	let tmplist = dirlist
	while strlen(tmplist) > 0
		" get next line
		let entry = s:GetNextLine(tmplist)
		let tmplist = s:CutFirstLine(tmplist)
		" get file size
		if g:op_longestsize < strlen(<SID>GetFileSize(entry))
			let g:op_longestsize = strlen(<SID>GetFileSize(entry))
		endif
		let entry = substitute(entry, '.*/', '', '')
		if g:op_longestname < strlen(entry) + a:xpos + 1
			let g:op_longestname = strlen(entry) + a:xpos + 1
		endif
	endwhile
	let tmplist = filelist
	while strlen(tmplist) > 0
		" get next line
		let entry = s:GetNextLine(tmplist)
		let tmplist = s:CutFirstLine(tmplist)
		" get file size
		if g:op_longestsize < strlen(<SID>GetFileSize(entry))
			let g:op_longestsize = strlen(<SID>GetFileSize(entry))
		endif
		let entry = substitute(entry, '.*/', '', '')
		if g:op_longestname < strlen(entry) + a:xpos + 1
			let g:op_longestname = strlen(entry) + a:xpos + 1
		endif
	endwhile

	" directories
	while strlen(dirlist) > 0
		" get next line
		let entry = s:GetNextLine(dirlist)
		let dirlist = s:CutFirstLine(dirlist)
		" add to tree if directory
		if isdirectory(entry)
			"+++ Cream: 
			" append size/date
			if s:file_details == 1
				let mydirsize = s:GetFileSize(entry)
				let mydirdate = s:GetFileDateFormatted(entry)
				let entry = substitute(entry, '.*/', '', '')
				let myentryspacer = s:SpaceString(g:op_longestname + a:xpos + 2 - strlen(entry) + strlen(g:op_longestsize) - strlen(mydirsize))
			else
				let mydirsize = ""
				let mydirdate = ""
				let entry = substitute(entry, '.*/', '', '')
				let myentryspacer = ""
			endif
			"+++
			if entry != "." && entry != ".."
				"+++ Cream:
				" append file size/date if details on
				if s:file_details == 1
					let entry = entry . "  " . myentryspacer . mydirsize . "  " . mydirdate
				endif
				"+++
				" indent, mark as node and append
				let entry = s:SpaceString(a:xpos + 1) . "+" . entry
				call append(row, entry)
				let row = row + 1
			endif
		endif
	endwhile
	" files
	while strlen(filelist) > 0
		" get next line
		let entry = s:GetNextLine(filelist)
		let filelist = s:CutFirstLine(filelist)
		" only files
		if entry != "." && entry != ".." && entry != ""
			if !isdirectory(entry) && filereadable(entry)
				"+++ Cream: 
				" append size/date
				if s:file_details == 1
					let myfilesize = s:GetFileSize(entry)
					let myfiledate = s:GetFileDateFormatted(entry)
					let entry = substitute(entry, '.*/', '', '')
					let myentryspacer = s:SpaceString(g:op_longestname + a:xpos + 2 - strlen(entry) + strlen(g:op_longestsize) - strlen(myfilesize))
				else
					let myfilesize = ""
					let myfiledate = ""
					let entry = substitute(entry, '.*/', '', '')
					let myentryspacer = ""
				endif
				"+++
				"+++ Cream:
				" append file size/date if details on
				if s:file_details == 1
					let entry = entry . "  " . myentryspacer . myfilesize . "  " . myfiledate
				endif
				"+++
				" indent and append
				let entry = s:SpaceString(a:xpos + 2) . entry
				call append(row, entry)
				let row = row + 1
			endif
		endif
	endwhile


	""======================================================================
	"" Option 2 (Much slower!)
	"" * Iterate through list, (array details,) array list
	"" * Iterate through array, write to screen
	""======================================================================
	"
	"" create arrays
	"" directories (with details if selected)
	"let i = 0
	"let mydirs = ""
	"let dircount = MvNumberOfElements(dirlist, "\n")
	"while i < dircount
	"    let mydir = MvElementAt(dirlist, "\n", i)
	"    let mydirname = substitute(mydir, '.*/', '', '')
	"    if isdirectory(mydir) && mydirname != "." && mydirname != ".."
	"        " add (optional) details
	"        if s:file_details == 1
	"            let mydirsize = <SID>GetFileSize(mydir)
	"            let mydirdate = s:GetFileDateFormatted(mydir)
	"            " calculate longest name, size (dates are uniform)
	"            if g:op_longestname < strlen(mydirname) + a:xpos + 1
	"                let g:op_longestname = strlen(mydirname) + a:xpos + 1
	"            endif
	"            if g:op_longestsize < strlen(mydirsize)
	"                let g:op_longestsize = strlen(mydirsize)
	"            endif
	"            let myelement = mydirname . "\t" . mydirsize . "\t" . mydirdate
	"            let mydirs = MvAddElement(mydirs, "\n", myelement)
	"        else
	"            let mydirs = MvAddElement(mydirs, "\n", mydirname)
	"        endif
	"    endif
	"    let i = i + 1
	"endwhile
	"" files (with details if selected)
	"let i = 0
	"let myfiles = ""
	"let filecount = MvNumberOfElements(filelist, "\n")
	"while i < filecount
	"    let myfile = MvElementAt(filelist, "\n", i)
	"    let myfilename = substitute(myfile, '.*/', '', '')
	"    if !isdirectory(myfile) && filereadable(myfile)
	"        " add (optional) details
	"        if s:file_details == 1
	"            let myfilesize = <SID>GetFileSize(myfile)
	"            let myfiledate = s:GetFileDateFormatted(myfile)
	"            " calculate longest name, size (dates are uniform)
	"            if g:op_longestname < strlen(myfilename) + a:xpos + 2
	"                let g:op_longestname = strlen(myfilename) + a:xpos + 2
	"            endif
	"            if g:op_longestsize < strlen(myfilesize)
	"                let g:op_longestsize = strlen(myfilesize)
	"            endif
	"            let myfiles = MvAddElement(myfiles, "\n", myfilename . "\t" . myfilesize . "\t" . myfiledate)
	"        else
	"            let myfiles = MvAddElement(myfiles, "\n", myfilename)
	"        endif
	"    endif
	"    let i = i + 1
	"endwhile

	"" write to screen
	"" directories
	"let myindent = s:SpaceString(a:xpos + 1)
	"let i = 0
	"let validdirs = MvNumberOfElements(mydirs, "\n")
	"while i < validdirs
	"    let mydir = MvElementAt(mydirs, "\n", i)
	"    if s:file_details == 1
	"        " get array's second dimension
	"        let mydirname = MvElementAt(mydir, "\t", 0)
	"        let mydirsize = MvElementAt(mydir, "\t", 1)
	"        let mydirdate = MvElementAt(mydir, "\t", 2)
	"        let myspacer = s:SpaceString(g:op_longestname + a:xpos + 1 - strlen(mydirname) + strlen(g:op_longestsize) - strlen(mydirsize))
	"        let myrow = myindent . "+" . mydirname . "  " . myspacer . mydirsize . "  " . mydirdate
	"    else
	"        let myrow = myindent . "+" . mydir
	"    endif
	"    " append
	"    call append(row, myrow)
	"    let row = row + 1
	"    let i = i + 1
	"endwhile
	"" files
	"let myindent = myindent . " "
	"let i = 0
	"let validfiles = MvNumberOfElements(myfiles, "\n")
	"while i < validfiles
	"    let myfile = MvElementAt(myfiles, "\n", i)
	"    if s:file_details == 1
	"        " get array's second dimension
	"        let myfilename = MvElementAt(myfile, "\t", 0)
	"        let myfilesize = MvElementAt(myfile, "\t", 1)
	"        let myfiledate = MvElementAt(myfile, "\t", 2)
	"        let myspacer = s:SpaceString(g:op_longestname + a:xpos + 2 - strlen(myfilename) + strlen(g:op_longestsize) - strlen(myfilesize))
	"        let myrow = myindent . myfilename . "  " . myspacer . myfilesize . "  " . myfiledate
	"    else
	"        let myrow = myindent . myfile
	"    endif
	"    " append
	"    call append(row, myrow)
	"    let row = row + 1
	"    let i = i + 1
	"endwhile


	setlocal nomodifiable
	setlocal nomodified

endfunction

function! <SID>BuildParentTree()
	normal gg$
	normal F/
	call <SID>OnDoubleClick()
endfunction

function! <SID>GetPathName(xpos, ypos)
	let xpos = a:xpos
	let ypos = a:ypos
	" check for directory
	if getline(ypos)[xpos] =~ "[+-]"
		let path = '/' . strpart(getline(ypos), xpos + 1, col('$'))
		"+++ Cream: if details on, remove
		if s:file_details == 1
			" strip trailing whitespace
			let path = substitute(path, '\s*$', '', 'g')
			" remove trailing date
			let path = strpart(path, 0, strlen(path) - 20)
			" remove trailing spaces and filesize
			let path = substitute(path, '\s*\d*$', '', 'g')
		endif
		"+++
	else
		" otherwise filename
		let path = '/' . strpart(getline(ypos), xpos, col('$'))
		"+++ Cream: if details on, remove
		if s:file_details == 1
			" strip trailing whitespace
			let path = substitute(path, '\s*$', '', 'g')
			" remove trailing date
			let path = strpart(path, 0, strlen(path) - 20)
			" remove trailing spaces and filesize
			let path = substitute(path, '\s*\d*$', '', 'g')
		endif
		"+++
		let xpos = xpos - 1
	endif

	" walk up tree and append subpaths
	let row = ypos - 1
	let indent = xpos
	while indent > 0
		" look for prev ident level
		let indent = indent - 1
		while getline(row)[indent] != '-'
			let row = row - 1
			if row == 0
				return ""
			endif
		endwhile
		" subpath found, append
		let subpath = strpart(getline(row), indent + 1)
		"+++ Cream: if details on, remove
		if s:file_details == 1
			" strip trailing whitespace
			let subpath = substitute(subpath, '\s*$', '', 'g')
			" remove trailing date
			let subpath = strpart(subpath, 0, strlen(subpath) - 20)
			" remove trailing spaces and filesize
			let subpath = substitute(subpath, '\s*\d*$', '', 'g')
		endif
		"let path = '/' . strpart(getline(row), indent + 1, strlen(getline(row))) . path
		let path = '/' . subpath . path
		"+++
	endwhile 
	" finally add base path
	" not needed, if in root
	if getline(s:treepos) != s:myroot
		let path = getline(s:treepos) . path
	endif
	return path

endfunction

function! s:TreeCollapse(xpos, ypos)
	setlocal modifiable
	" turn - into +, go to next line
	normal r+j
	" delete lines til next line with same indent
	while (getline('.')[a:xpos + 1] =~ '[ +-]') && (line('$') != line('.'))
		normal dd
	endwhile
	" go up again
	normal k
	setlocal nomodifiable
	setlocal nomodified
endfunction

function! s:TreeNodeAction(xpos, ypos)
	if getline(a:ypos)[a:xpos] == '+'
		call <SID>TreeExpand(a:xpos, a:ypos, <SID>GetPathName(a:xpos, a:ypos))
	elseif getline(a:ypos)[a:xpos] == '-'
		call s:TreeCollapse(a:xpos, a:ypos)
	endif
endfunction

function! s:IsTreeNode(xpos, ypos)
	if getline(a:ypos)[a:xpos] =~ '[+-]'
		" is it a directory or file starting with +/- ?
		if isdirectory(<SID>GetPathName(a:xpos, a:ypos))
			return 1
		else
			return 0
		endif
	else
		return 0
	endif
endfunction

"+++ Cream doesn't use these
function! <SID>ToggleShowExplorer()
	if exists("g:opsplorer_loaded")
		execute s:window_bufnr . "bd"
		unlet g:opsplorer_loaded
	else
		call Opsplore()
	endif
endfunction

function! <SID>CloseExplorer()
	wincmd change
endfunction
"+++

function! <SID>ToggleShowHidden()
	let s:show_hidden_files = 1 - s:show_hidden_files
	call s:BuildTree(getline(1))
endfunction

function! <SID>SetMatchPattern()
	let s:file_match_pattern = input("Match pattern: ", s:file_match_pattern)
	call s:BuildTree(getline(s:treepos))
endfunction

function! s:GetNextLine(text)
	"let pos = match(a:text, "\n")
	"return strpart(a:text, 0, pos)
	return strpart(a:text, 0, match(a:text, "\n"))
endfunction

function! s:CutFirstLine(text)
	"let pos = match(a:text, "\n")
	"return strpart(a:text, pos + 1, strlen(a:text))
	return strpart(a:text, match(a:text, "\n") + 1, strlen(a:text))
endfunction

function! s:SpaceString(width)
	let spacer = ""
	let width = a:width
	while width > 0
		let spacer = spacer . " "
		let width = width - 1
	endwhile
	return spacer
endfunction

"+++ Cream

function! s:GetFileSize(pathfile)
" get the size of the file
	return getfsize(a:pathfile)
endfunction

function! s:GetFileDate(pathfile)
" get the date of the file as a number
	return getftime(a:pathfile)
endfunction

function! s:GetFileDateFormatted(pathfile)
" Get the modification time for a file
	let filetime = getftime(a:pathfile)
	if filetime > 0
		"return strftime("%d %b %Y %H:%M", filetime) . " " . filetime
		return strftime("%d %b %Y %H:%M", filetime)
	else
		return ""
	endif
endfunction

"+++

" Motions {{{1

function! <SID>GotoNextNode()
	" in line 1 like next entry
	if line('.') == s:treepos
		call <SID>GotoNextEntry()
	else
		normal j1|g^
		while getline('.')[col('.') - 1] !~ "[+-]" && line('.') < line('$')
			normal j1|g^
		endwhile
	endif
endfunction

function! <SID>GotoPrevNode()
	" entering base path section?
	"+++ Cream: 
	"if line('.') < 3
	if line('.') < s:treepos + 2
	"+++
		call <SID>GotoPrevEntry()
	else
		normal k1|g^
		while getline('.')[col('.') - 1] !~ "[+-]" && line('.') > s:treepos
			normal k1|g^
		endwhile
	endif
endfunction

function! <SID>GotoNextEntry()
	let xpos = col('.')
	"+++ Cream:
	" if above tree pos, move to next word
	if line('.') < s:treepos
	    normal w
	"+++
	" different movement in line 1
	elseif line('.') == s:treepos
		" if over slash, move one to right
		if getline('.')[xpos - 1] == '/'
			normal l
			" only root path there, move down
			if col('.') == 1
				normal j1|g^
			endif
		else
			" otherwise after next slash
			normal f/l
			" if already last path, move down
			if col('.') == xpos
				normal j1|g^
			endif
		endif
	else
		" next line, first nonblank
		normal j1|g^
	endif
endfunction

function! <SID>GotoPrevEntry()
	"+++ Cream:
	" if above tree pos, move to next word
	if line('.') < s:treepos
	    normal b
	"+++
	" different movement in line 1
	elseif line('.') == s:treepos
		"+++ Cream:
		" if we're over a colon (Windows C:), go back another char
		if has("win32")
			normal hhF/l
			redir @x
			ascii
			redir END
			if @x[2] == ":"
				normal h
			endif
		else
			normal hF/l
		endif
		"+++
	else
		" enter line 1 at the end
		"+++ Cream:
		"if line('.') == 2
		if line('.') == s:treepos + 1
		"+++
			normal k$F/l
		else
			" prev line, first nonblank
			normal k1|g^
		endif
	endif
endfunction

" Menu {{{1

"+++ Cream:
function! s:BuildMenu()
"  +--------------------------+
"  | Filter:           Help X |
"  | Simple    Options  Favts |
"  |  Name.Ext Size Date Attr |
"
"  +--------------------------+
"  | Options Favorites Help X |
"  | Filter:                  |
"  | Simple (click to sort)   |
"  |  Name.Ext Size Date Attr |
"
	" add space to accommodate extra window width
	let myextra = winwidth(0) - s:split_width
	if myextra > 0
		let myspacer = <SID>SpaceString(myextra)
	else
		let myspacer = ""
	endif

	" determine options (show opposite of state for toggle!)
	if s:file_details == 1
		let mydetail = "Simple "
		call setline(1, " Options Favorites" . myspacer . " Help X ")
		call setline(2, " Filter:" . myspacer . "                  ")
		call setline(3, " " . mydetail . " (click to sort)" . myspacer . "  ")
		call setline(4, " >Name\.Ext Size Date Attr" . myspacer . " ")
	else
		let mydetail = "Details"
		call setline(1, " Options Favorites" . myspacer . " Help X ")
		call setline(2, " Filter:" . myspacer . "                  ")
		call setline(3, " " . mydetail . "                " . myspacer . "  ")
		call setline(4, "                         " . myspacer . " ")
	endif

	" go back to tree top
	execute "normal :" . s:treepos . "\<CR>"
endfunction

function! s:Menu(cmd)
" handle commands listed in the menu

	let mycmd = a:cmd
	let mypath = getline(s:treepos)

""*** DEBUG:
"let n = confirm(
"    \ "DEBUG:\n" .
"    \ "  mycmd    = " . mycmd . "\n" .
"    \ "\n", "&Ok\n&Cancel", 1, "Info")
"if n != 1
"    return
"endif
""***

	if mycmd == ""
		return
	"elseif mycmd == "Options"
	"elseif mycmd == "Favorites"
	elseif mycmd == "Filter"
		call <SID>SetMatchPattern()
	elseif mycmd == "Details"
		let s:file_details = 1
		call s:BuildTree(mypath)
		execute "cd " . mypath
	elseif mycmd == "Simple"
		let s:file_details = 0
		call s:BuildTree(mypath)
		execute "cd " . mypath
	"elseif mycmd == "Name"
	"elseif mycmd == "Ext"
	"elseif mycmd == "Size"
	"elseif mycmd == "Date"
	"elseif mycmd == "Attr"
	"elseif mycmd == "Help"
	elseif mycmd == "X"
		"close
		call Cream_close()

	" ignore
	elseif mycmd == ":"
	elseif mycmd == "("
	elseif mycmd == "click"
	elseif mycmd == "to"
	elseif mycmd == "sort"
	elseif mycmd == ")"
	else
		
		"*** DEBUG:
		call confirm(
			\ "Sorry, \"" . mycmd . "\" option not complete.\n" .
			\ "\n", "&Ok", 1, "Info")
		"***

	endif

endfunction
"+++

" Mouse Events {{{1

function! <SID>OnClick()
" act as DoubleClick only if
" * on Node
" * option to single-click as double-click is on

	let xpos = col('.') - 1
	let ypos = line('.')

	"+++ Cream: handle above-tree (menu) first
	if ypos < s:treepos
		" get word under cursor
		let mycmd = expand("<cword>")
		call s:Menu(mycmd)
		return
	endif
	"+++

	if s:IsTreeNode(xpos, ypos)
		call s:TreeNodeAction(xpos, ypos)
	elseif s:single_click_to_edit
		call <SID>OnDoubleClick(-1)
	endif
endfunction

function! <SID>OnDoubleClick(close_explorer)
" Arguments: 0  keeps window open
"            1  closes explorer window
"           -1  from mouse double-click, coordinates close option

	let s:close_explorer = a:close_explorer
	if s:close_explorer == -1
		let s:close_explorer = s:close_explorer_after_open
	endif
	let xpos = col('.') - 1
	let ypos = line('.')

	"+++ Cream: handle above-tree (menu) first
	if ypos < s:treepos
		" get word under cursor
		let mycmd = expand("<cword>")
		call s:Menu(mycmd)
		return
	endif
	"+++

	" clicked on node
	if s:IsTreeNode(xpos, ypos)
		call s:TreeNodeAction(xpos, ypos)
		return
	endif

	" go to first non-blank when line > 1
	if ypos > s:treepos
		normal 1|g^
		let xpos = col('.') - 1
		" check, if it's a directory
		let path = <SID>GetPathName(xpos, ypos)
		if isdirectory(path)
			"+++ Cream
			" just operate on node
			if s:click_on_dir_opens_node == 1
				" go to node (line beginning)
				normal g^
				if s:IsTreeNode(xpos, ypos)
					call s:TreeNodeAction(xpos, ypos)
				endif
			else
			"+++
				" build new root structure
				call s:BuildTree(path)
				execute "cd " . getline(s:treepos)
			endif
			return
		else
			" try to resolve filename
			" and open in other window
			let path = <SID>GetPathName(xpos, ypos)

			"+++ Cream: use our own file-open in buffer scheme
			if exists("$CREAM")
				call Cream_TryAnotherWindow()
				call Cream_file_open(path)
			else
				if filereadable(path)
					" go to last accessed buffer
					wincmd print
					" append sequence for opening file
					execute "cd " . fnamemodify(path, ":h")
					execute "edit " . path
					setlocal modifiable
				endif
			endif
			"+++
			if s:close_explorer == 1
				call <SID>ToggleShowExplorer()
			endif
		endif
		return
	endif

	" we're on line 1 here! getting new base path now...
	" advance to next slash
	normal f/
	if getline(s:treepos)[xpos] != '/'
		" no next slash == current directory, just rebuild
		if col('.') - 1 == xpos
			call s:BuildTree(getline(s:treepos))
			execute "cd " . getline(s:treepos)
			return
		endif
	"+++ Cream: fix problem going back up to root
	else
		call s:BuildTree('/')
		execute 'cd /'
		return
	"+++
	endif

	" cut ending slash
	normal h
	" rebuild tree with new path
	call s:BuildTree(strpart(getline(s:treepos), 0, col('.')))

endfunction

" Insert Filename or File Contents (unused) {{{1

function! <SID>InsertFilename()
	normal 1|g^
	let filename = <SID>GetPathName(col('.') - 1, line('.'))
	wincmd print
	execute "normal a" . filename
endfunction

function! <SID>InsertFileContent()
	normal 1|g^
	let filename = <SID>GetPathName(col('.') - 1, line('.'))
	if filereadable(filename)
		wincmd print
		execute "read " . filename
	endif
endfunction

" File operations (unused) {{{1

function! <SID>FileSee()
	normal 1|g^
	let filename = <SID>GetPathName(col('.') - 1, line('.'))
	if filereadable(filename)
		let i = system("see " . filename . "&")
	endif
endfunction

function! <SID>FileRename()
	normal 1|g^
	let filename = <SID>GetPathName(col('.') - 1, line('.'))
	if filereadable(filename)
		let newfilename = input("Rename to: ", filename)
		if filereadable(newfilename)
			if input("File exists, overwrite?") =~ "^[yY]"
				setlocal modifiable
				let i = system("mv -f " . filename . " " . newfilename)
				" refresh display
				normal gg$
				call <SID>OnDoubleClick(-1)
			endif
		else
			" rename file
			setlocal modifiable
			let i = system("mv " . filename . " " . newfilename)
			normal gg$
			call <SID>OnDoubleClick(-1)
		endif
	endif
endfunction

function! <SID>FileMove()
	normal 1|g^
	let filename = <SID>GetPathName(col('.') - 1, line('.'))
	if filereadable(filename)
		let newfilename = input("Move to: ", filename)
		if filereadable(newfilename)
			if input("File exists, overwrite?") =~ "^[yY]"
				" move file
				let i = system("mv -f " . filename . " " . newfilename)
				" refresh display
				normal gg$
				call <SID>OnDoubleClick(-1)
			endif
		else
			" move file
			let i = system("mv " . filename . " " . newfilename)
			" refresh display
			normal gg$
			call <SID>OnDoubleClick(-1)
		endif
	endif
endfunction

function! <SID>FileCopy()
	normal 1|g^
	let filename = <SID>GetPathName(col('.') - 1, line('.'))
	if filereadable(filename)
		let newfilename = input("Copy to: ", filename)
		if filereadable(newfilename)
			if input("File exists, overwrite?") =~ "^[yY]"
				" copy file
				let i = system("cp -f " . filename . " " . newfilename)
				" refresh display
				normal gg$
				call <SID>OnDoubleClick(-1)
			endif
		else
			" copy file
			let i = system("cp " . filename . " " . newfilename)
			" refresh display
			normal gg$
			call <SID>OnDoubleClick(-1)
		endif
	endif
endfunction

function! <SID>FileDelete()
	normal 1|g^
	let filename = <SID>GetPathName(col('.') - 1, line('.'))
	if filereadable(filename)
		if input("OK to delete " . fnamemodify(filename, ":t") . "? ")[0] =~ "[yY]"
			let i = system("rm -f " . filename)
			setlocal modifiable
			normal ddg^
			setlocal nomodifiable
		endif
	endif
endfunction

" vim:foldmethod=marker
