;;; -*- Syntax: Ansi-Common-Lisp; Base: 10; Mode: lisp; Package: odcl -*-
;;; $Id: simple-cache.lisp,v 1.8 2003/03/24 21:51:31 adam Exp $
;;;
;;; Copyright (c) 2000 - 2003 onShore Development, Inc.

;; simple cache protocol
;;

(in-package :odcl)

(defclass simple-cache-mixin ()
  ((keyfun  :initform nil
            :initarg :keyfun)
   (flushes :initform 0)
   (hits    :initform 0)
   (misses  :initform 0)
   (test    :initform 'eql
            :initarg :test)))

;; simple-cache-protocol

(defgeneric destroy-cache (cache)
  (:documentation "Destroy cache."))

(defgeneric cache-flush (cache)
  (:documentation "Flush all values in the cache."))

(defgeneric cache-get (cache key)
  (:documentation "Retrieve cached value for KEY.  Any marks
associated with the KEY will be returned as the second value."))

(defgeneric cache-put (cache key value &optional mark)
  (:documentation "Put value in cache under KEY.  The optional MARK
argument will mark the entry in the cache, or throw an error if the
cache does not support marking.  It will overwrite the existing VALUE
and MARK for KEY"))

(defgeneric cache-delete (cache key)
  (:documentation "Remove any value under KEY from the cache."))

(defgeneric cache-scanner (cache)
  (:documentation "Returns a function"))

(defgeneric cache-mark (cache key mark)
  (:documentation "Set the mark on KEY."))

(defmethod cache-print-stats (cache &optional (stream t))
  (with-slots (hits misses flushes)
    cache
    (format stream "~%~A Statistics: ~%~Thits/misses: ~d/~d ~%~Thitrate: ~f ~%~Tflushes: ~d"
            cache hits misses (* 100 (/ hits (+ hits misses))) flushes)))

(define-condition simple-cache-condition ()
  ())

(define-condition simple-cache-error (error simple-cache-condition)
  ((cache    :initarg :cache
             :reader error-cache)))

(define-condition simple-cache-mark-unsupported (simple-cache-error)
  ()
  (:report (lambda (c stream)
             (format stream "No cache marking for ~A" (error-cache c)))))

;; hash table based simple cache

(defclass hash-table-cache (simple-cache-mixin)
  ((hash        :initform nil)
   (markable    :initform nil)
   (default-val)
   (hash-args   :initform nil)))

;; note that this overrides initialize-instance entirely, so we need
;; to make sure we initialize all simple-cache-mixin slots as well
(defmethod initialize-instance ((cache hash-table-cache)
                                &rest initargs
                                &key (keyfun nil)
                                (markable nil)
                                &allow-other-keys)
  (let ((args (copy-list initargs)))
    (remf args :keyfun)
    (remf args :markable)
    (setf (slot-value cache 'default-val) (gensym)
          (slot-value cache 'hash-args) args
          (slot-value cache 'markable) markable
          (slot-value cache 'hits) 0
          (slot-value cache 'misses) 0
          (slot-value cache 'flushes) 0
          (slot-value cache 'keyfun) keyfun
          (slot-value cache 'hash)
          (apply #'make-hash-table args))))

(defmethod destroy-cache ((cache hash-table-cache))
  (slot-makunbound cache 'hash)
  nil)

(defmethod cache-flush ((cache hash-table-cache))
  (incf (slot-value cache 'flushes))
  (setf (slot-value cache 'hash)
        (apply #'make-hash-table (slot-value cache 'hash-args))))

(defmethod cache-get ((cache hash-table-cache) key)
  (with-slots (hash default-val markable misses hits)
    cache
    (let ((val (gethash key hash default-val)))
      (if (eql val default-val)
          (progn
            (incf misses)
            nil)
          (progn
            (incf hits)
            (if markable
                (values (car val)
                        (cdr val))
                val))))))

(defmethod cache-put ((cache hash-table-cache) key value &optional mark)
  (with-slots (hash markable)
    cache
    (let ((val (cond
                 ((and mark markable)
                  (cons value mark))
                 (mark
                  (error 'simple-cache-mark-unsupported :cache cache))
                 (markable
                  (cons value nil))
                 (t value))))
      (setf (gethash key hash) val)
      t)))

(defmethod cache-delete ((cache hash-table-cache) key)
  (with-slots (hash)
    cache
    (remhash key hash)))

(defmethod cache-scanner ((cache hash-table-cache))
  (error "Unimplemented."))

(defmethod cache-mark (cache key mark)
  (with-slots (hash markable)
    cache
    (if markable
        (cache-put cache key (cache-get cache key) mark)
        (error 'simple-cache-mark-unsupported :cache cache))))
