;;; -*- Syntax: Ansi-Common-Lisp; Base: 10; Mode: lisp; Package: odcl -*-
;;; $Id: lru-cache.lisp,v 1.17 2003/03/24 21:51:31 adam Exp $
;;;
;;; Copyright (c) 2000 - 2003 onShore Development, Inc.

;; Fixed-size cache with LRU replacement
;;
;; Operates like a fixed-size hashtable: When cache size reaches
;; capacity, the least recently accessed element is thrown out, unless
;; it is marked.
;;
;; Rehashing implemented, cache capacity doubles when cache fills with
;; marked entries.
;;
;; Marks are now generic: a non-nil value still prevents decaching,
;; but the value of the mark associated wtih an entry is returned
;; along with the entry's value.

(in-package :odcl)

(defclass lru-cache (simple-cache-mixin)
  ((capacity     :initarg :size
                 :initform 100)
   (count        :initform 0
                 :reader cache-size)
   (mcount       :initform 0
                 :accessor mark-count)
   (test         :initform #'eql
                 :initarg :test)
   (entries-size )
   (entries      )
   (insert-hook  :accessor insert-hook
                 :initarg :insert-hook
                 :initform nil)
   (delete-hook  :accessor delete-hook
                 :initarg :delete-hook
                 :initform nil)
   (buffer       :initform nil)))

(defmethod print-object ((self lru-cache) stream)
  (print-unreadable-object
   (self stream :type t
         :identity t)
   (if (slot-boundp self 'capacity)
       (with-slots (capacity count mcount)
         self
         (format stream "~d/~d/~d" mcount count capacity))
       (write-string "invalid" stream))))

(defmethod initialize-instance :after ((self lru-cache) &rest initargs)
  (declare (ignore initargs))
  (with-slots (capacity entries-size entries)
    self
    (let ((table-size (max 2 (floor (* capacity 1.5)))))
      (setf entries-size table-size
            entries (make-array table-size :initial-element nil)))))

(defmethod destroy-cache ((cache lru-cache))
  (with-slots (capacity entries-size entries buffer count)
    cache
    (slot-makunbound cache 'capacity)
    (setf entries-size 0
          count 0
          entries nil
          buffer nil)))

(defun %cache-resize (cache)
  (with-slots (capacity mcount test)
    cache
    (when (= capacity mcount)
      (let ((ncache (make-instance 'lru-cache :size (* 2 capacity) :test test))
            (scanner (%cache-scanner cache))
            (entry nil))
        (while (setf entry (funcall scanner))
          (destructuring-bind (k v m)
              entry
            (cache-put ncache k v m)))
        (setf capacity (* 2 capacity)
              (slot-value cache 'entries-size) (slot-value ncache 'entries-size)
              (slot-value cache 'entries) (slot-value ncache 'entries)
              (slot-value cache 'buffer) (slot-value ncache 'buffer))
        (destroy-cache ncache)))))

(defun cache-reset (cache)
  (let ((x (%cache-scanner cache))
        (y nil))
    (while (setf y (funcall x))
      (mapcar (lambda (hook)
                (funcall hook (second y)))
              (delete-hook cache))))
  (with-slots (entries entries-size count mcount buffer)
    cache
    (setf count 0
          mcount 0
          buffer nil)
    (dotimes (x entries-size)
      (setf (aref entries x) nil)))
  (values))

(defmethod cache-flush ((cache lru-cache))
  (incf (slot-value cache 'flushes))
  (cache-reset cache))

(defmethod cache-get ((cache lru-cache) key)
  (with-slots (hits misses)
    cache
    (if-bind (entry (%cache-get cache key))
        (progn (incf (lce-life entry))
               (incf hits)
               (values (lce-value entry)
                       (lce-mark entry)))
        (progn (incf misses)
               nil))))

(defmethod cache-put ((cache lru-cache) key value &optional mark)
  (with-slots (buffer count capacity entries)
    cache
    (flet ((free-lru ()
             (when buffer
               (do ((buf buffer (lce-next buf)))
                   ((and (not (lce-mark buf))
                         (= 0 (lce-life buf)))
                    (let ((entry buf))
                      (setf buffer (lce-remove buf))
                      (%cache-delete cache entry)))
                 (decf (lce-life buf))))))
      (multiple-value-bind (entry index)
          (%cache-get cache key)
        (if entry
            (let ((old-value (lce-value entry)))
              ;; (cmsg "Replace cache value: is correct?")
              (setf (lce-value entry) value)
              (update-mark cache entry mark)
              old-value)
            (let (en2)
              (when (= count capacity)
                (free-lru))
              (if (null buffer)
                  (progn
                    (setf buffer (lce-seed index))
                    (setf en2 buffer))
                  (setf en2 (lce-insert buffer index)))
              (setf (lce-key en2) key)
              (setf (lce-value en2) value)
              (setf (lce-index en2) index)
              (setf (lce-collision en2) (aref entries index))
              (setf (aref entries index) en2)
              (mapcar (lambda (hook)
                        (funcall hook value))
                      (insert-hook cache))
              (incf count)
              (update-mark cache en2 mark)
              nil))))))

(defun cache-update (cache key value)
  (with-slots (buffer count capacity entries)
    cache
    (multiple-value-bind (entry index)
        (%cache-get cache key)
      (declare (ignore index))
      (when entry
        (let ((old-value (lce-value entry)))
          (setf (lce-value entry) value)
          old-value)))))

(defun update-mark (cache entry mark)
  (cond ((and mark (not (lce-mark entry)))
         (incf (mark-count cache)))
        ((and (not mark) (lce-mark entry))
         (decf (mark-count cache))))
  (setf (lce-mark entry) mark)
  (%cache-resize cache))

(defmethod cache-mark ((cache lru-cache) key mark)
  (when-bind (entry (%cache-get cache key))
    (update-mark cache entry mark)))

(defmethod cache-delete ((cache lru-cache) key)
  (when-bind (entry (%cache-get cache key))
    (with-slots (buffer)
      cache
      (if (eq buffer entry)
          (setf buffer (lce-remove entry))
          (lce-remove entry))
      (%cache-delete cache entry)
      (lce-value entry))))

(defun %cache-get (cache key)
  (unless (slot-boundp cache 'capacity)
    (error "invalid cache"))
  (with-slots (entries entries-size test)
    cache
    (let ((hash (mod (sxhash key) entries-size)))
      (do ((entry (aref entries hash) (lce-collision entry)))
          ((or (null entry)
               (funcall test (lce-key entry) key))
           (values entry hash))))))

(defun %cache-scanner (cache)
  "returns a function that iterates the contents of the cache, returning (key value mark)"
  (with-slots (entries)
    cache
    (let ((max (1- (length entries)))
          (idx 0)
          (current-entry nil))
      (flet ((kvm (entry)
               (when entry
                 (list (lce-key entry)
                       (lce-value entry)
                       (lce-mark entry)))))
        (lambda ()
          (if (and current-entry
                   (lce-collision current-entry))
              (progn
                (setf current-entry (lce-collision current-entry))
                (kvm current-entry))
              (when (< idx max)
                (do ((entry (aref entries idx) (aref entries idx)))
                    ((or entry (<= max idx))
                     (progn
                       (setf current-entry entry)
                       (incf idx)
                       (kvm entry)))
                  (incf idx)))))))))

(defmethod cache-scanner ((cache lru-cache))
  (%cache-scanner cache))

(defun %cache-delete (cache entry)
  (with-slots (entries count mcount)
    cache
    (let ((en2 (aref entries (lce-index entry))))
      (if (eq en2 entry)
          (setf (aref entries (lce-index entry))
                (lce-collision entry))
          (do ((en3 en2 (lce-collision en3)))
              ((or (null en3)
                   (and (eq (lce-collision en3) entry)
                        (setf (lce-collision en3)
                              (lce-collision (lce-collision en3)))))))))
    ;; hook
    (mapcar (lambda (hook)
              (funcall hook (lce-value entry)))
            (delete-hook cache))
    (when (lce-mark entry)
      (decf mcount))
    (decf count)))

(defstruct (lru-cache-entry (:conc-name lce-)
                            (:print-function %print-lru-cache-entry))
  index
  (life 0)
  key
  value
  collision
  next
  prev
  mark)

(defmethod %print-lru-cache-entry (self stream depth)
  (declare (ignore depth))
  (print-unreadable-object
   (self stream :type t)
   (with-slots (value)
     self
     (princ value stream))))

(defun lce-seed (index)
  (let ((entry (make-lru-cache-entry :index index)))
    (setf (lce-next entry) entry
          (lce-prev entry) entry)
    entry))

(defun lce-insert (entry index)
  (let ((new-entry (make-lru-cache-entry :index index)))
    (setf (lce-prev new-entry)
          (lce-prev entry))
    (setf (lce-next (lce-prev new-entry))
          new-entry)
    (setf (lce-next new-entry)
          entry)
    new-entry))

(defun lce-remove (entry)
  (unless (eq entry (lce-next entry))
    (setf (lce-prev (lce-next entry))
          (lce-prev entry))
    (setf (lce-next (lce-prev entry))
          (lce-next entry))
    (lce-next entry)))
