;;; -*- Syntax: Ansi-Common-Lisp; Base: 10; Mode: lisp; Package: odcl -*-
;;; $Id: filesystem.lisp,v 1.50 2003/08/26 01:42:57 adam Exp $
;;;
;;; Copyright (c) 1999 - 2003 onShore Development, Inc.

(in-package :odcl)

(defun ensure-directory (path &key (if-does-not-exist :create)
                         (if-exists :ignore))
  "Ensure a directory PATH exists, and if not, create it if
:IF-DOES-NOT-EXIST is set to :CREATE or error out if set to :ERROR.
If :IF-EXISTS is set to :ERROR, an error will be flagged if the
directory already exists."
  (declare (optimize (speed 3)))
  (unless path
    (error "Directory PATH not set."))
  (ecase if-does-not-exist
    (:error
     (unless (probe-file path)
       (error "Directory ~s does not exist." path)))
    (:create
     (when (and (eq if-exists :error)
                (probe-file path))
       (error "Directory ~s already exists." path))
     (ensure-directories-exist
      ;; force path to be considered a dir by forcing trailing slash
      (extend-path path "")))))

#+cmu
(defun expand-directory (head)
  (let ((files nil)
        (subdirs nil)
        (dir (unix:open-dir head)))
    (when dir
      (unwind-protect
           (loop
            (let ((name (unix:read-dir dir)))
              (cond ((null name)
                     (return))
                    ((string= name "."))
                    ((string= name ".."))
                    (t
                     (let ((subdir (extend-path head name)))
                       (case (unix:unix-file-kind subdir t)
                         (:directory
                          (push (concatenate 'string subdir "/") subdirs))
                         (:file
                          (push subdir files))))))))
        (unix:close-dir dir)))
    (values files subdirs)))

;; requires expand-directory, which is cmucl only
#+cmu
(defun files-under (directory)
  (unless (null directory)
    (multiple-value-bind (subfiles subdirs)
	(expand-directory directory)
      (append
       subfiles
       (mapcan #'files-under subdirs)))))

(defun destroy-directory (directory &aux proc)
  "Delete the given DIRECTORY and files under it; returns t if delete
was sucessful, nil if there was no directory, and error if the
deletion failed."
  (if (probe-file directory)
      (progn
        #+asdf
        (let ((asdf::*verbose-out* nil))
          (setf proc (asdf:run-shell-command "rm -rf ~A" directory)))
        #+(and (not asdf) (or cmu sbcl))
        (setf proc (ext:process-exit-code
                    (ext:run-program "rm" (list "-rf" directory))))
        #+(and (not asdf) lispworks)
        (setf proc (system:call-system (list "rm" "-rf" "directory")
                                       :shell-type "/bin/sh"))
        (unless (= proc 0)
          (error "destroy-directory failed with ~D" proc))
        t)
      nil))

(defun temp-file (&optional (prefix "lisp") (suffix "tmp"))
  "Return a string describing a possible tempfile (will always be in
/tmp) given an optional PREFIX (defaults to 'lisp') and SUFFIX
(defaults to 'tmp')."
  (loop
   (let ((file (format nil "/tmp/~a.~a.~a" (random-string :length 8) prefix suffix)))
     (unless (probe-file file)
       (return-from temp-file file)))))

(defun copy-file (oldname newname &aux (buffer (make-array 8192)))
  "Copy file OLDNAME to new file NEWNAME."
  (with-open-file (istream oldname :direction :input)
    (with-open-file (ostream newname :direction :output
                             :if-does-not-exist :create
                             :if-exists :supersede)
      (loop
       (let ((pos (read-sequence buffer istream :start 0 :end 8191)))
         (write-sequence buffer ostream :start 0 :end pos)
         (when (< pos 8191)
           (return-from copy-file t)))))))

(defun extend-path (path element)
  "Add ELEMENT to the end of PATH, will add trailing slash to PATH as
needed."
  (concatenate 'string path
               (if (or (equal (last-char path) #\/)
                       (equal (first-char element) #\/))
                   ""
                   "/")
               element))

(defun write-to-file (path expr &aux retry)
  "Write EXPR to the file named PATH; will truncate the file if it
exists, or create it if not."
  (declare (optimize (speed 3)))
  (tagbody
     retry
     (when retry
       (ensure-directories-exist path))
     (handler-case
         (with-open-file (stream path :direction :output
                                 :if-does-not-exist :create
                                 :if-exists :supersede)
           (write expr :stream stream))
       (simple-error ()
         (when retry
           (error "Cannot write to filesystem."))
         (setf retry t)
         (go retry)))))

(defun read-from-file (path)
  "Read from file named PATH and return the contents."
  (with-open-file (stream path :direction :input)
    (read stream)))

;; filesystem DB
;;
;; Transaction management code has been moved to a higher level of
;; abstraction (see indexed-store.lisp).
;;
;; An FSDB a means of persisting key value pairs, where the key is a
;; non-negative, non-zero integer, and the value is a Lisp-readable
;; SEXP.
;;
;; Interface:
;;
;; make-fsdb \ should be merged
;; find-fsdb /
;; destroy-fsdb
;;
;; fsdb-add
;; fsdb-probe
;; fsdb-get
;; fsdb-del
;; fsdb-map
;; list-expressions

(defclass filesystem-db ()
  ((name :initarg :name)
   (path :initarg :path)
   (store-path :initarg :store-path)
   (version :initform 1)
   (maplen :initform nil
           :documentation "Current length of OID map.")
   (sequence :initform 0
             :documentation "Max OID at time of initialization, it is incremented to generate new OIDs.")
   (oidmap :initform nil
           :documentation "Bit vector indicating which OIDs have data.  It is 0 origin, while OIDs are 1 origin.")))

(defun make-fsdb (name path)
  "Create a new filesystem database at PATH."
  (let* ((store-path (concatenate 'string
                                  (extend-path path "store")
                                  "/"))
         (fsdb (make-instance 'filesystem-db :name name :path path :store-path store-path)))
    (init-fsdb fsdb)
    fsdb))

(defun %read-fsdb-version (fsdb)
  (with-slots (path)
      fsdb
    (let ((verpath (extend-path path "version")))
      (if (probe-file verpath)
          (ensure-integer (read-from-file verpath))
          0))))

(defun %write-fsdb-version (fsdb version)
  (with-slots (path)
      fsdb
    (let ((verpath (extend-path path "version")))
      (write-to-file verpath (ensure-integer version)))))

(defmethod migrate-fsdb-to-version (fsdb (version (eql 1)))
  (with-slots (store-path version)
      fsdb
    (dolist (entry (%fsdb-entries store-path))
      (when (equal :deleted (read-from-file entry))
        (cmsg "FSDB Entry ~A is deleted, removing file." entry)
        (delete-file entry))))
  t)

(defun migrate-fsdb (fsdb oldversion newversion)
  (when (< oldversion newversion)
    (unless (= (+ 1 oldversion) newversion)
      (migrate-fsdb fsdb oldversion (- newversion 1)))
    (cmsg "Migrating ~A from ~A to ~A" (slot-value fsdb 'name)
          oldversion newversion)
    (if (migrate-fsdb-to-version fsdb newversion)
        (%write-fsdb-version fsdb newversion)
        (error "FSDB migration for name from versions ~A to ~A failed."
               (slot-value fsdb 'name) oldversion newversion))))

(defun init-fsdb (fsdb)
  (with-slots (path store-path capacity name maplen oidmap sequence version)
      fsdb
    (unless (probe-file store-path)
      (cmsg "Initializing new FSDB ~A at path ~A, storage in ~A" name path store-path)
      (ensure-directory store-path))
    
    (unless (probe-file (extend-path path "version"))
      (%write-fsdb-version fsdb version))
    
    (let ((fs-version (%read-fsdb-version fsdb)))
      (migrate-fsdb fsdb fs-version version))
    (multiple-value-bind (map len)
        (%fsdb-oidmap fsdb)
      (setf oidmap map
            maplen len))
    (setf sequence (%fsdb-max-oid fsdb))))



(defun %fsdb-oidmap (fsdb)
  (let ((oidmap (make-array 200 :element-type 'bit :initial-element 0 :adjustable t))
        (maplen 200))
    (flet ((newlength (oid)
             (do ((maplen maplen (* 2 maplen)))
                 ((< oid maplen) maplen))))
      (with-slots (store-path)
        fsdb
        (dolist (entry (%fsdb-entries store-path))
          (when-bind (oid (%full-path-oid entry))
            (if (< oid maplen)
                (setf (elt oidmap (1- oid)) 1)
                (setf oidmap (adjust-array oidmap (newlength oid))
                      maplen (length oidmap)
                      (elt oidmap (1- oid)) 1)))))
      (values oidmap maplen))))


(defun %fsdb-check-oid (oidmap oid)
  (declare (inline))
  (ignore-errors (= 1 (elt oidmap (1- oid)))))

(defun %fsdb-register-oid (fsdb oid)
  (with-slots (oidmap maplen)
    fsdb
    (flet ((newlength (oid)
             (do ((maplen maplen (* 2 maplen)))
                 ((< oid maplen) maplen))))
      (if (< oid maplen)
          (setf (elt oidmap (1- oid)) 1)
          (setf oidmap (adjust-array oidmap (newlength oid))
                maplen (length oidmap)
                (elt oidmap (1- oid)) 1)))))
  
(defun %fsdb-unregister-oid (fsdb oid)
  (with-slots (oidmap maplen)
    fsdb
    (when (< oid maplen)
      (setf (elt oidmap (1- oid)) 0))))

(defun %fsdb-max-oid (fsdb)
  "Return current max OID that has been written to disk and is in the OID map."
  (with-slots (oidmap maplen)
    fsdb
    (do ((idx (1- maplen) (1- idx)))
        ((< idx 0))
      (when (= 1 (elt oidmap idx))
        (return-from %fsdb-max-oid (+ 1 idx))))
    0))

(defun fsdb-next-oid (fsdb)
  "Returns the next usable OID in the FSDB."
  (with-slots (sequence)
    fsdb
    (incf sequence)))

(defun %fsdb-entries (path &optional prefix &aux entries)
  (dolist (entry (directory path
                            :follow-links t
                            :check-for-subdirs t))
    (when-bind (subdir (directory-namestring entry))
      (unless (string= subdir path)
        (setf entries (append entries (%fsdb-entries subdir subdir)))))
    (when-bind (data (file-namestring entry))
      (if prefix
          (push (concatenate 'string prefix data) entries)
          (push data entries))))
  entries)


(defun destroy-fsdb (fsdb)
  "Destroy a filesystem database."
  (with-slots (name path)
    fsdb
    (when (probe-file path)
      (destroy-directory path)
      t)))

(defvar +decimal-printer+ #(#\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9))

(defun inscribe-base-10 (output offset size decimal)
  (declare (type fixnum offset size decimal)
           (type (simple-vector 10) +decimal-printer+))
  (dotimes (x size)
    (declare (type fixnum x)
             (optimize (safety 0)))
    (multiple-value-bind (next this)
        (floor decimal 10)
      (setf (aref output (+ (- size x 1) offset))
            (aref +decimal-printer+ this))
      (setf decimal next))))

(defun %get-oid-log (oid)
  "Map an integer onto a filesystem path."
  (declare (optimize (speed 1))
           (type fixnum oid))
  (flet ((get-bits (bitcount)
           (prog1
               (logand oid (1- (expt 2 bitcount)))
             (setf oid (ash oid (- bitcount))))))
    (let (path file file-position)
      (setf file-position (get-bits 0))
      (setf file (get-bits 8))
      (push (get-bits 8) path)
      (while (< 0 oid)
        (push (get-bits 8) path))
      (let ((string (make-string (+ (* 4 (length path)) 4) :initial-element #\/)))
        (dotimes (ii (length path))
          (inscribe-base-10 string (* 4 ii) 3 (nth ii path)))
        (inscribe-base-10 string (* 4 (length path)) 4 file)
        (values string file-position)))))

(defun %full-path-oid (path)
  (let* ((elts (split path #\/))
         (oid 0)
         (shift 0)
         (filestart (or (rest (member "store" elts :test #'string-equal))
                        (return-from %full-path-oid nil))))
    (dolist (component (reverse filestart))
      (when-bind (comp (ensure-integer component))
        (setf oid (logior (ash comp shift) oid))
        (incf shift 8)))
    oid))

(defun %oid-full-path (db oid)
  (with-slots (store-path)
    db
    (extend-path store-path (%get-oid-log oid))))

(defun fsdb-add (db oid expr)
  "Persistently map OID to EXPR in DB by storing it in the host
filesystem."
  (with-slots (oidmap)
    db
    (when (<= 0 oid)
      (write-to-file (%oid-full-path db oid) expr)
      (%fsdb-register-oid db oid))))

;; The only function (as of 10/10/02) that calls this is fsdb-probe,
;; for computing sequence position at init time.  This only checks to
;; see if oid was used at some point, but not whether or not it was
;; deleted.

(defun fsdb-probe (db oid)
  "Returns T if OID maps to an expression in DB."
  (with-slots (oidmap)
    db
    (when (<= 0 oid)
      (and
       (%fsdb-check-oid oidmap oid)
       (probe-file (%oid-full-path db oid))))))

(defun fsdb-get (db oid)
  "Return the expression in DB to which OID maps."
  (with-slots (oidmap maplen)
    db
    (when (and (<= 0 oid (1- maplen))
               (%fsdb-check-oid oidmap oid))
      (let ((file (%oid-full-path db oid)))
        (when (probe-file file)
          (let ((data (read-from-file file)))
              data))))))

(defun fsdb-del (db oid)
  "Delete the expression in DB to which OID maps."
  (when (<= 0 oid )
      (let ((fpath (%oid-full-path db oid)))
	(when (probe-file fpath)
	  (delete-file fpath))
        (%fsdb-unregister-oid db oid))))

(defun fsdb-oids (db &aux results)
  "Return list of oids in FSDB."
  (with-slots (oidmap maplen)
    db
    (dotimes (mapid (1- maplen))
      (when (= 1 (elt oidmap mapid))
        ;; mapids are 0 origin, OIDs are 1 origin
        (let ((file (%oid-full-path db (1+ mapid))))
          (when (probe-file file)
            (push (1+ mapid) results))))))
  (reverse results))

(defun fsdb-map (fn db)
  "Map two-arg FN taking oid and expression against a filesystem db"
  (with-slots (oidmap maplen)
    db
    (dotimes (mapid (1- maplen))
      (when (= 1 (elt oidmap mapid))
        ;; mapids are 0 origin, OIDs are 1 origin
        (let ((file (%oid-full-path db (1+ mapid))))
          (when (probe-file file)
            (let ((data (read-from-file file)))
              (funcall fn (1+ mapid) data))))))))

(defun list-expressions (db &aux result)
  "returns an unordered list of (OID . EXP) pairs"
  (fsdb-map (lambda (k v)
              (push (cons k v) result)) db)
  result)
