/* cache-config.c
 *
 ****************************************************************
 * Copyright (C) 2003 Tom Lord, 2004 Aaron Bentley
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "config-options.h"
#include "po/gettext.h"
#include "hackerlab/cmd/main.h"
#include "libfsutils/dir-as-cwd.h"
#include "libarch/my.h"
#include "commands/version.h"
#include "libarch/pfs.h"
#include "errno.h"
#include "cache-config.h"



static t_uchar * usage = N_("[options] [dir]");

#define OPTS(OP) \
  OP (opt_help_msg, "h", "help", 0, \
      N_("Display a help message and exit.")) \
  OP (opt_long_help, "H", 0, 0, \
      N_("Display a verbose help message and exit.")) \
  OP (opt_version, "V", "version", 0, \
      N_("Display a release identifier string\n" \
      "and exit.")) \
  OP (opt_disable, "d", "disable", 0, \
      N_("Disable caching, and forget the directory"))

t_uchar arch_cmd_cache_config_help[] = N_("Print or change your cache settings\n"
                                               "With no argument, and without -d, print the path to your cache.\n"
                                               "\n"
                                               "With an argument, record DIR as the path to your \n"
                                               "\n"
                                               "With the option -d (--disable) and no argument, disable caching \n"
                                               "and forget the cache directory\n");

enum options
{
  OPTS (OPT_ENUM)
};

static struct opt_desc opts[] =
{
  OPTS (OPT_DESC)
    {-1, 0, 0, 0, 0}
};



static int dir_or_missing (t_uchar * path)
{
  struct stat statb;
  int errn;
  if (vu_stat (&errn, path, &statb) == -1)
  {
    if (errn == ENOENT)
      {
        errn = 0;
        return 1;
      }
    else 
      {
        safe_printfmt (2, "Error checking cache directory:\n %s\n", errno_to_string (errn));
        exit (1);
      }
  }
  return (S_ISDIR ((statb.st_mode)));
}

extern int
arch_cmd_cache_config (t_uchar * program_name, int argc, char * argv[])
{
  int o;
  struct opt_parsed * option;
  char * dir = 0;
  char * new_dir = 0;
  int disable = 0;
  int status = 0;

  safe_buffer_fd (1, 0, O_WRONLY, 0);

  option = 0;

  while (1)
    {
      o = opt_standard (lim_use_must_malloc, &option, opts, &argc, argv, program_name, usage, libarch_version_string, arch_cmd_cache_config_help, opt_help_msg, opt_long_help, opt_version);
      if (o == opt_none)
        break;
      switch (o)
        {
        default:
          safe_printfmt (2, "unhandled option `%s'\n", option->opt_string);
          panic ("internal error parsing arguments");

        usage_error:
          opt_usage (2, argv[0], program_name, usage, 1);
          exit (1);

          /* bogus_arg: */
          safe_printfmt (2, "ill-formed argument for `%s' (`%s')\n", option->opt_string, option->arg_string);
          goto usage_error;

        case opt_disable:
          {
            disable = 1;
            break;
          }
        }
    }


  if (argc > 2)
    goto usage_error;

  if (argc == 2)
    new_dir = arch_abs_path(argv[1]);

  if (new_dir != 0 && disable)
      goto usage_error;
  if (new_dir != 0)
    {
      if (!dir_or_missing (new_dir))
        {
          safe_printfmt (2, _("Not a directory: %s\n"), argv[1]);
          status = 1;
        }
      else
        {
          arch_set_my_cache_path (new_dir);
        }
    }
  else if (disable)
    arch_set_my_cache_path ("");
  else
    {
      dir = arch_my_cache_path ();
      if (dir != 0)
        safe_printfmt (2, _("Location: %s\n"), dir);
      else
        safe_printfmt (2, _("Cache disabled\n"));
    }

  lim_free(0, new_dir);
  lim_free(0, dir);
  return status;
}




/* tag: 684e0852-38d9-4737-9505-bc5154b895f4 */
