#!/usr/bin/env python
# arch-tag: 6e2d3c18-a7b2-4d3c-a4ae-0fb1773e7214
# Copyright (C) 2003 Ben Burns <bburns@mailops.com>
#               2004 David Allouche <david@allouche.net>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""Main test suite
"""

import types
import sys
import os
import shutil

import pybaz as arch
from pybaz import errors
from pybaz.pathname import DirName, FileName, PathName

import framework
from framework import TestCase
import fixtures


def explicit_id_name(name):
    if isinstance(name, FileName):
        dirname = name.dirname()
        basename = name.basename()
        return dirname/'.arch-ids'/basename+'.id'
    elif isinstance(name, DirName):
        return name/'.arch-ids'/'=id'
    else:
        raise AssertionError

def explicit_files_import_listing(names):
    # baz-1.1 fixes a bug in the New-files patchlog header of import
    # revisions. Older releases would not show explicit id files on
    # import, but would show them on commit. Starting with baz-1.1,
    # explicit id files are listed in the patchlog of import
    # revisions.
    ret = [ n for n in names if isinstance(n, FileName) ]
    if arch.backend.version.release >= (1, 1, 0):
        for name in names:
            ret.append(explicit_id_name(name))
    ret.sort()
    return ret


class EscapedWhitespace(TestCase):

    tests = []

    def extraSetup(self):
        self.params.set_my_id()
        self.params.create_archive()
        self.params.create_working_tree(self.params.version)

    def setUp(self):
        TestCase.setUp(self)
        self.tree = self.params.working_tree
        self.file_name = FileName('foo bar')
        open(self.tree/self.file_name, 'w').write('')
        self.tree.add_tag(self.file_name)
        tagfile = self.tree/'.arch-ids'/self.file_name+'.id'
        print >> open(tagfile, 'w'), 'baby jesus cries'
        self.tree.import_()

    def get_index_escaped(self):
        """Changeset.get_index supports escaped file names."""
        open(self.tree/self.file_name, 'w').write('Hello, World!')
        m = self.tree.log_message()
        m['Summary'] = 'first revision'
        self.tree.commit(m)
        cset = self.params.version['patch-1'].get_patch(self.tree/',cset')
        expected = { 'x_baby_jesus_cries': self.file_name }
        result = cset.get_index('orig-files')
        self.assertEqual(expected, result)
    tests.append('get_index_escaped')

    def file_find_pristine_escaped(self):
        """file_find with pristine supports escaped names."""
        revision = self.params.version['base-0']
        result = self.tree.file_find(self.file_name, revision)
        pristine = self.tree.find_pristine(revision)
        expected = pristine/self.file_name
        self.assertEqual(expected, result)
    tests.append('file_find_pristine_escaped')


class TreeRevision(TestCase):

    def extraSetup(self):
        self.params.set_my_id()
        self.params.create_archive()

    tests = []

    def tree_revision(self):
        """tree_revision works"""
        os.mkdir(self.params.working_dir)
        wt = arch.init_tree(self.params.working_dir, self.params.version)
        wt.import_()
        self.assertEqual(self.params.version['base-0'], wt.tree_revision)
        m = wt.log_message()
        m['Summary'] = m.description = 'first revision'
        wt.commit(m)
        self.assertEqual(self.params.version['patch-1'], wt.tree_revision)
    tests.append('tree_revision')


class LatestVersion(TestCase):

    def extraSetup(self):
        self.params.set_my_id()

    tests = []

    def latest_version(self):
        return self.params.version.branch.latest_version()

    def not_registered(self):
        """Branch.latest_version ValueError if archive is not registered."""
        self.assertRaises(ValueError, self.latest_version)
    tests.append('not_registered')

    def missing_branch(self):
        """Branch.latest_version ValueError if branch does not exist."""
        self.params.create_archive()
        self.assertRaises(ValueError, self.latest_version)
    tests.append('missing_branch')

    def works(self):
        """Branch.latest_version works."""
        self.params.create_archive()
        self.params.create_working_tree(self.params.version)
        tree = self.params.working_tree
        tree.import_()
        version = self.params.version
        branch = version.branch
        self.assertEqual(version, branch.latest_version())
        next_version = arch.Version(version.fullname + '.1')
        tree.tree_revision.make_continuation(next_version)
        self.assertEqual(next_version, branch.latest_version())
    tests.append('works')


class LatestRevision(TestCase):

    def extraSetup(self):
        self.params.set_my_id()

    tests = []

    def latest_revision(self):
        return self.params.version.latest_revision()

    def not_registered(self):
        """Version.latest_revision ValueError if archive is not registered."""
        self.assertRaises(ValueError, self.latest_revision)
    tests.append('not_registered')

    def missing_version(self):
        """Version.latest_revision ValueError if version does not exist."""
        self.params.create_archive()
        self.assertRaises(ValueError, self.latest_revision)
    tests.append('missing_version')

    def empty_version(self):
        """Version.latest_revision ValueError if version is empty."""
        self.params.create_archive()
        location = DirName(self.params.archive.location)
        os.mkdir(location/self.params.version.nonarch)
        self.params.version.exists()
        self.assertRaises(ValueError, self.latest_revision)
    tests.append('empty_version')

    def works(self):
        """Version.latest_revision works."""
        self.params.create_archive()
        self.params.create_working_tree(self.params.version)
        tree = self.params.working_tree
        tree.import_()
        version = self.params.version
        self.assertEqual(version['base-0'], version.latest_revision())
        self.params.commit_summary(tree, 'first revision')
        self.assertEqual(version['patch-1'], version.latest_revision())
    tests.append('works')


class VersionIndex(TestCase):

    def make_revisions(self):
        self.params.set_my_id()
        self.params.create_archive()
        self.params.create_working_tree(self.params.version)
        tree = self.params.working_tree
        tree.import_()
        self.params.commit_summary(tree, 'first revision')
        self.params.commit_summary(tree, 'second revision')

    tests = []

    def version_index_zero(self):
        """Version[0] gives the first revision."""
        self.make_revisions()
        first = self.params.version.iter_revisions().next()
        self.assertEqual(self.params.version[0], first)

    tests.append('version_index_zero')

    def version_index_minus_one(self):
        """Version[-1] gives the latest revision."""
        self.make_revisions()
        latest = self.params.version.latest_revision()
        self.assertEqual(self.params.version[-1], latest)

    tests.append('version_index_minus_one')

    def version_index_error(self):
        """Version[n] gives IndexError when n is out of bounds."""
        import operator
        self.make_revisions()
        count = len(list(self.params.version.iter_revisions()))
        self.assertRaises(IndexError, operator.getitem,
                          self.params.version, count)
        self.assertRaises(IndexError, operator.getitem,
                          self.params.version, -count-1)

    tests.append('version_index_error')

    def version_index_type_error(self):
        """Version[idx] gives TypeError if idx is neither str nor int."""
        import operator
        self.assertRaises(TypeError, operator.getitem,
                          self.params.version, object())

    tests.append('version_index_type_error')


class NamespaceUnregistered(framework.NewTestCase):
    fixture = fixtures.UnregisteredArchiveFixture()

    def test_category_exists_unregistered(self):
        """Category.exists raises ArchiveNotRegistered if not registered."""
        category = self.fixture.archive['cat']
        self.assertRaises(errors.ArchiveNotRegistered, category.exists)

    def test_branch_exists_unregistered(self):
        """Branch.exists raises ArchiveNotRegistered if not registered."""
        branch = self.fixture.archive['cat']['brn']
        self.assertRaises(errors.ArchiveNotRegistered, branch.exists)

    def test_version_exists_unregistered(self):
        """Version.exists raises ArchiveNotRegistered if not registered."""
        version = self.fixture.archive['cat']['brn']['0']
        self.assertRaises(errors.ArchiveNotRegistered, version.exists)

    def test_revision_exists_unregistered(self):
        """Revision.exists raises ArchiveNotRegistered if not registered."""
        revision = self.fixture.archive['cat']['brn']['0']['patch-1']
        self.assertRaises(errors.ArchiveNotRegistered, revision.exists)



class NamespaceExists(framework.NewTestCase):
    fixture = fixtures.HistoryFixture({}) # empty import

    def test_category_exists(self):
        """Category.exists works."""
        category = self.fixture.version.category
        self.assertEqual(category.exists(), True)
        other_category = category.archive['not-here']
        self.assertEqual(other_category.exists(), False)
        class CategorySubclass(arch.Category): pass
        categorysub = CategorySubclass(category)
        self.assertEqual(categorysub.exists(), True)

    def test_branch_exists(self):
        """Branch.exists works."""
        branch = self.fixture.version.branch
        self.assertEqual(branch.exists(), True)
        other_branch = branch.category['not-here']
        self.assertEqual(other_branch.exists(), False)
        in_other_cat = branch.archive['not-here']['not-here']
        self.assertEqual(in_other_cat.exists(), False)
        class BranchSubclass(arch.Branch): pass
        branchsub = BranchSubclass(branch)
        self.assertEqual(branchsub.exists(), True)

    def test_version_exists(self):
        """Version.exists works."""
        version = self.fixture.version
        self.assertEqual(version.exists(), True)
        other_version = version.branch['42']
        self.assertEqual(other_version.exists(), False)
        in_other_brn = version.category['not-here']['0']
        self.assertEqual(in_other_brn.exists(), False)
        class VersionSubclass(arch.Version): pass
        versionsub = VersionSubclass(version)
        self.assertEqual(versionsub.exists(), True)

    def test_revision_exists(self):
        """Revision.exists works."""
        revision = self.fixture.version['base-0']
        self.assertEqual(revision.exists(), True)
        other_revision = revision.version['patch-1']
        self.assertEqual(other_revision.exists(), False)
        in_other_vsn = revision.branch['42']['base-0']
        self.assertEqual(in_other_vsn.exists(), False)
        class RevisionSubclass(arch.Revision): pass
        revisionsub = RevisionSubclass(revision)
        self.assertEqual(revisionsub.exists(), True)


class Cachedrevs(framework.NewTestCase):
    fixture = fixtures.SimpleRevisionFixture()

    def ls_cached(self):
        gen = self.fixture.version.iter_cachedrevs()
        self.assert_(isinstance(gen, types.GeneratorType))
        return list(gen)

    def test_cachedrevs_none(self):
        """Version.iter_cachederevs shows no initially cached revs."""
        self.assertEqual(self.ls_cached(), [])

    def test_cacherev(self):
        """Revision.cache() effects iter_cachedrevs."""
        self.fixture.version['base-0'].cache()
        self.assertEqual(self.ls_cached(), [self.fixture.version['base-0']])
        self.fixture.version['patch-1'].cache()
        self.assertEqual(self.ls_cached(),
                         map(lambda(x): self.fixture.version[x],
                             ('base-0', 'patch-1')))

    def test_uncacherev(self):
        """Revision.uncache() effects iter_cachedrevs."""
        self.fixture.version['base-0'].cache()
        self.fixture.version['patch-1'].cache()
        self.fixture.version['base-0'].uncache()
        self.assertEqual(self.ls_cached(), [self.fixture.version['patch-1']])
        self.fixture.version['patch-1'].uncache()
        self.assertEqual(self.ls_cached(), [])


class PopulatedArchive(TestCase):

    def extraSetup(self):
        self.params.set_my_id()
        self.params.create_archive()
        self.params.create_working_tree(self.params.version)

    def setUp(self):
        TestCase.setUp(self)
        self.tree = self.params.working_tree
        self.tree.import_()
        self.params.commit_summary(self.tree, 'first revision')

    tests = []

    def iter_logs(self):
        """SourceTree.iter_log for the tree-version."""
        def ls_logs_revision(vsn=None):
            gen = self.tree.iter_logs(vsn)
            self.assert_(isinstance(gen, types.GeneratorType))
            return map(lambda(x): x.revision, gen)
        rvsns = map(lambda(x): self.params.version[x], ('base-0', 'patch-1'))
        self.assertEqual(ls_logs_revision(), rvsns)
        self.assertEqual(ls_logs_revision(self.params.version), rvsns)
        self.assertEqual(ls_logs_revision(self.params.version.fullname), rvsns)
    tests.append('iter_logs')

    def continuation_from_version(self):
        """Create a continuation version (from version)."""
        version = self.params.version
        other_version = self.params.other_version
        assert not other_version.exists()
        version.latest_revision().make_continuation(other_version)
        self.failUnless(other_version.exists())
        revisions = tuple(other_version.iter_revisions())
        other_base = other_version['base-0']
        self.failUnlessEqual(revisions, (other_base,))
        summary = other_base.patchlog.summary
        self.failUnlessEqual(summary, 'tag of %s' % version.latest_revision())
    tests.append('continuation_from_version')

    def mirror_basic(self):
        """Archive.mirror without fromto, pushing."""
        master = self.params.archive
        mirror_name = master.name + '-MIRROR'
        location = self.params.arch_dir/mirror_name
        mirror = master.make_mirror(mirror_name, location)
        master.mirror()
        def nonarch(x): return x.nonarch
        master_revisions = map(nonarch, master.iter_revisions())
        mirror_revisions = map(nonarch, mirror.iter_revisions())
        self.failUnlessEqual(master_revisions, mirror_revisions)
    tests.append('mirror_basic')

    def mirror_fromto(self):
        """Archive.mirror with fromto."""
        master = self.params.archive
        mirror_name = master.name + '-MIRROR'
        arch_dir = self.params.arch_dir
        mirror = master.make_mirror(mirror_name, arch_dir/mirror_name)
        master.mirror()
        third_name = master.name + '-third'
        third = master.make_mirror(third_name, arch_dir/third_name)
        master.mirror(fromto=(mirror, third))
        def nonarch(x): return x.nonarch
        mirror_revisions = map(nonarch, mirror.iter_revisions())
        third_revisions = map(nonarch, third.iter_revisions())
        self.failUnlessEqual(mirror_revisions, third_revisions)
    tests.append('mirror_fromto')

    def mirror_limit(self):
        """Archive.mirror with limit version"""
        version = self.params.version
        other_version = self.params.other_version
        version.latest_revision().make_continuation(other_version)
        master = self.params.archive
        mirror_name = master.name + '-MIRROR'
        location = self.params.arch_dir/mirror_name
        mirror = master.make_mirror(mirror_name, location)
        master.mirror(limit=(version,))
        def nonarch(x): return x.nonarch
        mirror_versions = tuple(map(nonarch, mirror.iter_versions()))
        self.failUnlessEqual(mirror_versions, (version.nonarch,))
    tests.append('mirror_limit')

    def _check_logs(self, tree, levels):
        expected_revs = map(lambda (x): self.params.version[x], levels)
        found_revs = map(lambda (x): x.revision, tree.iter_logs())
        self.assertEqual(expected_revs, found_revs)

    def deprecated_get(self):
        """The deprecated arch.get function works."""
        shutil.rmtree(self.params.working_dir)
        tree = arch.get(self.params.version['base-0'], self.params.working_dir)
        self._check_logs(tree, ['base-0'])
    tests.append('deprecated_get')

    def revision_get(self):
        """Get a working tree, given a revision."""
        shutil.rmtree(self.params.working_dir)
        tree = self.params.version['base-0'].get(self.params.working_dir)
        self._check_logs(tree, ['base-0'])
    tests.append('revision_get')

    def version_get(self):
        """Get a working tree, given a version."""
        shutil.rmtree(self.params.working_dir)
        tree = self.params.version.get(self.params.working_dir)
        self._check_logs(tree, ['base-0', 'patch-1'])
        self.params.commit_summary(tree, 'second revision')
        shutil.rmtree(self.params.working_dir)
        tree = self.params.version.get(self.params.working_dir)
        self._check_logs(tree, ['base-0', 'patch-1', 'patch-2'])
    tests.append('version_get')

    def tree_update(self):
        """WorkingTree.update, without options."""
        shutil.rmtree(self.params.working_dir)
        tree = self.params.version['base-0'].get(self.params.working_dir)
        tree.update()
        self._check_logs(tree, ['base-0', 'patch-1'])
    tests.append('tree_update')


class Merges(PopulatedArchive):

    tests = []

    def setup_first_merge(self):
        version = self.params.version
        other_version = self.params.other_version
        tree = self.tree
        version.latest_revision().make_continuation(other_version)
        other_tree = other_version['base-0'].get(self.params.other_working_dir)
        self.other_tree = other_tree
        self.params.commit_summary(tree, 'second rev')
        self.params.commit_summary(tree, 'third rev')

    def star_merge(self):
        """star-merge with a version"""
        self.setup_first_merge()
        other_tree = self.other_tree
        other_tree.star_merge(self.params.version)
        msg = 'merge'
        self.params.commit_summary(other_tree, msg)
        log = self.params.other_version['patch-1'].patchlog
        self.assertEqual(log.summary, msg)
        vsn = self.params.version
        expected = [vsn['patch-2'], vsn['patch-3']]
        self.assertEqual(log.merged_patches, expected)
    tests.append('star_merge')

    def setup_star_merge(self):
        version, other_version = self.params.version, self.params.other_version
        open(self.tree/'foo', 'w').write('hello')
        self.tree.add_tag('foo')
        self.params.commit_summary(self.tree, 'second rev')
        version.latest_revision().make_continuation(other_version)
        other_tree = other_version['base-0'].get(self.params.other_working_dir)
        self.other_tree = other_tree
        open(self.tree/'foo', 'w').write('hello world')
        self.params.commit_summary(self.tree, 'third rev')

    def assertObject(self, obj, class_, **attrs):
        self.assert_(isinstance(obj, class_))
        for k, v in attrs.items():
            self.assertEqual(getattr(obj, k), v)

    def iter_star_merge(self):
        """star-merge incremental output"""
        self.setup_star_merge()
        changes = {}
        for line in self.other_tree.iter_star_merge(self.params.version):
            if isinstance(line, arch.Chatter):
                continue
            changes[line.name] = line
        expected = (('foo', arch.FileModification, {'binary': False}),
                    ('{arch}/cat/cat--brn/cat--brn--1.0/'
                     'jdoe@example.com--example--9999/patch-log/patch-3',
                     arch.FileAddition, {'directory': False}))
        self.assertEqual(len(expected), len(changes))
        for name, class_, attrs in expected:
            self.assert_(name in changes)
            change = changes[name]
            self.assertObject(change, class_, **attrs)
    tests.append('iter_star_merge')

    def iter_star_merge_conflict(self):
        """incremental star-merge conflict"""
        self.setup_star_merge()
        open(self.tree/'foo', 'w').write('hello world')
        self.params.commit_summary(self.tree, 'third rev')
        open(self.other_tree/'foo', 'w').write('Hello, World!')
        self.params.commit_summary(self.other_tree, 'first rev')
        cset_app = self.other_tree.iter_star_merge(self.params.version)
        for unused in cset_app: pass
        self.assert_(cset_app.conflicting)
    tests.append('iter_star_merge_conflict')

    def setup_merges(self):
        self.setup_first_merge()
        version, other_version = self.params.version, self.params.other_version
        tree, other_tree = self.tree, self.other_tree
        other_tree.star_merge(version)
        self.params.commit_summary(other_tree, 'merge patch-3')
        self.params.commit_summary(other_tree, 'empty merge')

    def depth_first(self, merges_iter):
        merges = []
        for i, F in merges_iter:
            for f in F:
                merges.append([i, f])
        return merges

    def iter_depth(self, reverse=False):
        """Version.iter_merges depth first"""
        self.setup_merges()
        vsn, oth = self.params.version, self.params.other_version
        expected = [[oth['base-0'], vsn['base-0']],
                    [oth['base-0'], vsn['patch-1']],
                    [oth['base-0'], oth['base-0']],
                    [oth['patch-1'], vsn['patch-2']],
                    [oth['patch-1'], vsn['patch-3']],
                    [oth['patch-1'], oth['patch-1']],
                    [oth['patch-2'], oth['patch-2']]]
        if reverse: expected.reverse()
        merges_iter = self.params.other_version.iter_merges(reverse=reverse)
        merges = self.depth_first(merges_iter)
        self.assertEqual(expected, merges)
    tests.append('iter_depth')

    def iter_depth_reverse(self):
        """Version.iter_merges depth first in reverse"""
        self.iter_depth(reverse=True)
    tests.append('iter_depth_reverse')

    def iter_not_me(self):
        """Version.iter_merges metoo=False"""
        self.setup_merges()
        vsn, oth = self.params.version, self.params.other_version
        expected = [[oth['base-0'], vsn['base-0']],
                    [oth['base-0'], vsn['patch-1']],
                    [oth['patch-1'], vsn['patch-2']],
                    [oth['patch-1'], vsn['patch-3']]]
        other_version = self.params.other_version
        merges = self.depth_first(other_version.iter_merges(metoo=False))
        self.assertEqual(expected, merges)
    tests.append('iter_not_me')

    def iter_version(self):
        """Version.iter_merges limited to a single source version"""
        self.setup_merges()
        vsn, oth = self.params.version, self.params.other_version
        expected = [[oth['base-0'], oth['base-0']],
                    [oth['patch-1'], oth['patch-1']],
                    [oth['patch-2'], oth['patch-2']]]
        other_version = self.params.other_version
        merges_iter = other_version.iter_merges(self.params.other_version)
        merges = self.depth_first(merges_iter)
        self.assertEqual(expected, merges)
    tests.append('iter_version')

    def breadth_first(self, iterator):
        merges = map(list, iterator)
        for step in merges:
            step[1] = list(step[1])
        return merges

    def iter_breadth(self):
        """Version.iter_merges with breadth first traversal."""
        self.setup_merges()
        merges_iter = self.params.other_version.iter_merges()
        merges = self.breadth_first(merges_iter)
        vsn, oth = self.params.version, self.params.other_version
        expected = [[oth['base-0'], [vsn['base-0'],
                                     vsn['patch-1'],
                                     oth['base-0']]],
                    [oth['patch-1'], [vsn['patch-2'],
                                      vsn['patch-3'],
                                      oth['patch-1']]],
                    [oth['patch-2'], [oth['patch-2']]]]
        self.assertEqual(expected, merges)
    tests.append('iter_breadth')


class RevisionChecksum(framework.NewTestCase):

    def test_parse_checksum(self):
        """Parsing checksum file data"""
        checksum = '\n'.join((
            "-----BEGIN PGP SIGNED MESSAGE-----",
            "Hash: SHA1",
            "",
            "Signature-for: ddaa@ddaa.net--2004/pyarch--ddaa--0.5--base-0",
            "md5 pyarch--ddaa--0.5--base-0.tar.gz"
            " f1998e8294e50070993471cec6b66b52",
            "foo bar-baz-blah 0123456789abcdef",
            "sha1 pyarch--ddaa--0.5--base-0.tar.gz"
            " 633c0efcceab4e6588a04a9ce73233bc206bcdf9",
            "-----BEGIN PGP SIGNATURE-----",
            "Version: GnuPG v1.2.4 (GNU/Linux)",
            "",
            "iD8DBQFAUQ/bZEH9AkgfRL0RAiiFAKCnV9jenFvCsckbIigtQmHYfZgcKwCgk5FY",
            "XQwvigYzMiiR9lFZh6vYUJM=",
            "=PCU/",
            "-----END PGP SIGNATURE-----",
            ""))
        rvsn  = arch.Revision('ddaa@ddaa.net--2004/pyarch--ddaa--0.5--base-0')
        checksums = rvsn._parse_checksum(checksum)
        expected = {'pyarch--ddaa--0.5--base-0.tar.gz':
                    {'md5': 'f1998e8294e50070993471cec6b66b52',
                     'sha1': '633c0efcceab4e6588a04a9ce73233bc206bcdf9'},
                    'bar-baz-blah': {'foo': '0123456789abcdef'}}
        self.assertEqual(expected, checksums)


class RevisionFiles(framework.NewTestCase):
    fixture = fixtures.SimpleRevisionFixture()

    def _help_file_names(self, rvsn, expected_names):
        files = rvsn.iter_files()
        file_names = map(lambda (x): x.name, files)
        file_names.sort()
        expected_names.sort()
        self.assertEqual(file_names, expected_names)

    def test_import_files(self):
        """Import revisions have expected files."""
        rvsn  = self.fixture.version['base-0']
        expected_names = ['checksum', 'log', rvsn.nonarch + '.src.tar.gz' ]
        self._help_file_names(rvsn, expected_names)

    def test_commit_files(self):
        """Commit revisions (uncached) have expected files."""
        rvsn = self.fixture.version['patch-1']
        expected_names = ['checksum', 'log', rvsn.nonarch + '.patches.tar.gz']
        self._help_file_names(rvsn, expected_names)

    def test_cachedrev_files(self):
        """Commit revisions (cached) have expected files."""
        rvsn = self.fixture.version['patch-1']
        rvsn.cache()
        expected_names = ['checksum', 'log', rvsn.nonarch + '.patches.tar.gz',
                          'checksum.cacherev', rvsn.nonarch + '.tar.gz']
        self._help_file_names(rvsn, expected_names)

    def _revision_dir(self):
        archive_dir = self.fixture.location(self.fixture.archive)
        vsn = self.fixture.version.nonarch
        rev_dir = archive_dir / vsn / 'patch-1'
        return DirName(rev_dir)

    def test_revision_files_data(self):
        """Accessing the data of Revision files."""
        rvsn = self.fixture.version['patch-1']
        rvsn.cache()
        rev_dir = self._revision_dir()
        for rev_file in rvsn.iter_files():
            self.assert_(rev_file.data == open(rev_dir/rev_file.name).read())


class TlaRevisionFiles(RevisionFiles):
    fixture = fixtures.SimpleTlaRevisionFixture()

    def test_import_files(self):
        """tla-format import revisions have expected files."""
        RevisionFiles.test_import_files(self)

    def test_commit_files(self):
        """tla-format commit revisinos (uncached) have expected files."""
        RevisionFiles.test_commit_files(self)

    def test_cachedrev_files(self):
        """tla-format commit revisions (cached) have expected files."""
        RevisionFiles.test_cachedrev_files(self)

    def _revision_dir(self):
        archive_dir = self.fixture.location(self.fixture.archive)
        cat = self.fixture.version.category.nonarch
        brn = self.fixture.version.branch.nonarch
        vsn = self.fixture.version.nonarch
        rev_dir = archive_dir / cat / brn / vsn / 'patch-1'
        return DirName(rev_dir)

    def test_revision_files_data(self):
        """tla-format Revision files data access."""
        RevisionFiles.test_revision_files_data(self)


class PristineTrees(PopulatedArchive):

    tests = []

    def iter_pristines_one(self):
        """iter_pristines works"""
        result = list(self.tree.iter_pristines())
        expected = [self.params.version['patch-1']]
        self.assertEqual(expected, result)
    tests.append('iter_pristines_one')

    def add_pristine(self):
        """add_pristine works"""
        version = self.params.version
        self.tree.add_pristine(version['base-0'])
        result = list(self.tree.iter_pristines())
        expected = [version['base-0'], version['patch-1']]
        self.assertEqual(expected, result)
    tests.append('add_pristine')

    def find_pristine(self):
        """find_pristine works"""
        version = self.params.version
        revision = version['patch-1']
        result = self.tree.find_pristine(revision)
        expected = (self.tree / '{arch}' / '++pristine-trees' / 'unlocked' /
                    version.category.nonarch /
                    version.branch.nonarch / version.nonarch /
                    self.params.archive.name / revision.nonarch)
        self.assertEqual(expected, result)
        self.assertEqual(arch.ArchSourceTree, type(result))
    tests.append('find_pristine')

    def find_pristine_missing(self):
        """find_pristine raises NoPristineFoundError"""
        revision = self.params.version['base-0']
        self.assert_(revision not in self.tree.iter_pristines())
        def thunk(): return self.tree.find_pristine(revision)
        self.assertRaises(arch.errors.NoPristineFoundError, thunk)
    tests.append('find_pristine_missing')


class Changeset(PopulatedArchive):

    tests = []

    def setUp(self):
        PopulatedArchive.setUp(self)
        self.base0 = self.params.version['base-0']
        self.patch1 = self.params.version['patch-1']
        self.base0_tree = self.base0.get(self.params.arch_dir/'base0')
        self.patch1_tree = self.patch1.get(self.params.arch_dir/'patch1')
        self.changeset_name = self.params.arch_dir/'cset'

    def changeset_does_nothing(self):
        """Changeset.does_nothing works."""
        cset = arch.delta(
            self.patch1_tree, self.patch1_tree, self.changeset_name)
        self.assertEqual(True, cset.does_nothing)
        shutil.rmtree(cset)
        cset = arch.delta(
            self.base0_tree, self.patch1_tree, self.changeset_name)
        self.assertEqual(False, cset.does_nothing)
    tests.append('changeset_does_nothing')


class Delta(Changeset):

    tests = []

    def _patchlog_name(self, rvsn):
        return '{arch}/%s/%s/%s/%s/patch-log/%s' % \
               (rvsn.category.nonarch, rvsn.branch.nonarch,
                rvsn.version.nonarch, rvsn.archive.name, rvsn.patchlevel)

    def _delta_helper(self, orig, mod):
        name = self._patchlog_name(self.patch1)
        expected_created = [('A_./'+name, name)]
        cset = arch.delta(orig, mod, self.changeset_name)
        created = list(cset.iter_created_files(True))
        self.assertEqual(expected_created, created)

    def delta_tree_tree(self):
        """delta(tree, tree) works."""
        self._delta_helper(self.base0_tree, self.patch1_tree)
    tests.append('delta_tree_tree')

    def delta_tree_revision(self):
        """delta(tree, revision) works."""
        self._delta_helper(self.base0_tree, self.patch1)
    tests.append('delta_tree_revision')

    def delta_revision_tree(self):
        """delta(revision, tree) works."""
        self._delta_helper(self.base0, self.patch1_tree)
    tests.append('delta_revision_tree')

    def delta_revision_revision(self):
        """delta(revision, revision) works."""
        self._delta_helper(self.base0, self.patch1)
    tests.append('delta_revision_revision')

    def iter_delta(self):
        """iter_delta works."""
        orig, mod = self.base0, self.patch1
        name = self._patchlog_name(self.patch1)
        idelta = arch.iter_delta(orig, mod, self.changeset_name)
        def get_changeset(): return idelta.changeset
        self.assertRaises(AttributeError, get_changeset)
        lines = [ L for L in idelta if not isinstance(L, arch.Chatter)]
        self.assertEqual(1, len(lines))
        self.assertEqual(arch.FileAddition, type(lines[0]))
        self.assertEqual(name, lines[0].name)
        cset = get_changeset()
        created = list(cset.iter_created_files(True))
        expected_created = [('A_./'+name, name)]
        self.assertEqual(expected_created, created)
    tests.append('iter_delta')


class HasChanges(TestCase):

    def extraSetup(self):
        self.params.set_my_id()
        self.params.create_archive()
        self.params.create_working_tree(self.params.version)

    def setUp(self):
        TestCase.setUp(self)
        self.tree = self.params.working_tree
        self.tree.import_()

    tests = []

    def has_changes(self):
        """WorkingTree.has_changes works"""
        self.assertEqual(False, self.tree.has_changes())
        open(self.tree/'foo', 'w').write("hello\n")
        self.tree.add_tag('foo')
        self.assertEqual(True, self.tree.has_changes())
    tests.append('has_changes')


class TreeChanges(Delta):
    tests = []

    def changes(self):
        """WorkingTree.changes works"""
        cset = self.tree.changes(None, self.changeset_name)
        self.assertEqual(True, cset.does_nothing)
        shutil.rmtree(cset)
        cset = self.tree.changes(self.base0, self.changeset_name)
        log_name = self._patchlog_name(self.patch1)
        expected = [('A_./%s' % log_name, log_name)]
        newfiles = list(cset.iter_created_files(True))
        self.assertEqual(expected, newfiles)
        shutil.rmtree(cset)
        open(self.tree/'foo', 'w').write("hello\n")
        self.tree.add_tag('foo')
        cset = self.tree.changes(None, self.changeset_name)
        expected = ['.arch-ids/foo.id', 'foo']
        newfiles = [C[1] for C in cset.iter_created_files(True)]
        newfiles.sort()
        self.assertEqual(expected, newfiles)
    tests.append('changes')


class FileHistory(TestCase):

    file_history = [
        { '%add': ['foo'], 'foo': "hello world\n" },
        { 'foo': "Hello, World!\n" } ]

    def extraSetup(self):
        self.params.set_my_id()
        self.params.create_archive()
        self.params.create_working_tree(self.params.version)

    def setUp(self):
        TestCase.setUp(self)
        self.tree = self.params.working_tree
        self.params.make_history(self.tree, self.file_history)


class FileFind(FileHistory):

    tests = []

    def file_find_pristine(self):
        """file_find works with pristines"""
        revision = self.params.version['patch-1']
        path = self.tree.file_find('foo', revision)
        self.assertEqual(PathName, type(path))
        pristine = self.tree.find_pristine(revision)
        expected = pristine/'foo'
        self.assertEqual(expected, path)
        result = open(path).read()
        expected = self.file_history[1]['foo']
        self.assertEqual(expected, result)
    tests.append('file_find_pristine')

    def file_find_revlib(self):
        """file_find works with the revision library"""
        revlib_dir = self.params.arch_dir/'revlib'
        os.mkdir(revlib_dir)
        arch.register_revision_library(revlib_dir)
        revision = self.params.version['patch-1']
        shutil.rmtree(self.tree/'{arch}'/'++pristine-trees')
        self.assertEqual([], list(self.tree.iter_pristines()))
        revision.library_add()
        path = self.tree.file_find('foo', revision)
        self.assertEqual(PathName, type(path))
        pristine = revision.library_find()
        expected = pristine/'foo'
        self.assertEqual(expected, path)
        result = open(path).read()
        expected = self.file_history[1]['foo']
        self.assertEqual(expected, result)
    tests.append('file_find_revlib')

    def file_find_pristine_missing(self):
        """file_find works with missing pristine"""
        revision = self.params.version['base-0']
        path = self.tree.file_find('foo', revision)
        self.assertEqual(PathName, type(path))
        pristine = self.tree.find_pristine(revision)
        expected = pristine/'foo'
        self.assertEqual(expected, path)
        result = open(path).read()
        expected = self.file_history[0]['foo']
        self.assertEqual(expected, result)
    tests.append('file_find_pristine_missing')

    def _file_find_raises_helper(self, flavour):
        revision = self.params.version['patch-1']
        name = {'source': 'bar', 'junk': ',bar'}[flavour]
        open(self.tree/name, 'w').write('Yadda!\n')
        if flavour == 'source': self.tree.add_tag(name)
        def thunk(): return self.tree.file_find(name, revision)
        self.assertRaises(arch.errors.MissingFileError, thunk)

    def file_find_new_raises(self):
        """file_find raises MissingFileError for new source files"""
        self._file_find_raises_helper('source')
    tests.append('file_find_new_raises')

    def file_find_junk_raises(self):
        """file_find raises MissingFileError for junk files"""
        self._file_find_raises_helper('junk')
    tests.append('file_find_junk_raises')


class ApplyChangeset(FileHistory, Delta):

    tests = []

    def setUp(self):
        FileHistory.setUp(self)
        shutil.rmtree(self.params.working_dir)
        self.tree = None
        self.base0 = self.params.version['base-0']
        self.patch1 = self.params.version['patch-1']
        self.changeset_name = self.params.arch_dir/'cset'
        self.cset = arch.delta(self.base0, self.patch1, self.changeset_name)

    def iter_apply(self):
        """Changeset.iter_apply works"""
        tree = self.base0.get(self.params.working_dir)
        lines = list(self.cset.iter_apply(tree))
        self.assertEqual(2, len(lines))
        self.assertEqual(arch.FileAddition, type(lines[0]))
        self.assertEqual(self._patchlog_name(self.patch1), lines[0].name)
        self.assertEqual(arch.FileModification, type(lines[1]))
        self.assertEqual('foo', lines[1].name)
        self.assertEqual(self.patch1, tree.tree_revision)
        self.assert_(not tree.has_changes())
    tests.append('iter_apply')

    def iter_apply_conflict(self):
        """Changeset.iter_apply works with conflicts."""
        tree = self.patch1.get(self.params.working_dir)
        lines = list(self.cset.iter_apply(tree))
        self.assertEqual(2, len(lines))
        self.assertEqual(arch.FileAddition, type(lines[0]))
        self.assertEqual(self._patchlog_name(self.patch1), lines[0].name)
        self.assertEqual(arch.PatchConflict, type(lines[1]))
        self.assertEqual('foo', lines[1].name)
        self.assertEqual(self.patch1, tree.tree_revision)
        self.assert_(not tree.has_changes())
    tests.append('iter_apply_conflict')

    def iter_apply_reverse(self):
        """Changeset.iter_apply reverse=True works."""
        tree = self.patch1.get(self.params.working_dir)
        lines = list(self.cset.iter_apply(tree, reverse=True))
        self.assertEqual(arch.FileDeletion, type(lines[0]))
        self.assertEqual(self._patchlog_name(self.patch1), lines[0].name)
        self.assertEqual(arch.FileModification, type(lines[1]))
        self.assertEqual('foo', lines[1].name)
        self.assertEqual(self.base0, tree.tree_revision)
        self.assert_(not tree.has_changes())
    tests.append('iter_apply_reverse')

    def simple_apply(self):
        """Changeset.apply works"""
        tree = self.base0.get(self.params.working_dir)
        self.assertEqual(open(tree/'foo').read(), "hello world\n")
        self.cset.apply(tree)
        self.assertEqual(self.patch1, tree.tree_revision)
        self.assertEqual(open(tree/'foo').read(), "Hello, World!\n")
    tests.append('simple_apply')

    def simple_apply_conflict(self):
        """Changeset.apply raises errors.ChangesetConflict on conflict."""
        tree = self.patch1.get(self.params.working_dir)
        self.assertRaises(errors.ChangesetConflict, self.cset.apply, tree)
    tests.append('simple_apply_conflict')

    def simple_apply_reverse(self):
        """Changeset.apply reverse=True works."""
        tree = self.patch1.get(self.params.working_dir)
        self.assertEqual(open(tree/'foo').read(), "Hello, World!\n")
        self.cset.apply(tree, reverse=True)
        self.assertEqual(self.base0, tree.tree_revision)
        self.assertEqual(open(tree/'foo').read(), "hello world\n")
    tests.append('simple_apply_reverse')


class ApplyRevision(ApplyChangeset):

    tests = []

    def simple_apply(self):
        """Revision.apply works"""
        tree = self.base0.get(self.params.working_dir)
        self.assertEqual(open(tree/'foo').read(), "hello world\n")
        self.patch1.apply(tree)
        self.assertEqual(self.patch1, tree.tree_revision)
        self.assertEqual(open(tree/'foo').read(), "Hello, World!\n")
    tests.append('simple_apply')

    def simple_apply_conflict(self):
        """Revision.apply raises errors.ChangesetConflict on conflict."""
        tree = self.patch1.get(self.params.working_dir)
        self.assertRaises(errors.ChangesetConflict, self.patch1.apply, tree)
    tests.append('simple_apply_conflict')

    def simple_apply_reverse(self):
        """Revision.apply reverse=True works."""
        tree = self.patch1.get(self.params.working_dir)
        self.assertEqual(open(tree/'foo').read(), "Hello, World!\n")
        self.patch1.apply(tree, reverse=True)
        self.assertEqual(self.base0, tree.tree_revision)
        self.assertEqual(open(tree/'foo').read(), "hello world\n")
    tests.append('simple_apply_reverse')


framework.register(__name__)
