# arch-tag: fd18ae24-e724-44c2-92fe-627495793f4d
# Copyright (C) 2005 Canonical Ltd.
#               Author: David Allouche <david@canonical.com>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""Support archive locations."""

import urllib

from pybaz import errors

__all__ = [
    'ArchiveLocation',
    'ArchiveLocationParams',
    ]


def _factory():
    import pybaz
    return pybaz.factory

def _backend():
    import pybaz
    return pybaz.backend

def is_valid_url(url):
    """Is `url` a valid url for an archive location?

    :type url: str
    :rtype: bool
    """
    # FIXME: this assumes unix file names -- David Allouche 2005-07-08
    if url.startswith('/'):
        return True # absolute path name
    if '://' not in url:
        return False # not a url
    scheme = url.split('://')[0]
    if scheme not in ('ftp', 'http', 'sftp', 'file'):
        return False # malformed url or unspported scheme
    if scheme == 'file' and not url.startswith('file:///'):
        return False # only support same-host file urls
    return True


class ArchiveLocation(object):
    """A location identified by an url and containing a Bazaar archive."""

    def __init__(self, url):
        error_message = ("ArchiveLocation parameter should be a str containing"
                         " an http, ftp, sftp url or an absolute file path,"
                         " but was: %r")
        if not isinstance(url, str):
            raise TypeError(error_message % (url,))
        if not is_valid_url(url):
            raise ValueError(error_message % (url,))
        self._url = str(url)
        self._archive = None

    def _get_url(self):
        return self._url

    url = property(_get_url, doc="""
    Url of this location.

    :type: str
    """)

    def __eq__(self, other):
        """Compare equal to instances of ArchiveLocation with the same url."""
        if not _factory().isArchiveLocation(other):
            return False
        return self.url == other.url

    def __ne__(self, other):
        """Logical complement of __eq__."""
        return not self == other

    def __repr__(self):
        # NOTE: relies on the __module__ of this class to be overriden
        return '%s.%s(%r)' % (
            self.__class__.__module__, self.__class__.__name__, self.url)

    def create_master(self, archive, params):
        """Create a new master archive at this location.

        :precondition: not self.is_registered()
            and not archive.is_registered()
            and <url does not exist and is writable>
        :postcondition: archive.is_registered() and archive.location == self
            and <url exists>

        :type archive: Archive
        :type params: ArchiveLocationParams
        """
        self._check_make_archive_params(archive, params)
        args = ['make-archive']
        if params.signed:
            args.append('--signed')
        if params.listing:
            args.append('--listing')
        if params._tla:
            args.append('--tla')
        args.extend((archive.name, self.url))
        _backend().null_cmd(args)

    def create_mirror(self, archive, params):
        """Create a new archive mirror at this location.

        :precondition: not self.is_registered()
            and <url does not exist and is writable>
        :postcondition: self.is_registered()
            and <url exists>

        :type archive: Archive
        :type params: ArchiveLocationParams
        """
        self._check_make_archive_params(archive, params)
        args = ['make-archive', '--mirror', archive.name]
        if params.signed:
            args.append('--signed')
        if params.listing:
            args.append('--listing')
        if params._tla:
            args.append('--tla')
        args.append(self.url)
        _backend().null_cmd(args)

    def _check_make_archive_params(self, archive, params):
        """Check sanity of create_master and create_mirror params."""
        if not _factory().isArchive(archive):
            raise TypeError("ArchiveLocation.create_master archive argument"
                            " must be an archive, but was: %r" % archive)
        if not isinstance(params, ArchiveLocationParams):
            raise TypeError(
                "ArchiveLocation.create_master params argument"
                " must be an ArchiveLocationParams, but was %r" % archive)
        # TODO: preconditions are not checked yet that would require spawning a
        # couple of baz commands. Breaking preconditions will currently
        # translate into an ExecProblem -- David Allouche 2005-07-09

    def is_registered(self):
        """Is this location registered?

        :rtype: bool
        """
        for line in _backend().sequence_cmd(['archives', '--all-locations']):
            if not line.startswith('    '):
                continue
            url = urllib.unquote(line[4:])
            if self.url == url:
                return True
        return False

    def unregister(self):
        """Unregister this location:

        :precondition: self.is_registered()
        :poscondition: not self.is_registered()
        :raises errors.LocationNotRegistered: this location was not registered.
        """
        if not self.is_registered():
            # costly precondition check, but unregister is not a frequently
            # used command.
            raise errors.LocationNotRegistered(self.url)
        _backend().null_cmd(('register-archive', '--delete', self.url))

    def register(self):
        """Register this location.

        :precondition: not self.is_registered()
        :postcondition: self.is_registered()
        :raises errors.LocationAlreadyRegistered: this location was already
            registered.
        """
        if self.is_registered():
            # costly precondition check, but register is not a frequently used
            # command.
            raise errors.LocationAlreadyRegistered(self.url)
        _backend().null_cmd(['register-archive', self.url])

    def meta_info(self, key):
        """Read a meta-info from this location.

        :precondition: self.is_registered()
        :param key: name of the meta-info to read.
        :type key: str
        :raises errors.MetaInfoError: this location has no such meta-info.
        :raises errors.LocationNotRegistered: this location is not registered.

        :bug: will raise `errors.MetaInfoError` when the location could not be
            accessed, because baz gives us exit status 1 for ''meta-info not
            present'' and ''could not access location''.
        """
        error_message = ("ArchiveLocation.meta_info parameter should be a"
                         " non-empty str, but was: %r" % key)
        if not isinstance(key, str):
            raise TypeError(error_message)
        if key == '':
            raise ValueError(error_message)
        if not self.is_registered():
            # expensive precondition check
            raise errors.LocationNotRegistered(self.url)
        # XXX: baz does not allow us to distinguish between "location could not
        # be accessed" and "meta-info was not present" from the exit status. So
        # we will raise MetaInfoError inappropriately when the archive could
        # not be accessed. -- David Allouche 2005-07-12
        args = ['archive-meta-info', self.url + '/' + key]
        status, output = _backend().status_one_cmd(args, expected=(0, 1))
        if status != 0:
            raise errors.MetaInfoError(self.url, key)
        return output

    def _meta_info_present(self, key):
        """Is the specified meta info present at this location?

        :precondition: self.is_registered()
        :param key: name of the meta-info to look for
        :type key: str
        :raises errors.LocationNotRegistered: this location is not registered.

        :bug: because of the `meta_info` bug, that is completely unreliable,
            that's why it's a protected method. Still it's useful for testing.
        """
        try:
            self.meta_info(key)
        except errors.MetaInfoError:
            return False
        else:
            return True

    def archive(self):
        """Archive that is associated to this location.

        That's a convenience method based on meta_info() that memoises its
        result.

        :rtype: `Archive`
        """
        if self._archive is None:
            name = self.meta_info('name')
            self._archive = _factory().Archive(name)
        return self._archive

    def _version_string(self):
        """Version string of the archive.

        Contents of the ``.archive-version`` file at the root of the archive.

        :rtype: str
        :bug: only works with schemes supported by ``urllib.urlopen``.
        """
        url = self.url
        version_url = self.url + '/.archive-version'
        version_file = urllib.urlopen(version_url)
        try:
            version = version_file.read()
        finally:
            version_file.close()
        return version.rstrip()

    def make_mirrorer(self, target):
        """Create a mirrorer to mirror from this location to the target.

        :param target: specific location the `MirrorMethod` will mirror to.
        :type target: `ArchiveLocation`
        :rtype: `MirrorMethod`

        :raises error.LocationNotRegistered: at least one of self and target is
            not a registered location.
        :raises errors.MirrorLocationMismatch: self and target are registered
            locations for different archives.
        """
        if not _factory().isArchiveLocation(target):
            raise TypeError("ArchiveLocation.make_mirrorer argument should be"
                            " and ArchiveLocation, but was: %r" % target)
        if self.url == target.url:
            raise ValueError("ArchiveLocation.make_mirrorer argument was an"
                             " ArchiveLocation with the same url as self.")
        if not self.is_registered():
            raise errors.LocationNotRegistered(self)
        if not target.is_registered():
            raise errors.LocationNotRegistered(target)
        if self.archive() != target.archive():
            raise errors.MirrorLocationMismatch(self, target)
        mirrorer = MirrorMethod(self, target)
        return mirrorer


class ArchiveLocationParams(object):
    """Parameter Object used for creating archives masters and mirrors.

    :ivar signed: create signed location?
    :type signed: bool
    :ivar listing: create location with listings for http access.
    :type listing: bool
    """

    def __init__(self):
        self.signed = False
        self.listing = False
        self._tla = False

    def tla_format(self):
        """Set the parameter object to create a tla-compatible archive."""
        self._tla = True


class MirrorMethod(object):
    """Method Object used for mirroring operations.

    This class should never be instantiated directly. Instead, it is created by
    factory methods like `ArchiveLocation.make_mirrorer`.
    """

    def __init__(self, source, target):
        """Do not instanciate this class directly."""
        self._source = source
        self._target = target

    def _limit_type_is_bad(self, limit):
        """Internal helper for sanity checking of limit argument."""
        if _factory().isVersion(limit):
            return False
        if _factory().isRevision(limit):
            return False
        return True

    def _limit_value_is_bad(self, limit):
        """Internal helper for sanity checking of limit argument."""
        return limit.archive != self._source.archive()

    def mirror(self, limit=None):
        """Perform mirroring.

        :param limit: if provided, limit the mirroring to the specified
            version or revision.
        :type limit: `pybaz.Version`, `pybaz.Revision`, None
        """
        assert self._source is not None
        assert self._target is not None
        args = ['archive-mirror', self._source.url, self._target.url]
        if limit is not None:
            message = ("MirrorMethod.mirror parameter should be a Version or"
                       " Revision in %s, but was: %r")
            archive = self._source.archive()
            if self._limit_type_is_bad(limit):
                raise TypeError(message % (archive.name, limit))
            if self._limit_value_is_bad(limit):
                raise ValueError(message % (archive.name, limit))
            args.append(limit.nonarch)
        _backend().null_cmd(args)
