//
// System.Configuration.SectionInfo.cs
//
// Authors:
//	Lluis Sanchez (lluis@novell.com)
//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
// 
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//
// Copyright (C) 2004 Novell, Inc (http://www.novell.com)
//
#if NET_2_0
using System;
using System.Collections;
using System.Collections.Specialized;
using System.Xml;
using System.IO;

namespace System.Configuration
{
	internal class SectionInfo: ConfigInfo
	{
		bool allowLocation = true;
		ConfigurationAllowDefinition allowDefinition = ConfigurationAllowDefinition.Everywhere;
		ConfigurationAllowExeDefinition allowExeDefinition = ConfigurationAllowExeDefinition.MachineToApplication;

		public SectionInfo ()
		{
		}
		
		public SectionInfo (string sectionName, string typeName,
				    bool allowLocation, ConfigurationAllowDefinition allowDefinition, ConfigurationAllowExeDefinition allowExeDefinition)
		{
			Name = sectionName;
			TypeName = typeName;
			this.allowLocation = allowLocation;
			this.allowDefinition = allowDefinition;
			this.allowExeDefinition = allowExeDefinition;
		}
		
		public override object CreateInstance ()
		{
			object ob = base.CreateInstance ();
			ConfigurationSection sec = ob as ConfigurationSection;
			if (sec != null) {
				sec.SectionInformation.AllowLocation = allowLocation;
				sec.SectionInformation.AllowDefinition = allowDefinition;
				sec.SectionInformation.AllowExeDefinition = allowExeDefinition;
			}
			return ob;
		}
		
		public override bool HasDataContent (Configuration config)
		{
			return config.GetSectionInstance (this, false) != null || config.GetSectionXml (this) != null;
		}
		
		public override bool HasConfigContent (Configuration cfg)
		{
			return StreamName == cfg.FileName;
		}

		public override void ReadConfig (Configuration cfg, string streamName, XmlTextReader reader)
		{
			StreamName = streamName;
			ConfigHost = cfg.ConfigHost;

			while (reader.MoveToNextAttribute ()) {
				switch (reader.Name)
				{
					case "allowLocation":
						string allowLoc = reader.Value;
						allowLocation = (allowLoc == "true");
						if (!allowLocation && allowLoc != "false")
							ThrowException ("Invalid attribute value", reader);
						break;
	
					case "allowDefinition":
						string allowDef = reader.Value;
						try {
							allowDefinition = (ConfigurationAllowDefinition) Enum.Parse (
									   typeof (ConfigurationAllowDefinition), allowDef);
						} catch {
							ThrowException ("Invalid attribute value", reader);
						}
						break;
	
					case "allowExeDefinition":
						string allowExeDef = reader.Value;
						try {
							allowExeDefinition = (ConfigurationAllowExeDefinition) Enum.Parse (
									   typeof (ConfigurationAllowExeDefinition), allowExeDef);
						} catch {
							ThrowException ("Invalid attribute value", reader);
						}
						break;
	
					case "type":
						TypeName = reader.Value;
						break;
					
					case "name":
						Name = reader.Value;
						if (Name == "location")
							ThrowException ("location is a reserved section name", reader);
						break;
						
					default:
						ThrowException ("Unrecognized attribute.", reader);
						break;
				}
			}

			if (Name == null || TypeName == null)
				ThrowException ("Required attribute missing", reader);

			reader.MoveToElement();
			reader.Skip ();
		}
		
		public override void WriteConfig (Configuration cfg, XmlWriter writer, ConfigurationSaveMode mode)
		{
			writer.WriteStartElement ("section");
			writer.WriteAttributeString ("name", Name);
			writer.WriteAttributeString ("type", TypeName);
			if (!allowLocation)
				writer.WriteAttributeString ("allowLocation", "false");
			if (allowDefinition != ConfigurationAllowDefinition.Everywhere)
				writer.WriteAttributeString ("allowDefinition", allowDefinition.ToString ());
			if (allowExeDefinition != ConfigurationAllowExeDefinition.MachineToApplication)
				writer.WriteAttributeString ("allowExeDefinition", allowExeDefinition.ToString ());
			writer.WriteEndElement ();
		}
		
		public override void ReadData (Configuration config, XmlTextReader reader, bool overrideAllowed)
		{
			if (!config.HasFile && !allowLocation)
				throw new ConfigurationErrorsException ("The configuration section <" + Name + "> cannot be defined inside a <location> element.", reader); 
			if (!config.ConfigHost.IsDefinitionAllowed (config.ConfigPath, allowDefinition, allowExeDefinition)) {
				object ctx = allowExeDefinition != ConfigurationAllowExeDefinition.MachineToApplication ? (object) allowExeDefinition : (object) allowDefinition;
				throw new ConfigurationErrorsException ("The section <" + Name + "> can't be defined in this configuration file (the allowed definition context is '" + ctx + "').", reader);
			}
			if (config.GetSectionXml (this) != null)
				ThrowException ("The section <" + Name + "> is defined more than once in the same configuration file.", reader);
			config.SetSectionXml (this, reader.ReadOuterXml ());
		}
		
		public override void WriteData (Configuration config, XmlWriter writer, ConfigurationSaveMode mode)
		{
			string xml;
			
			ConfigurationSection section = config.GetSectionInstance (this, false);
			if (section != null) {
				ConfigurationSection parentSection = config.Parent != null ? config.Parent.GetSectionInstance (this, false) : null;
				xml = section.SerializeSection (parentSection, Name, mode);
			}
			else {
				xml = config.GetSectionXml (this);
			}
			
			if (xml != null) {
				writer.WriteRaw (xml);
/*				XmlTextReader tr = new XmlTextReader (new StringReader (xml));
				writer.WriteNode (tr, true);
				tr.Close ();*/
			}
		}
	}
}

#endif
