/**
 * ================================================
 * LibLoader : a free Java resource loading library
 * ================================================
 *
 * Project Info:  http://reporting.pentaho.org/libloader/
 *
 * (C) Copyright 2006-2008, by Pentaho Corporation and Contributors.
 *
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation;
 * either version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * [Java is a trademark or registered trademark of Sun Microsystems, Inc.
 * in the United States and other countries.]
 *
 *
 * ------------
 * FileResourceData.java
 * ------------
 */
package org.pentaho.reporting.libraries.resourceloader.loader.file;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;

import org.pentaho.reporting.libraries.resourceloader.ResourceData;
import org.pentaho.reporting.libraries.resourceloader.ResourceKey;
import org.pentaho.reporting.libraries.resourceloader.ResourceLoadingException;
import org.pentaho.reporting.libraries.resourceloader.ResourceManager;
import org.pentaho.reporting.libraries.resourceloader.loader.AbstractResourceData;

/**
 * A simple file reader. This class, as all core implementation, does not hold
 * any references to the data read from the file. Caching is left to the cache
 * provider.
 *
 * @author Thomas Morgner
 */
public class FileResourceData extends AbstractResourceData
{
  private ResourceKey key;
  private File file;
  private static final long serialVersionUID = -5719048997437795736L;

  public FileResourceData(final ResourceKey key) throws ResourceLoadingException
  {
    if (key == null)
    {
      throw new NullPointerException();
    }
    final File file = (File) key.getIdentifier();
    if (file.isFile() == false)
    {
      throw new ResourceLoadingException
              ("File-handle given does not point to a regular file.");
    }
    if (file.canRead() == false)
    {
      throw new ResourceLoadingException
              ("File '" + file + "' is not readable.");
    }
    this.key = key;
    this.file = file;
  }

  public InputStream getResourceAsStream(final ResourceManager caller) throws ResourceLoadingException
  {
    if (caller == null)
    {
      throw new NullPointerException();
    }
    try
    {
      return new BufferedInputStream(new FileInputStream(file));
    }
    catch (FileNotFoundException e)
    {
      throw new ResourceLoadingException("Unable to open Stream: ", e);
    }
  }

  public Object getAttribute(final String attrkey)
  {
    if (attrkey == null)
    {
      throw new NullPointerException();
    }
    if (attrkey.equals(ResourceData.FILENAME))
    {
      return file.getName();
    }
    if (attrkey.equals(ResourceData.CONTENT_LENGTH))
    {
      return new Long(file.length());
    }
    return null;
  }

  public long getVersion(final ResourceManager caller)
          throws ResourceLoadingException
  {
    if (caller == null)
    {
      throw new NullPointerException();
    }
    return file.lastModified();
  }

  public ResourceKey getKey()
  {
    return key;
  }
}
