/*
 *
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/*
 * NAME
 *	setpriority04.c
 *
 * DESCRIPTION
 *	setpriority04 - test for an expected failure by using an invalid
 *			process id
 *
 * ALGORITHM
 *	loop if that option was specified
 *	issue the system call with an invalid process id
 *	check the errno value
 *	  issue a PASS message if we get ESRCH - errno 3
 *	otherwise, the tests fails
 *	  issue a FAIL message
 *	  break any remaining tests
 *	  call cleanup
 *
 * USAGE:  <for command-line>
 *  setpriority04 [-c n] [-e] [-i n] [-I x] [-p x] [-t]
 *     where,  -c n : Run n copies concurrently.
 *             -e   : Turn on errno logging.
 *	       -i n : Execute test n times.
 *	       -I x : Execute test for x seconds.
 *	       -P x : Pause for x seconds between iterations.
 *	       -t   : Turn on syscall timing.
 *
 * HISTORY
 *	03/2001 - Written by Wayne Boyer
 *
 * RESTRICTIONS
 *	none
 */

#include "test.h"
#include "usctest.h"

#include <errno.h>
#include <sys/time.h>
#include <sys/resource.h>

/*
 *  * When attempting to build on SUSE 10, the make fails trying to compile
 *   * because CONFIG_BASE_SMALL is undefined.
 *    */
#ifndef CONFIG_BASE_SMALL
#define CONFIG_BASE_SMALL 0
#endif

/*
 * See the Makefile for comments about the following preprocessor code.
 */
#if !defined (__powerpc__) && !defined(__s390__) && !defined (__s390x__) && !defined (__i386__) 
#ifndef _LTP_TASKS_H
 #include <linux/threads.h>     /* for PID_MAX value - new */
 #else
 #include <linux/tasks.h>       /* for PID_MAX value - old */
 #endif
#endif
/*
 * This is a workaround for ppc64 kernels that do not have PID_MAX defined.
 */
#ifndef PID_MAX
#define PID_MAX 0x8000
#endif


void cleanup(void);
void setup(void);

char *TCID= "setpriority04";
int TST_TOTAL = 1;
extern int Tst_count;

int exp_enos[] = {ESRCH, 0};

int main(int ac, char **av)
{
	int lc;				/* loop counter */
	char *msg;			/* message returned from parse_opts */
	int new_val = 2;

	/* parse standard options */
	if ((msg = parse_opts(ac, av, (option_t *)NULL, NULL)) != (char *)NULL){
		tst_brkm(TBROK, tst_exit, "OPTION PARSING ERROR - %s", msg);
	}

	setup();			/* global setup */

	/* The following loop checks looping state if -i option given */

	for (lc = 0; TEST_LOOPING(lc); lc++) {
		/* reset Tst_count in case we are looping */
		Tst_count = 0;

		/*
		 * Try to access an invalid process.
		 * This should give an ESRCH error.
		 */

		/* call the system call with the TEST() macro */
#ifdef PID_MAX_DEFAULT
		TEST(setpriority(PRIO_PROCESS, PID_MAX_DEFAULT + 1, new_val));
#elif defined(PID_MAX)
		TEST(setpriority(PRIO_PROCESS, PID_MAX + 1, new_val));
#endif
	
	
		if (TEST_RETURN == 0) {
	                tst_resm(TFAIL, "call failed to produce expected error "
				 "- errno = %d - %s", TEST_ERRNO,
				 strerror(TEST_ERRNO));
			continue;
		}
	
		TEST_ERROR_LOG(TEST_ERRNO);

		switch (TEST_ERRNO) {
		case ESRCH:
			tst_resm(TPASS, "expected failure - errno = %d - %s",
				 TEST_ERRNO, strerror(TEST_ERRNO));
			break;
		default:
			tst_resm(TFAIL, "call failed to produce expected error "
				 "- errno = %d - %s", TEST_ERRNO,
				 strerror(TEST_ERRNO));
		}
	}

	cleanup();

	/*NOTREACHED*/

  return(0);

}

/*
 * setup() - performs all the ONE TIME setup for this test.
 */
void
setup(void)
{
	/* capture signals */
	tst_sig(NOFORK, DEF_HANDLER, cleanup);

	/* Set up the expected error numbers for the -e option */
	TEST_EXP_ENOS(exp_enos);

	/* Pause if that option was specified */
	TEST_PAUSE;
}

/*
 * cleanup() - performs all the ONE TIME cleanup for this test at completion
 * 	       or premature exit.
 */
void
cleanup(void)
{
	/*
	 * print timing stats if that option was specified.
	 * print errno log if that option was specified.
	 */
	TEST_CLEANUP;

	/* exit with return code appropriate for results */
	tst_exit();
}
