/***************************************************************************
 *   Copyright (C) 2005 - 2006 by Christian Muehlhaeuser, Last.fm Ltd.     *
 *   chris@last.fm                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02111-1307, USA.          *
 ***************************************************************************/

#include "coverloader.h"

#include <QDebug>
#include <QFile>
#include <QFileInfo>
#include <QHttp>
#include <QDir>

CoverLoader::CoverLoader( QObject *parent, QString cachePath ) : QObject( parent ),
	m_transport( new QHttp( this ) ),
    m_cachePath( cachePath )

{
    if ( !QDir( cachePath ).exists() )
        QDir().mkpath( cachePath );

    connect( m_transport, SIGNAL( done( bool ) ),
	         this, SLOT( slotRequestFinished( bool ) ) );
}


void CoverLoader::setHost( const QString &host )
{
	m_host = host;
	m_transport->setHost( host );
}


void CoverLoader::setProxy( const QString &host, int port,
                            const QString &user, const QString &password )
{
	m_transport->setProxy( host, port, user, password );
}


void CoverLoader::get( const QString &path, QIODevice *to )
{
	m_cacheKey = m_host + path;
	m_cacheKey.replace( '/', '_' );

	if ( to && ( !to->isReadable() && !to->isWritable() ) ) {
		to->close();
		to->open( QIODevice::ReadWrite );
	}

	m_buffer = to;

// 	qDebug() << "Attempting to fetch cover " << m_cacheKey;
// 	qDebug() << "Have cached copy for " << m_cacheKey << "?";
	if ( haveCachedCopy() && to ) {
		QByteArray data = getCachedCopy();
		if ( !data.isEmpty() ) {
// 			qDebug() << "Reusing cached copy for " << m_cacheKey;
			to->write( getCachedCopy() );
			emit requestFinished( false );
			return;
		}
	}
// 	qDebug() << "Didn't find a cached copy for " << m_cacheKey << ", downloading.";
	m_transport->get( path, to );
}


void CoverLoader::slotRequestFinished( bool error )
{
	if ( !error && !haveCachedCopy() && m_buffer->size() > 0 ) {
		m_buffer->reset();
		QByteArray data = m_buffer->readAll();
// 		qDebug() << "Storing copy of " << m_cacheKey << " in cache (" << data.size() << "bytes)";
		putCachedCopy( data );
	}
	emit requestFinished( error );
}


bool CoverLoader::haveCachedCopy() const
{
	return QFile::exists( pathToCachedCopy() ) &&
	       QFileInfo( pathToCachedCopy() ).isReadable();
}


QByteArray CoverLoader::getCachedCopy() const
{
	QFile f( pathToCachedCopy() );
	if ( !f.open( QIODevice::ReadOnly ) ) {
		return QByteArray();
	}
	return f.readAll();
}


void CoverLoader::putCachedCopy( const QByteArray &data )
{
	QFile f( pathToCachedCopy() );
	if ( !f.open( QIODevice::WriteOnly ) ) {
		return;
	}
	f.write( data );
}


QString CoverLoader::pathToCachedCopy() const
{
    return m_cachePath + m_cacheKey;
}

