#!/bin/sh

# This script generates the dummy HFS filesystem used for the PowerPC boot
# blocks. It uses hfsutils (emulators/hfsutils) to generate a template
# filesystem with the relevant interesting files. These are then found by
# grep, and the offsets written to a Makefile snippet.
#
# Because of licensing concerns, and because it is overkill, we do not
# distribute hfsutils as a build tool. If you need to regenerate the HFS
# template (e.g. because the boot block or the CHRP script have grown),
# you must install it from ports.

# $FreeBSD$

HFS_SIZE=1600 			#Size in 512-byte blocks of the produced image

CHRPBOOT_SIZE=2k
BOOT1_SIZE=30k

# Generate 800K HFS image
OUTPUT_FILE=hfs.tmpl

dd if=/dev/zero of=$OUTPUT_FILE bs=512 count=$HFS_SIZE
hformat -l "FreeBSD Bootstrap" $OUTPUT_FILE
hmount $OUTPUT_FILE

# Create and bless a directory for the boot loader
hmkdir ppc
hattrib -b ppc
hcd ppc

# Make two dummy files for the the CHRP boot script and boot1
echo 'Bootinfo START' | dd of=bootinfo.txt.tmp cbs=$CHRPBOOT_SIZE count=1 conv=block
echo 'Boot1 START' | dd of=boot1.elf.tmp cbs=$BOOT1_SIZE count=1 conv=block

hcopy boot1.elf.tmp :boot1.elf
hcopy bootinfo.txt.tmp :bootinfo.txt
hattrib -c chrp -t tbxi bootinfo.txt
humount

rm bootinfo.txt.tmp
rm boot1.elf.tmp

# Locate the offsets of the two fake files
BOOTINFO_OFFSET=$(hd $OUTPUT_FILE | grep 'Bootinfo START' | cut -f 1 -d ' ')
BOOT1_OFFSET=$(hd $OUTPUT_FILE | grep 'Boot1 START' | cut -f 1 -d ' ')

# Convert to numbers of blocks
BOOTINFO_OFFSET=$(echo 0x$BOOTINFO_OFFSET | awk '{printf("%x\n",$1/512);}')
BOOT1_OFFSET=$(echo 0x$BOOT1_OFFSET | awk '{printf("%x\n",$1/512);}')

echo '# This file autogenerated by generate-hfs.sh - DO NOT EDIT' > Makefile.hfs
echo '# $FreeBSD$' >> Makefile.hfs
echo "BOOTINFO_OFFSET=0x$BOOTINFO_OFFSET" >> Makefile.hfs
echo "BOOT1_OFFSET=0x$BOOT1_OFFSET" >> Makefile.hfs

bzip2 $OUTPUT_FILE
echo 'HFS template boot filesystem created by generate-hfs.sh' > $OUTPUT_FILE.bz2.uu
echo 'DO NOT EDIT' >> $OUTPUT_FILE.bz2.uu
echo '$FreeBSD$' >> $OUTPUT_FILE.bz2.uu

uuencode $OUTPUT_FILE.bz2 $OUTPUT_FILE.bz2 >> $OUTPUT_FILE.bz2.uu
rm $OUTPUT_FILE.bz2

