/*************************************************************************
 * 
 * irmp3 - Multimedia Audio Jukebox for Linux
 * http://irmp3.sourceforge.net
 *
 * $Source: /cvsroot/irmp3/irmp3/src/irmp3d/mod_playlistlist.c,v $ -- support for lists of playlists
 * $Id: mod_playlistlist.c,v 1.7 2004/08/10 21:08:32 boucman Exp $
 *
 * Copyright (C) by 
 *
 * Please contact the current maintainer, Jeremy Rosen <jeremy.rosen@enst-bretagne.fr>
 * for information and support regarding irmp3.
 *
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <time.h>
#include <dirent.h>
#include <fnmatch.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "config.h"
#include "irmp3tools.h"
#include "irmp3log.h"
#include "irmp3mod.h"
#include "mod_playlistlist.h"


/*************************************************************************
 * GLOBALS
 */

mod_playlistlist_t *mod_playlistlist_list = NULL;
mod_playlistlist_t *mod_playlistlist_current = NULL;


/*************************************************************************
 * MODULE INFO
 */
mod_t mod_playlistlist = {
	"mod_playlistlist",
	mod_playlistlist_deinit,	// deinit
	NULL,				// reload
	&blank_fd,				// watchfd
	NULL,				// poll
	NULL,				// update
	mod_playlistlist_message,	// message
	NULL,				// SIGCHLD handler
	NULL,
	NULL,				// avoid warning
};


/*************************************************************************
 * CLEAR PLAYLISTLIST
 */
void mod_playlistlist_free (void)
{
	mod_playlistlist_t *p,*n;
	if(mod_playlistlist_list)
	{
		for (p=mod_playlistlist_list; p; p=p->next)
		{
			n=p->next;
			free(p);
		}
	}
	mod_playlistlist_list = NULL;
	mod_playlistlist_current = NULL;
	log_printf(LOG_DEBUG, "mod_playlistlist_free(): cleared the list of playlists\n");
}


/*************************************************************************
 * ADD TO PLAYLISTLIST
 */
int mod_playlistlist_add (char *playlist)
{
	mod_playlistlist_t *lastentry, *newentry;

	// find last playlistlist entry
	lastentry = mod_playlistlist_list;
	while (lastentry && lastentry->next)
		lastentry = lastentry->next;

	// setup new playlistlist entry
	newentry = malloc(sizeof(mod_playlistlist_t));
	if (!newentry)
		return -1;
	strncpy(newentry->playlist, playlist, sizeof(newentry->playlist)-1);
	newentry->playlist[sizeof(newentry->playlist)-1] = 0;
	newentry->next = NULL;
	newentry->prev = lastentry;

	// link new entry into playlistlist chain
	if(lastentry) {
		lastentry->next = newentry;
	}else{
		mod_playlistlist_list = newentry;
	}
	if (!mod_playlistlist_current)
		mod_playlistlist_current = mod_playlistlist_list;
	log_printf(LOG_NOISYDEBUG, "mod_playlistlist_add(): added playlist '%s'\n", newentry->playlist);

	return 0;
}


/*************************************************************************
 * LOAD PLAYLISTLIST
 */
int mod_playlistlist_load (char *name)
{
	char *c, root[512], playlist[512], buf[512];
	int fd, count;

	if (!name || !*name)
		return -1;

	// get root path of playlistlist
	strncpy(root, name, sizeof(root)-1);
	c = strrchr(root, '/');
	if (c)
		*c = 0;
	else
		getcwd(root, sizeof(root)-1);

	// read playlistlist file
	fd = open(name, O_RDONLY);
	if (fd < 0) {
		log_printf(LOG_DEBUG, "mod_playlistlist_load(): unable to open list of playlists '%s'\n", name);
		return -2;
	}
	count = 0;
	while (readline(fd, playlist, sizeof(playlist)) >= 0) {
		trim(playlist);
		if (*playlist != '/') {
			snprintf(buf, sizeof(buf)-1, "%s/%s", root, playlist);
			mod_playlistlist_add(buf);
		} else
			mod_playlistlist_add(playlist);
		count++;
	}
	close(fd);

	log_printf(LOG_DEBUG, "mod_playlistlist_load(): added %d playlists\n", count);

	return count;
}


/*************************************************************************
 * SKIP TO PLAYLIST (relative)
 */
int mod_playlistlist_skip (int offset)
{
	int i, count;

	if (!offset || !mod_playlistlist_current )
		return 0;
	count = 0;

	// skip forward
	if (offset > 0)
		for (i=0; i<offset; i++) {
			if (!(mod_playlistlist_current->next))
				mod_playlistlist_current = mod_playlistlist_list;
			else
				mod_playlistlist_current = mod_playlistlist_current->next;
			count++;
			if (!mod_playlistlist_current)
				break;
		}

	// skip backwards
	else if (offset < 0)
		for (i=0; i<-offset; i++) {
			if (!(mod_playlistlist_current->prev)) {
				mod_playlistlist_current = mod_playlistlist_list;
				while (mod_playlistlist_current && (mod_playlistlist_current->next))
					mod_playlistlist_current = (mod_playlistlist_current->next);
			} else
				mod_playlistlist_current = mod_playlistlist_current->prev;
			count--;
			if (!mod_playlistlist_current)
				break;
		}

	log_printf(LOG_NORMAL, "List of playlists skipped %d playlists\n", count);
	return count;
}



/*************************************************************************
 * RECEIVE MESSAGE
 */
void mod_playlistlist_message (int msgtype, char *rawmsg,const char __attribute__((unused))*sender)
{
	char *c1, *c2, *c3, msg[512];
	int i;

	strncpy(msg,rawmsg,sizeof(msg));	// pad msg with nulls

	// handle input messages
	if (msgtype == MSGTYPE_INPUT) {
		c1 = strtok(msg, " \t");
		if(!c1) return ;
		c2 = c1 ? strtok(NULL, " \t") : NULL;

		// process playlistlist commands
		if (!strcasecmp(c1, "playlistlist")) {
			c2 = strtok(NULL, " \t");
			if(!c2) return ;
			// load playlistlist from file
			if ( !strcasecmp(c2, "load")) {
				c3 = strtok(NULL, "");
				if(!c3) return ;
				mod_playlistlist_free();
				log_printf(LOG_NORMAL, "Loading list of playlists '%s'\n", c3);
				if (mod_playlistlist_load(c3) < 0)
					log_printf(LOG_ERROR, "Unable to load list of playlist!\n");
				else {
					mod_sendmsgf(MSGTYPE_INPUT, "playlist load %s", mod_playlistlist_current->playlist);
				}

			// clear playlistlist
			} else if (!strcasecmp(c2, "clear")) {
				mod_sendmsg(MSGTYPE_INPUT, "stop");
				mod_playlistlist_free();

			// jump/skip within the playlistlist
			} else if ( !strcasecmp(c2, "jump") ) {
				c3 = strtok(NULL, "");
				if(!c3) return ;
				if (*c3!='+' && *c3!='-')
					mod_playlistlist_current = mod_playlistlist_list;
				i = (*c3=='-') ? atoi(c3) : atoi(c3+1);
				mod_playlistlist_skip(i);
				if (mod_playlistlist_current)
					mod_sendmsgf(MSGTYPE_INPUT, "playlist load %s", mod_playlistlist_current->playlist);
				else
					mod_sendmsg(MSGTYPE_INPUT, "stop");
			}
		}
	}
}




/*************************************************************************
 * MODULE DEINIT FUNCTION
 */
void mod_playlistlist_deinit (void)
{
	// clear up playlist
	mod_playlistlist_free();
	log_printf(LOG_DEBUG, "mod_playlistlist_deinit(): deinitialized\n");
}


/*************************************************************************
 * EOF
 */
