/*******************************************************\
* irmp3-ncurses - An ncurses frontend for irmp3 using   *
* the Network Control Module                            *
* (C) 2005 Ross Axe                                     *
*                                                       *
* netio.c - low-level network I/O                       *
\*******************************************************/

#if HAVE_CONFIG_H
#  include <config.h>
#endif

#include "autosocket.h"

#include "irmp3-ncurses.h"

vcid("$Id: netio.c,v 1.10 2005/03/26 21:03:48 ross Exp $");

struct sockaddr_in irmp3_addr;

#if USE_WINSOCK
bool start_network(void)
{
    WSADATA wd;
    int ret = WSAStartup(0x0101, &wd);

    switch(ret) {
    case 0:
	return true;
    default:
	fprintf(stderr, "WSAStartup failed with error %d\n", ret);
	break;
    }
    return false;
}

bool stop_network(void)
{
    return WSACleanup();
}

#else

bool start_network(void)
{
    return true;
}

bool stop_network(void)
{
    return true;
}
#endif


bool set_address(const char *hostname, const char *port)
{
    irmp3_addr.sin_family = AF_INET;
    inet_aton("127.0.0.1", &irmp3_addr.sin_addr);
    irmp3_addr.sin_port = htons(IRMP3_PORT);
    
    if(hostname) {
	struct hostent *he;

	he = gethostbyname(hostname);
	if(!he) {
	    char buf[1024];

	    snprintf(buf, sizeof buf, _("Could not find host %s"), hostname);
	    herror(buf);
	    return false;
	}
	/* should try the other adresses as well */
	memcpy(&irmp3_addr.sin_addr, he->h_addr_list[0],
	       sizeof irmp3_addr.sin_addr);
    }

    if(port) {
	irmp3_addr.sin_port = htons(atoi(port));
	if(ntohs(irmp3_addr.sin_port) == 0) {
	    fprintf(stderr, _("Invalid port %s\n"), port);
	    return false;
	}
    }

    return true;
}


bool handle_net(int sock)
{
    static char buf[1024];
    static long len = 0;
    long recvd;
    char *ptr;

    recvd = recv(sock, buf + len, sizeof buf - len - 1, 0);
    switch(recvd) {
    case 0:
	sbar_printf(_("Quit"));
	return false;
    case -1:
	switch(errno) {
	case EAGAIN:
	case EINTR:
	    return true;
	default:
	    sbar_printf(_("Recieve error: %d %s"), errno, strerror(errno));
	    error_sleep();
	    return false;
	}
    }

    len += recvd;
    buf[len] = '\0';
    dbg_printf(2, "%ld bytes were recieved. buf (%ld chars) now contains %s",
	       recvd, len, buf);
    if(memchr(buf, '\0', len)) {
	sbar_printf("Error: irmp3d sent us a NUL");
	error_sleep();
	return false;
    }
    assert(len == strlen(buf));

    /* strip any blank lines from start of input
       (I thought that this couldn't happen, but in the case where
       a CRLF is cut in half by the previous read, it can...) */
    for(ptr = buf; ptr[0] == '\r' || ptr[0] == '\n'; ptr++) ;
    if(ptr != buf) {		/* blank lines *were* found */
	memmove(buf, ptr, strlen(ptr) + 1);
	len -= (ptr - buf);
    }

    do {
	ptr = strpbrk(buf, "\r\n");	/* look for EOL */
	if(!ptr) {
	    if(len < sizeof buf)
		break;		       /* no EOL, exit & wait for more input */
	    ptr = strchr(buf, '\0');   /* give up, the input line's just too
					  long */
	    sbar_printf(_("Line too long!"));
	    /* drop through and try and parse what we have */
	}
	while(ptr[0] == '\r' || ptr[0] == '\n') {
	    ptr[0] = '\0';
	    ptr++;
	}
	/* now buf is a NUL terminated line, and ptr is the next line */
	dbg_printf(1, "Recieved line: %s", buf);
	if(buf[0] == '\0')
	    dbg_printf(1, "WTF! buf is zero length. ptr (at offset %d) "
		       "holds %s", ptr - buf, ptr);

	if(!(brwnd ? browser_net_callback : main_net_callback)(sock, buf))
	    return false;

	/* shunt next lines back to the start of buf */
	memmove(buf, ptr, strlen(ptr) + 1);
	len -= (ptr - buf);

	if(buf[0] != '\0' && len == 0)
	    sbar_printf("WTF! len==0 but buf holds %s", buf);
	if(buf[0] == '\r' || buf[0] == '\n')
	    sbar_printf("WTF! buf starts with EOL. but holds %s", buf);
    } while(buf[0]);

    return true;		/* ok, no errors */
}

int connect_irmp3(void)
{
    int sock;

#ifdef SIGPIPE
    signal(SIGPIPE, SIG_IGN);
#endif
    sock = socket(PF_INET, SOCK_STREAM, 0);
    if(sock == -1) {
	sbar_printf(_("Could not open socket: %s"), strerror(errno));
	return -1;
    }
    if(connect(sock, (struct sockaddr *)&irmp3_addr, sizeof irmp3_addr)) {
	sbar_printf(_("Connection to %s:%d failed: %s"),
		    inet_ntoa(irmp3_addr.sin_addr), ntohs(irmp3_addr.sin_port),
		    strerror(errno));
	close(sock);
	return -1;
    }
    sbar_printf(_("Connected to %s:%d"), inet_ntoa(irmp3_addr.sin_addr),
		ntohs(irmp3_addr.sin_port));
    return sock;
}


int netputs(const char *msg, int sock)
{
#ifdef MSG_NOSIGNAL
#  define SENDFLAGS MSG_NOSIGNAL
#else
#  define SENDFLAGS 0
#endif
    int rval = send(sock, msg, strlen(msg), SENDFLAGS);

    send(sock, "\r\n", 2, SENDFLAGS);
    dbg_printf(1, "sent: \"%s\"", msg);
    return rval;
}

int netprintf(int sock, const char *fmt, ...)
{
    va_list ap;
    int rval;
    char buf[1024];

    va_start(ap, fmt);
    rval = vsnprintf(buf, sizeof buf, fmt, ap);
    va_end(ap);
    netputs(buf, sock);
    return rval;
}
