/*******************************************************\
* irmp3-ncurses - An ncurses frontend for irmp3 using   *
* the Network Control Module                            *
* (C) 2003 Ross Axe                                     *
*                                                       *
* net.c - interpreter for network data                  *
\*******************************************************/

#if HAVE_CONFIG_H
#  include "config.h"
#endif

#include "irmp3-ncurses.h"

vcid("$Id: net.c,v 1.30 2005/03/21 02:55:45 ross Exp $");


void send_query(int sock, const char *value)
{
    if(strcmp(value, "balance") == 0)
	netputs("mixer balance +0", sock);	/* hack! */
    else
	netprintf(sock, "query %s", value);
}

/*
 * Issue the initial queries to get things
 * rockin-and-a-rollin
 */
void init_queries(int sock)
{
    send_query(sock, "shuffle");
    send_query(sock, "repeat");
    send_query(sock, "status");		/* irmp3 doesn't issue "230: info stop"
					   Maybe it does now, I'm not sure */
    send_query(sock, "volume");
    send_query(sock, "mute");
    send_query(sock, "treble");
    send_query(sock, "bass");
    send_query(sock, "balance");

    /* This lot only makes sense if status is play or pause...
       Never mind eh. It's harmless */
#if SUPPORT_04X
    /* Also, it's a no-op on irmp3 0.5.6. `query songcmd' will give the info */
    send_query(sock, "mp3file");	/* only 1 of these 2 will get */
    send_query(sock, "oggfile");	/* a response, obviously */
#endif
#if SUPPORT_05X
    send_query(sock, "songcmd");
    send_query(sock, "time");
    send_query(sock, "artistguess");	/* we do the guesses first so that */
    send_query(sock, "titleguess");	/* if IDv3 tags come through later */
#endif					/* they will override the guesses  */

    send_query(sock, "artist");
    send_query(sock, "album");
    send_query(sock, "title");
    send_query(sock, "year");
    send_query(sock, "genre");
    send_query(sock, "comment");

    if(show_playlist)
	send_query(sock, "plfiles");
}



/*
 *               Message Handling Overview
 *
 * The main handle_net function at the end of this module
 * will determine the message number and pass the message
 * to the appropriate handle_xxx function. These top-level
 * handlers then determine the message meaning using the
 * HANDLE_MSG_xxx macros, and either pass it to a
 * subcategory handler (which itself uses HANDLE_MSG_xxx)
 * or passes it straight to the appropriate single-function
 * handler (which is often also a macro).
 *
 * All of the handlers (apart from handle_net itself) return
 * true to indicate that they handled the message, or false
 * if they did not.
 * The top-level and subcategory handlers all take two 
 * arguments, the socket and the buffer that they are to
 * process. At each level of parsing, the parsed portion of
 * the string is chopped off, so eg. the handle_200 function
 * will not see `200: ' on the start of its buf argument.
 * The socket argument is usually ignored.
 */


/*
 * A bunch of macros for handling messages.
 * HANDLE_MSG_STR checks if the first part of buf matches s 
 *            and, if it does, passes the rest of buf to fn
 * HANDLE_MSG_INT is the same but converts buf to an integer
 * HANDLE_MSG_VOID is for messages that don't have an argument
 *            It checks for exact match of buf and s
 *
 * All of these return 'true' if the message was handled and 
 *   fall through if it was not
 */
#define HANDLE_MSG_STR(s, buf, sock, fn) do {if(strstart((s),(buf)) && fn((sock),(buf)+strlen((s)))) return true; } while(0)
#define HANDLE_MSG_INT(s, buf, sock, fn) do {if(strstart((s),(buf)) && fn((sock),atoi((buf)+strlen((s))))) return true; } while(0)
#define HANDLE_MSG_VOID(s, buf, sock, fn) do {if(strcmp((s),(buf))==0 && fn((sock))) return true; } while(0)

/* some no-ops for the above */
#define ignore_int(sock,n) (true)
#define ignore_str(sock, str) (true)
#define ignore_void(sock) (true)

/* and now some real ones */
#define handle_volume(sock, volume) (playerstate.mute ? : setvolume(volume), true)
#define handle_mute(sock, mute) (setmute(mute), true)
#define handle_balance(sock, balance) (setbalance(balance), true)
#define handle_treble(sock, treble) (settreble(treble), true)
#define handle_bass(sock, bass) (setbass(bass), true)
#define handle_shuffle(sock, shuffle) (setshufflemode(shuffle), true)
#define handle_repeat(sock, repeat) (setrepeatmode(repeat), true)

static inline bool handle_sorted(int sock, int count)
{
    sbar_printf(_("Playlist sorted"));
#if PLAYLIST
    if(count != plcount && plcount > 0)
	sbar_printf(ngettext("Looks like I had the playlist wrong, there is "
			     "%d item but I counted %d.", "Looks like I had "
			     "the playlist wrong, there are %d items but I "
			     "counted %d.", count), count, plcount);
    playlist_dirty(sock);
#endif
    return true;
}

static inline bool handle_idle(int sock, int idletime)
{
    if(idletime > 0)
	sbar_printf_noscroll(_("Idle %d:%.2d"), idletime / 60, idletime % 60);
    return true;
}

static bool handle_play_void(int sock)
     /* for when `play' gives no other information */
{
    setplaystate(PS_PLAY);
#if SUPPORT_05X
    send_query(sock, "artistguess");	/* we do the guesses first so that */
    send_query(sock, "titleguess");	/* if IDv3 tags come through later */
#endif					/* they will override the guesses  */
    send_query(sock, "artist");
    send_query(sock, "album");
    send_query(sock, "title");
    send_query(sock, "year");
    send_query(sock, "genre");
    send_query(sock, "comment");

    return true;
}

static bool handle_play_name(int sock, char *buf)
     /* for handling `play' with filename */
{
    clearsonginfo();
    setfilename(buf);
    if(show_playlist)
	playlist_setplaying(playlist_find(buf));
    return handle_play_void(sock);
}

static bool handle_play_mod(int sock, char *buf)
     /* for handling `play' with module info */
{
    char *ptr;

    ptr = strchr(buf, ' ');
    if(ptr) {
	ptr[0] = '\0';
	setplayermodule(buf);
	ptr++;
    } else {			/* if no space found, assume that module name
				   is missing */
	ptr = buf;
    }
    return handle_play_name(sock, ptr);
}

#define handle_pause(sock) (setplaystate(PS_PAUSE), true)
#define handle_unpause(sock) (setplaystate(PS_PLAY), true)
#define handle_stop(sock) (setplaystate(PS_STOP), true)
#define handle_halt(sock) (setplaystate(PS_HALT), true)
#define handle_playererror(sock) (sbar_printf(_("Player Error!")), true)

#define handle_time(sock, buf) (printtime(atol(buf),atol(strchr(buf,' ')+1)), true)

#define refresh_playlist(sock) (playlist_dirty((sock)), true)

#define handle_artist(sock, buf) (setartist(buf), true)
#define handle_album(sock, buf) (setalbum(buf), true)
#define handle_title(sock, buf) (settitle(buf), true)
#define handle_year(sock, buf) (setyear(buf), true)
#define handle_genre(sock, buf) (setgenre(buf), true)
#define handle_comment(sock, buf) (setcomment(buf), true)

static inline bool handle_file(int sock, const char *buf)
{
    setfilename(buf);
    if(show_playlist)
	playlist_setplaying(playlist_find(buf));
    return true;
}

#define handle_endofsong(sock) (clearsonginfo(), true)
#define handle_playerversion(sock, buf) (setplayerversion(buf), true)
#if !PLAYLIST
#  define handle_plfiles ignore_str
#endif


static bool handle_100(int sock, char *buf)
{
    setirmp3version(buf);
    return true;
}

static bool handle_110(int sock, char *buf)
{
    if(!isdigit(buf[0]))
	return false;		/* should just be client #, but,
				   if not, act baffled */
    return true;		/* just ignore this message - we don't
				   really care about client no. */
}

/* handle_mixer is vaguely irmp3 0.5.6 specific */
static bool handle_mixer(int sock, char *buf)
{
    HANDLE_MSG_INT("volume ", buf, sock, handle_volume);
    HANDLE_MSG_INT("mute ", buf, sock, handle_mute);
    HANDLE_MSG_INT("balance ", buf, sock, handle_balance);
    HANDLE_MSG_INT("treble ", buf, sock, handle_treble);
    HANDLE_MSG_INT("bass ", buf, sock, handle_bass);
    return false;
}

static bool handle_200_update(int sock, char *buf)
{
    HANDLE_MSG_STR("artist ", buf, sock, handle_artist);
    HANDLE_MSG_STR("album ", buf, sock, handle_album);
    HANDLE_MSG_STR("title ", buf, sock, handle_title);
    HANDLE_MSG_STR("year ", buf, sock, handle_year);
    HANDLE_MSG_STR("genre ", buf, sock, handle_genre);
    HANDLE_MSG_STR("comment ", buf, sock, handle_comment);
    return false;
}

static bool handle_200(int sock, char *buf)
{
    HANDLE_MSG_INT("shuffle ", buf, sock, handle_shuffle);
    HANDLE_MSG_INT("repeat ", buf, sock, handle_repeat);
#if SUPPORT_05X
    HANDLE_MSG_STR("mixer ", buf, sock, handle_mixer);
    HANDLE_MSG_INT("idle ", buf, sock, handle_idle);

    HANDLE_MSG_VOID("pause", buf, sock, handle_pause);
    HANDLE_MSG_VOID("unpause", buf, sock, handle_unpause);
    HANDLE_MSG_VOID("stop", buf, sock, handle_stop);
    HANDLE_MSG_VOID("halt", buf, sock, handle_halt);
    HANDLE_MSG_STR("play ", buf, sock, handle_play_mod);

    HANDLE_MSG_VOID("player error", buf, sock, handle_playererror);
    HANDLE_MSG_VOID("endofsong", buf, sock, handle_endofsong);

    HANDLE_MSG_STR("update ", buf, sock, handle_200_update);
#endif
    return false;
}

static bool handle_210(int sock, char *buf)
{
#if SUPPORT_04X
    HANDLE_MSG_STR("play ", buf, sock, handle_play_name);
#endif
    HANDLE_MSG_VOID("playlist clear", buf, sock, refresh_playlist);
    HANDLE_MSG_VOID("playlist clearnostop", buf, sock, refresh_playlist);
#if 0				/* change 0 to 1 to see unhandled 210s */
    return false;
#else
    return true;
#endif
}

#define handle_browser_info(sock, buf) refresh_playlist((sock))

static bool handle_browser(int sock, char *buf)
{
    HANDLE_MSG_STR("info ", buf, sock, handle_browser_info);
    return true;		/* ignore all other browser messages */
}

static bool handle_220_info(int sock, char *buf)
{
    HANDLE_MSG_INT("volume ", buf, sock, handle_volume);
    HANDLE_MSG_INT("mute ", buf, sock, handle_mute);
    HANDLE_MSG_INT("balance ", buf, sock, handle_balance);
    HANDLE_MSG_INT("treble ", buf, sock, handle_treble);
    HANDLE_MSG_INT("bass ", buf, sock, handle_bass);
    return false;
}

static bool handle_220(int sock, char *buf)
{
    HANDLE_MSG_INT("volume ", buf, sock, handle_volume);
    HANDLE_MSG_INT("mute ", buf, sock, handle_mute);
    HANDLE_MSG_INT("balance ", buf, sock, handle_balance);
    HANDLE_MSG_INT("treble ", buf, sock, handle_treble);
    HANDLE_MSG_INT("bass ", buf, sock, handle_bass);
    HANDLE_MSG_STR("info ", buf, sock, handle_220_info);
    HANDLE_MSG_INT("sorted: ", buf, sock, handle_sorted);
#if SUPPORT_05X
    HANDLE_MSG_STR("mixer ", buf, sock, handle_mixer);
    HANDLE_MSG_STR("plfiles: ", buf, sock, handle_plfiles);
    HANDLE_MSG_STR("time ", buf, sock, handle_time);
    HANDLE_MSG_STR("player version ", buf, sock, handle_playerversion);

    HANDLE_MSG_STR("artist ", buf, sock, handle_artist);
    HANDLE_MSG_STR("album ", buf, sock, handle_album);
    HANDLE_MSG_STR("title ", buf, sock, handle_title);
    HANDLE_MSG_STR("year ", buf, sock, handle_year);
    HANDLE_MSG_STR("genre ", buf, sock, handle_genre);
    HANDLE_MSG_STR("comment ", buf, sock, handle_comment);
    HANDLE_MSG_STR("file ", buf, sock, handle_file);
    HANDLE_MSG_STR("knowntype ", buf, sock, ignore_str);

    HANDLE_MSG_STR("canplay ", buf, sock, ignore_str);
    HANDLE_MSG_STR("songcmd ", buf, sock, handle_file);
    HANDLE_MSG_VOID("stop", buf, sock, handle_stop);
    HANDLE_MSG_STR("play ", buf, sock, handle_play_mod);
    HANDLE_MSG_VOID("pause", buf, sock, handle_pause);

    HANDLE_MSG_INT("shuffle ", buf, sock, handle_shuffle);
    HANDLE_MSG_INT("repeat ", buf, sock, handle_repeat);

    HANDLE_MSG_STR("browser ", buf, sock, handle_browser);
#endif	 /* SUPPORT_05X */

    return false;
}

static bool handle_230_info_status(int sock, char *buf)
{
    HANDLE_MSG_VOID("play", buf, sock, handle_play_void);
    HANDLE_MSG_VOID("pause", buf, sock, handle_pause);
    HANDLE_MSG_VOID("stop", buf, sock, handle_stop);
    /* irmp3 doesn't always seem to issue "230: info stop".
       Fortunately, I think everything will work as expected
       without it since stop is assumed at startup and any
       subsequent stop will produce a "230: stop" */
    return false;
}

static bool handle_230_info(int sock, char *buf)
{
    HANDLE_MSG_STR("plfiles: ", buf, sock, handle_plfiles);

    HANDLE_MSG_STR("artist ", buf, sock, handle_artist);
    HANDLE_MSG_STR("album ", buf, sock, handle_album);
    HANDLE_MSG_STR("title ", buf, sock, handle_title);
    HANDLE_MSG_STR("year ", buf, sock, handle_year);
    HANDLE_MSG_STR("genre ", buf, sock, handle_genre);
    HANDLE_MSG_STR("comment ", buf, sock, handle_comment);
    HANDLE_MSG_STR("file ", buf, sock, handle_file);

    HANDLE_MSG_INT("shuffle ", buf, sock, handle_shuffle);
    HANDLE_MSG_INT("repeat ", buf, sock, handle_repeat);

    HANDLE_MSG_STR("status ", buf, sock, handle_230_info_status);
    return false;
}

static bool handle_230(int sock, char *buf)
{
    HANDLE_MSG_STR("time ", buf, sock, handle_time);
    HANDLE_MSG_STR("info ", buf, sock, handle_230_info);
    HANDLE_MSG_STR("version ", buf, sock, handle_playerversion);

    HANDLE_MSG_VOID("endofsong", buf, sock, handle_endofsong);
    HANDLE_MSG_VOID("play", buf, sock, handle_play_void);
#if SUPPORT_05X
    HANDLE_MSG_VOID("playing", buf, sock, handle_play_void);
    HANDLE_MSG_STR("play ", buf, sock, handle_play_mod);
    HANDLE_MSG_VOID("stopall", buf, sock, ignore_void);
    /* I don't treat stopall as stop because it doesn't 
       actually seem to stop... */

    HANDLE_MSG_INT("error ", buf, sock, ignore_int);
    HANDLE_MSG_VOID("error", buf, sock, ignore_void);

    /* we can probably do better here */
    HANDLE_MSG_STR("seek ", buf, sock, ignore_str);
#endif

    HANDLE_MSG_VOID("pause", buf, sock, handle_pause);
    HANDLE_MSG_VOID("stop", buf, sock, handle_stop);

#if SUPPORT_04X
    HANDLE_MSG_STR("browser ", buf, sock, handle_browser);
#endif
    return false;
}

static bool handle_240(int sock, char *buf)
{
    /* Really got to figure out what should go here.
       Seems to be an echo response to `query' commands
       An appropriate action may be to clear the corresponding field
       For now, we just ignore it */
    /* The plan is to use these messages, in conjunction with the send_query()
       function, to determine what queries are pending so we can avoid
       reissuing them */
#if 0				/* change 0 to 1 to see unhandled 240s */
    return false;
#else
    return true;
#endif
}


/*
 * main_net_callback
 * Handles all network events
 * Returns false to exit program
 */
bool main_net_callback(int sock, char *buf)
{
    /* welcome messages */
    HANDLE_MSG_STR("100: ", buf, sock, handle_100);	/* Welcome banner */
    HANDLE_MSG_STR("110:", buf, sock, handle_110);	/* Client no. */

    /* (Roughly) from irmp3mod.h ...        With my interperation...
       200: event notification              Information volunteered by irmp3
       210: message from input module       Simple command echo (including
					    internal commands)
       220: message from player module      Similar to 200, but usually asked
					    for externally
       230: other info, answers to queries  Yeh, what he said
       240: queries                         Simple echo of `query xxx' messages
     */
    HANDLE_MSG_STR("200: ", buf, sock, handle_200);
    HANDLE_MSG_STR("210: ", buf, sock, handle_210);
    HANDLE_MSG_STR("220: ", buf, sock, handle_220);
    HANDLE_MSG_STR("230: ", buf, sock, handle_230);
    HANDLE_MSG_STR("240: ", buf, sock, handle_240);

#if SUPPORT_04X
    /* maybe it's an idle time then? (older irmp3 only) */
    if(strchr(buf, ':') == NULL) {
	int idletime = atoi(buf);

	if(idletime > 0) {
	    sbar_printf_noscroll(_("Idle %d:%0.2d"), idletime / 60,
				 idletime % 60);
	    return true;
	}
    }
#endif

    /* dump the unhandled stuff to debug window */
    sbar_printf(_("Unknown message \"%s\""), buf);
    return true;
}
