/* GBINDADMIN, a GTK+ frontend for ISC BIND
 * Copyright C 2005, 2006 Magnus Loef <magnus-swe@telia.com> 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307, USA.
 *
*/


#include "../config.h"
#include <gtk/gtk.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "widgets.h"
#include "gettext.h"
#include "show_info.h"
#include "populate_domain_treeview.h"
#include "populate_resource_treeview.h"
#include "reread_conf.h"
#include "allocate.h"
#include "reverse_address.h"
#include "commands.h"

extern char CHROOT_PATH_BUF[1024];



void add_domain(struct w *widgets)
{
    FILE *fp;
    long conf_size;
    char *line;
    int zone_exists = 0;
    G_CONST_RETURN gchar *name, *address;
    gint active_index;
    gchar *info;
    gchar *forward_zone, *reverse_zone;
    gchar *command, *named_conf, *zone_file;
    char *forward_address;

    name = gtk_entry_get_text(GTK_ENTRY(widgets->add_domain_name_entry));
    address = gtk_entry_get_text(GTK_ENTRY(widgets->add_domain_address_entry));

    /* Domain type combo */
    active_index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgets->add_domain_type_combo));
    

    if( strlen(name) < 3 )
    {
	info = g_strdup_printf(_("The domain name is too short, the domain was not added.\n"));
	show_info(info);
	g_free(info);
	return;
    }
    if( strlen(address) < 5 )
    {
	info = g_strdup_printf(_("The network address is too short, the domain was not added.\n"));
	show_info(info);
	g_free(info);
	return;
    }


    /* Check if its a valid ip-address */
    forward_address = reverse_address((char *)address);
    if( strstr(forward_address, "ADDRESS_ERROR") )
    {
	info = g_strdup_printf(_("The address is not valid\n"));
	show_info(info);
	g_free(info);
	free(forward_address);
	return;
    }
    free(forward_address);


    named_conf = g_strdup_printf("%s/etc/named.conf", CHROOT_PATH_BUF);
    
    /* Check if the site exists */
    if((fp=fopen(named_conf, "r"))==NULL)
    {
        info = g_strdup_printf(_("Domain not added. Could not find named.conf here:\n%s\n"), named_conf);
	show_info(info);
	g_free(info);
	g_free(named_conf);
        return;
    }
    fseek(fp, 0, SEEK_END);
    conf_size = ftell(fp);
    rewind(fp);
    line = allocate(conf_size+1);

    forward_address = reverse_address((char *)address);
    forward_zone = g_strdup_printf("zone \"%s\" IN {\n", name);
    reverse_zone = g_strdup_printf("zone \"%s.in-addr.arpa\" {\n", forward_address);

    while(fgets(line, conf_size, fp)!=NULL)
    {
	if( ! zone_exists && !strcmp(line, forward_zone) )
	{
	    info = g_strdup_printf(_("This forward zone already existed, the domain was not added.\n"));
	    show_info(info);
	    g_free(info);
	    zone_exists = 1;
	}
	if( ! zone_exists && !strcmp(line, reverse_zone) )
	{
	    info = g_strdup_printf(_("This reverse zone already existed, the domain was not added.\n"));
	    show_info(info);
	    g_free(info);
	    zone_exists = 1;
	}
    }
    free(line);
    free(forward_address);
    g_free(forward_zone);
    g_free(reverse_zone);
    fclose(fp);

    if( zone_exists )
    {
	g_free(named_conf);
        return;
    }
    
    /* Write the forward and reverse descriptions to named.conf */
    if((fp=fopen(named_conf, "a"))==NULL)
    {
        info = g_strdup_printf(_("Domain not added. Could not find named.conf here:\n%s\n"), named_conf);
	show_info(info);
	g_free(info);
        return;
    }

    /* This is free'd last */
    forward_address = reverse_address((char *)address);

    /* These are very long zone declarations for named.conf */
    if( active_index <= 0 )
    {
	/* Master domain */
	forward_zone = g_strdup_printf("\nzone \"%s\" IN {\n\ttype master;\n\tfile \"sites/%s/forward.zone\";\n\tallow-transfer { 127.0.0.1; };\n\tallow-update { none; };\n\tallow-query { any; };\n\tzone-statistics yes;\n\tnotify no;\n\talso-notify { };\n};\n", name, name);
	reverse_zone = g_strdup_printf("\nzone \"%s.in-addr.arpa\" {\n\ttype master;\n\tfile \"sites/%s/reverse.zone\";\n\tallow-transfer { 127.0.0.1; };\n\tallow-update { none; };\n\tallow-query { any; };\n\tzone-statistics yes;\n\tnotify no;\n\talso-notify { };\n};\n", forward_address, name);
    }
    else  /* The difference is that allow-update isnt added and masters is */
    if( active_index == 1 )
    {
	/* Slave domain */
	forward_zone = g_strdup_printf("\nzone \"%s\" IN {\n\ttype slave;\n\tfile \"sites/%s/forward.zone\";\nallow-transfer { 127.0.0.1; };\n\tallow-query { any; };\n\tmasters { 127.0.0.1; };\n\tzone-statistics yes;\n\tnotify no;\n\talso-notify { };\n\tallow-notify { };\n};\n", name, name);
	reverse_zone = g_strdup_printf("\nzone \"%s.in-addr.arpa\" {\n\ttype slave;\n\tfile \"sites/%s/reverse.zone\";\n\tallow-transfer { 127.0.0.1; };\n\tallow-query { any; };\n\tmasters { 127.0.0.1; };\n\tzone-statistics yes;\n\tnotify no;\n\talso-notify { };\n\tallow-notify { };\n};\n", forward_address, name);
    }    
    
    fputs(forward_zone, fp);
    fputs(reverse_zone, fp);
    fclose(fp);    

    g_free(forward_zone);
    g_free(reverse_zone);
    g_free(named_conf);

    /* Now create the directory /var/named/etc/sites/site.org */
    command = g_strdup_printf("mkdir -p %s/etc/sites/%s", CHROOT_PATH_BUF, name);
    if( ! run_command(command) )
    {
        info = g_strdup_printf(_("Error running command:\n%s\n"), command);
        show_info(info);
        g_free(info);
    }
    g_free(command);

    

    /* Add the forward zone    FIXME todays date +00 (2002031500) */
    zone_file = g_strdup_printf("%s/etc/sites/%s/forward.zone", CHROOT_PATH_BUF, name);
    if((fp=fopen(zone_file, "w+"))==NULL)
    {
        info = g_strdup_printf(_("Couldnt write the forward zone file here:\n%s\n"), zone_file);
	show_info(info);
	g_free(info);
	g_free(zone_file);
	free(forward_address);
        return;
    }
    forward_zone = g_strdup_printf("$TTL 3D; Forward zone DNS->IP\n@\tIN\tSOA\t%s.\thostmaster.%s. (\n", name, name);
    fputs(forward_zone, fp);
    forward_zone = g_strdup_printf("\t\t\t2002031500\t;\n");
    fputs(forward_zone, fp);

    forward_zone = g_strdup_printf("\t\t\t8H\t\t;\n");
    fputs(forward_zone, fp);

    forward_zone = g_strdup_printf("\t\t\t2H\t\t;\n");
    fputs(forward_zone, fp);

    forward_zone = g_strdup_printf("\t\t\t1W\t\t;\n");
    fputs(forward_zone, fp);

    forward_zone = g_strdup_printf("\t\t\t1D )\t\t;\n\n");
    fputs(forward_zone, fp);

    fclose(fp);
    g_free(zone_file);


    /* Add the reverse zone    FIXME todays date +00 (2006011800) */
    zone_file = g_strdup_printf("%s/etc/sites/%s/reverse.zone", CHROOT_PATH_BUF, name);
    if((fp=fopen(zone_file, "w+"))==NULL)
    {
        info = g_strdup_printf(_("Couldnt write reverse zone file here: \n%s\n"), zone_file);
        show_info(info);
        g_free(info);
	g_free(zone_file);
	free(forward_address);
        return;
    }

    reverse_zone = g_strdup_printf("$TTL 3D; Reverse zone IP->DNS\n@\tIN\tSOA\t%s.\thostmaster.%s. (\n", name, name);
    fputs(reverse_zone, fp);
    reverse_zone = g_strdup_printf("\t\t\t2002031500\t;\n");
    fputs(reverse_zone, fp);

    reverse_zone = g_strdup_printf("\t\t\t8H\t\t;\n");
    fputs(reverse_zone, fp);

    reverse_zone = g_strdup_printf("\t\t\t2H\t\t;\n");
    fputs(reverse_zone, fp);

    reverse_zone = g_strdup_printf("\t\t\t1W\t\t;\n");
    fputs(reverse_zone, fp);

    reverse_zone = g_strdup_printf("\t\t\t1D )\t\t;\n\n");
    fputs(reverse_zone, fp);

    fclose(fp);
    g_free(zone_file);
    free(forward_address);


    /* Chown -R nobody:nobody /var/named/etc/sites/ so that named can write a journal (.jnl) etc */
    command = g_strdup_printf("chown -R %s:%s %s/etc/sites/*", NAMED_USER, NAMED_USER, CHROOT_PATH_BUF);
    if( ! run_command(command) )
    {
        info = g_strdup_printf(_("Error running command:\n%s\n"), command);
        show_info(info);
        g_free(info);
    }
    g_free(command);

    /* Populate the relevant gui sections */
    populate_domain_treeview(widgets);

    populate_resource_treeview(widgets);

    /* FIXME set this as the selected domain, the first domain is selected as is. */
}
