// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#include "Puma/Signature.h"
#include "Puma/StrCol.h"
#include <string.h>
#include <iostream>
using namespace std;

namespace Puma {


Signature::Signature (const char *sig) {
  parse (sig);
}


// Translate multi-word identifiers (unsigned int and friends) to
// non-spaced identifiers by replacing any space with a minus. 
void Signature::translate_multiword_identifiers (char *s) {
  const char *multiwords[] = { 
    "unsigned char", "signed char",
    "unsigned short", "signed short",
    "unsigned int", "signed int", 
    "unsigned long", "signed long",
    "long long", 
    "unsigned long long", "signed long long",
    "long double",
    "operator "
  };
  for (unsigned i = 0; i < sizeof (multiwords) / sizeof (char*); i++) {
    const char *w = multiwords[i];
    char *j;
    while ((j = strstr (s, w)) != 0) {
      for (int k = 0; w[k]; k++) {
        if (w[k] == ' ') {
          j[k] = '-';
        }
      }
    }
  }
}


void Signature::parse (const char *sig) {
  _argc = 0;
  _method = false;

  // remove heading, tailing and multiple spaces 
  while (*sig == ' ') sig++;
  char *s = new char[strlen(sig)+1];
  char *q = s;
  int   space_count = 0;
  while (*sig) {
    if (! strcmp (sig, "const")) {
      sig += 5;
      continue;
    }
    if (! strcmp (sig, "volatile")) {
      sig += 8;
      continue;
    }
    if (*sig == ' ') {
      space_count++;
    } else {
      if (space_count) {
          *q++ = ' ';
        space_count = 0;
      }
      *q++ = *sig;
    }
    sig++;
  }
  *q = 0;
    
  // translate multi-word identifiers into single words
  translate_multiword_identifiers (s);

  // get the parameter list
  if ((q = strchr (s, 0))[-1] == ')') {
    q--;
    _method = true;
        
    // strip the parameter list from the signature
    int level = 1;
    while (level != 0) {
      switch (*--q) {
        case ')': level++; break;
        case '(': level--; break;
      }
    }
    *q++ = 0;
    strchr (q,0)[-1] = 0;

    // remove all spaces from the parameter list
    char *p = new char[strlen (q) + 1];
    char *r = p;
    while (*q) {
      if (*q != ' ') 
        *r++ = *q;
      q++;
    }
    *r = 0;
        
    // parse the arguments
    r = q = p;
    level = 0;
    while (*r) {
      switch (*r) {
        case ',':
          if (level == 0) {
            *r = 0;
            _argv[_argc++] = StrCol::dup (q);
            q = r+1;
          }
          break;
        case '(': level++; break;
        case ')': level++; break;
      }
      r++;
    }
    if (r > q) {
      _argv[_argc++] = StrCol::dup (q);
    }
        
    delete[] p;
  }

  q = strrchr (s, ' ');
  if (q) {
    *q = 0;
    q++;
  } else {
    q = "";
  }
  _name = q;
  _type = s;
}


const char *Signature::name () {
  return _name;
}


const char *Signature::type () {
  return _type;
}


const char *Signature::argv (int i) {
  return _argv[i];
}


unsigned Signature::argc () {
  return _argc;
}


bool Signature::method () {
  return _method;
}


} // namespace Puma
