// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#include "Puma/Config.h"
#include "Puma/PathIterator.h"
#include "Puma/ErrorStream.h"
#include "Puma/PathManager.h"
#include "Puma/FileUnit.h"
#include "Puma/MacroUnit.h"
#include "Puma/RegComp.h"
#include "Puma/SysCall.h"
#include <string.h>
#include <stdlib.h>
#ifdef _MSC_VER
#define PATH_MAX _MAX_PATH
#endif

namespace Puma {


PathManager::~PathManager () {
  for (int i = numProts () - 1; i >= 0; i--)
    delete _protected[i];
}


// Join the paths of the given manager with the paths 
// of this path manager.
void PathManager::join (PathManager &pm) {
  for (int i = 0; i < pm.numPaths (); i++)
    addPath (pm.src (i), pm.dest (i));   
  for (int i = 0; i < pm.numProts (); i++)
    _protected.append (new RegComp (*pm.prot (i)));
}


// Add the separator '/' to the copy of the given string.
char *PathManager::addSeparator (const char *s) const {
  if (! s) return (char*) 0;
  int pos = strlen (s);

  char *str = new char[pos + 2];
  strcpy (str, s);
  if (str[pos - 1] != '/') // Add a trailing '/'.
    strcat (str, "/");
  return str;
}


// Add a new source directory.
void PathManager::addPath (const char *source, const char *destination) {
  if (source) {        
    char *newsrc = addSeparator (source);

    // Don't add a source directory twice.
    // Don't add a sub-directory of an existing source directory.
    for (int pos = numPaths () - 1; pos >= 0 && src (pos); pos--)
      if (strncmp (src (pos), newsrc, strlen (src (pos))) == 0) {
        delete[] newsrc;
        return;
      }
        
    // Add the source path.
    _paths[numPaths ()].src (newsrc);
        
    if (destination) {   
      // Add the destination path.
      char *newdest = addSeparator (destination);
      _paths[numPaths () - 1].dest (newdest);
      delete[] newdest;
    }

    // add the files to iterator file list
    traverse (newsrc);

    delete[] newsrc;
  }
  else if (destination) {
    // Add the destination path.
    char *newdest = addSeparator (destination);
    _paths[numPaths ()].dest (newdest);
    delete[] newdest;
    _paths[numPaths () - 1].src (0);
  }
}


// find the canonical filename representation for a file
bool PathManager::canonFilename (Filename name, Filename &abs_name) const {

  string file_abs = "";
  const char *path = name.path ();
  if (strcmp (path, "") == 0)
    path = ".";
    
  Filename canon_file, canon_dir;
  if (SysCall::canonical (name, canon_file))
    file_abs = canon_file.name ();
  else if (SysCall::canonical (name.path (), canon_dir)) {
    file_abs = canon_dir.name ();
    const char *nodir = strrchr (name.name (), '/');
    if (!nodir) nodir = strrchr (name.name (), '\\');
    if (nodir)
    file_abs += nodir;
  }
  else
    return false; // neither file nor directory exist

  abs_name.name (file_abs.c_str ());
  return true;
}


// checks if a give file (by name) is a registered file of this path manager
// In case of success (found!) the an iterator is returned, which can be
// used to access more information about the file.
bool PathManager::isBelow (const char *file, PFMConstIter &iter) const {

  // determine the canonical name (which has to exist)
  Filename file_abs;
  if (!canonFilename (file, file_abs))
    return false;

  // search for the name and return the result
  iter = _files.find (string (file_abs.name ()));
  return iter != _files.end ();
}


// Add a new file to the project file list
PFMConstIter PathManager::addFile (const ProjectFile &file) {

  Filename file_abs;
  bool have_canon = canonFilename (file.name (), file_abs);
  if (!have_canon) {
    assert (false);
    return _files.end ();
  }

  // insert the file with its canonical name as the key
  pair<PFMConstIter, bool> insert_result =
    _files.insert (PFMPair (string (file_abs.name ()), file));
    
  // return the iterator
  return insert_result.first; 
}


// Add a new file to the project file list
PFMConstIter PathManager::addFile (Filename file) {
  // determine the destination path and call addFile with two args
  std::ostringstream path;
  bool have_dest = getDestinationPath (file.name (), path);
  path << std::ends;
  string dest_path = path.str ();
  return addFile (file, have_dest ? dest_path.c_str () : "<unused-dest-name>");
}


// Add a new file to the project file list with destination path
PFMConstIter PathManager::addFile (Filename name, Filename dest) {
  return addFile (ProjectFile (name, dest));
}


// Set the destination directory of the given source directory.
void PathManager::setDest (const char *source, const char *destination) {
  if (! source) return;
    
  // Search the corresponding path info object.
  int pos;
  for (pos = numPaths () - 1; pos >= 0 && src (pos); pos--)
    if (strcmp (src (pos), source) == 0) 
      break;
  if (pos < 0) return; // Source path doesn't exist.
  if (destination) {   
    // Set the destination path.
    char *newdest = addSeparator (destination);
    _paths[pos].dest (newdest);
    delete[] newdest;
  } else 
    _paths[pos].dest (destination);
}


// Configure the project from the command line or a file.
void PathManager::configure (const Config &c) {
  const ConfOption *d = 0, *p = 0;

  for (unsigned i = 0; i < c.Options (); i++) {
    const ConfOption *o = c.Option (i);
    bool new_p = false, new_d = false;
    
    if (! strcmp (o->Name (), "-w")) {
      if (o->Arguments () != 1) continue;
      protect (o->Argument (0));
    } else if (! strcmp (o->Name (), "-p")) {
      if (o->Arguments () != 1) continue;
      new_p = true;
    } else if (! strcmp (o->Name (), "-d")) {
      if (o->Arguments () != 1) continue;
      new_d = true;
    }
    
    if (new_p) {
      if (p) {
        addPath (p->Argument (0), d ? d->Argument (0): 0);
        if (d) d = 0;
      }
      p = o;
    }
    
    if (new_d) {
      if (d) {
        addPath (p ? p->Argument (0) : 0, d->Argument (0));
        if (p) p = 0;
      }
      d = o;
    }
  }
  
  if (p || d)
    addPath (p ? p->Argument (0) : 0, d ? d->Argument (0) : 0);
}


// Initial globbing implementation.
void PathManager::glob (char *pattern) {
  // Explore the source directories of the manager and for 
  // any file matching the given pattern call the method 
  // action(filename). The default pattern is ".*".
  PathIterator iter (pattern);
  while (iterate (iter))
    action (iter);
}


// Iterate the contents of the paths.
const char *PathManager::iterate (PathIterator &iter) const {

  // a new iterator should start at the beginning
  if (!iter.in_use ()) {
    iter.init (_files.begin ());
  }
  else
    ++iter;
    
  // search for the next matching element
  while (iter != _files.end () &&
         ! iter._regexp->match (iter->second.name ().name ())) {
    ++iter;
  }

  // end or element
  if (iter == _files.end ()) {
    iter.done ();
    return 0;
  }
  else {
    return iter->second.name ().name ();
  }
}


// find all files of a directory tree and add them to _files
void PathManager::traverse (const char *path) {
  // open the current directory
  DirHandle dp = SysCall::opendir (path, &err ());
  if (! dp) { // Skip the directory, may be an access problem.
    err () << sev_error << "Couldn't open directory `" 
           << path << "'." << endMessage;
    return;
  }
  
  // Read the current directory entries.
  const char *entry;
  while ((entry = SysCall::readdir (dp, &err()))) {
    // Create the full name of the current entry, e.g. /tmp/file.c.
    char *name = new char[strlen (path) + strlen (entry) + 2];
    strcpy (name, path);
    if (name[strlen (name) - 1] != '/') 
      strcat (name, "/");
    strcat (name, entry);
    
    // Read the attributes of the current entry.
    FileInfo fi;
    if (! SysCall::stat (name, fi, &err())) continue;
    
    // Test whether entry is a directory.
    if (fi.is_dir ())
      traverse (name); // Dive into the sub-directory.
    else
      // File entry found.
      addFile (name);
    delete[] name;
  }
  
  // Close the current directory and go one up.
  SysCall::closedir (dp, &err());        
}


// Add a regular pattern specifying a path that has to be
// protected from writing.
void PathManager::protect (const char *path) {
  if (path) 
    _protected.append (new RegComp (path));
}


// Return true if the given file or path is protected
// from writing or if it isn't located within one of the 
// source directories, because then it's protected, too.
bool PathManager::isProtected (const char *file) const {
  if (! file) return false;
  
  // Protected by protect patterns?
  for (int i = numProts () - 1; i >= 0; i--) {
    if (prot (i)->match (file)) 
      return true;
  }
  
  // From outside the source directories?
  return ! isBelow (file);
}


const char *PathManager::getDestination (Filename file, ostream *name) const {

  // determine the canonical name (which has to exist)
  Filename file_abs;
  if (!canonFilename (file, file_abs))
    return 0;
  
  for (int i = numPaths () - 1; i >= 0 && src (i); i--) {  
    // TODO: don't determine the canonical directory names every time
    Filename dir_abs;
    if (!canonFilename (src (i), dir_abs)) {
      assert (false);
      return 0;
    }
    // is the canonical dir name a prefix of the filename?
    int len = strlen (dir_abs.name ());
    if (strncmp (dir_abs.name (), file_abs.name (), len) == 0 &&
        *(file_abs.name () + len) == '/') {
      if (name)
        *name << file_abs.name () + len + 1;
      return dest (i);
    }
  }
  return 0;
}


bool PathManager::getDestinationPath (const char *filename, ostream &out) const {
  std::ostringstream rest;
  const char *dir = getDestination (filename, &rest);
  rest << std::ends;
  if (dir) {
    out << dir;
    if (dir [strlen (dir) - 1] != '/')
      out << "/";
    out << rest.str ().c_str ();
    return true;
  }
  return false;
}


} // namespace Puma
