// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __Token_h__
#define __Token_h__

// Abstract interface of a language token. A token is created by an
// invocation of a scanner's 'scan' method. The token encapsulates a
// lexical unit of the scanned text.

#include <stdlib.h>
#include "Puma/LanguageID.h"
#include "Puma/Location.h"
#include "Puma/Printable.h"
#include "Puma/ListElement.h"
#include "Puma/Array.h"
#include <assert.h>
#include "Puma/DString.h"
#include "Puma/StrCol.h"

namespace Puma {


class Token : public ListElement {
  int _type, _real_type;

  Location _location;
  LanguageID _language;
  DString _text;

  Array<int> *_cont_lines;
   
  enum TK_Flags {
          TK_COMMENT = (0x01 << 0),
          TK_CORE = (0x01 << 1),
          TK_MACRO_CALL = (0x01 << 2),
          TK_MACRO_GENERATED = (0x01 << 3)
  };
  unsigned short _flags;

public:
  enum {
    ID_END_OF_FILE = -1,
    ID_UNKNOWN = -2,
    ID_ERROR = -3,
    ID_WARNING = -4
  };

  static LanguageID comment_id, pre_id, dir_id, cpp_id, white_id, 
                    wildcard_id, keyword_id, open_id, macro_op_id,
                    identifier_id, comma_id, close_id;

public:
  Token (int = 0, LanguageID lang = LanguageID(0), const char * = "");
  Token (const Token &);
  virtual ~Token ();

//  virtual void print (ostream &) const;
  void print (ostream &) const;
  virtual ListElement *duplicate () { return new Token (*this); }

  void reset (int = 0, const char * = 0, LanguageID = LanguageID(0));
  
  void location (const Location &loc)      { _location = loc; }
  void cont_lines (Array<int> *cl)  { _cont_lines = cl; }
  int line_breaks () const;
  void macro_generated ()           { _flags |= TK_MACRO_GENERATED; }
  void macro_call ()                { _flags |= TK_MACRO_CALL; }
  
  const DString &text () const     { return _text; }
  char *get_static_text () const;
  int type () const                 { return _type; }
  const Location &location () const { return _location; }
   
  // functions to query token flags
  bool is_macro_generated () const  { return (_flags & TK_MACRO_GENERATED) != 0; }
  bool is_macro_call () const       { return (_flags & TK_MACRO_CALL) != 0; }
  bool is_core () const;
  bool is_identifier () const;    // subset of core
  bool is_keyword () const;       // subset of identifier
  bool is_wildcard () const;      // subset of core
  bool is_preprocessor () const;  // a preprocessor directive (include, etc.)
  bool is_directive () const;     // a compiler directive (pragma, line, etc.)
  bool is_whitespace () const;
  bool is_comment () const;
  bool is_macro_op () const;
  bool is_open () const;          // subset of core
  bool is_comma () const;         // subset of core
  bool is_close () const;         // subset of core

  void *operator new (size_t);
  void  operator delete (void *);

private:
  int get_general_type () const;
  LanguageID language () const { return _language; }
  void is_comment_internal ();
  void is_core_internal ();
};


inline void Token::is_core_internal () {
  if (_language == cpp_id || _language == wildcard_id || 
      _language == open_id || _language == comma_id || 
      _language == close_id || is_identifier ())
    _flags |= TK_CORE;
}
inline void Token::is_comment_internal () { 
  if (_language == comment_id) 
    _flags |= TK_COMMENT;
}

inline bool Token::is_core () const 
 { return (_flags & TK_CORE) != 0; }
inline bool Token::is_identifier () const 
 { return _language == identifier_id || is_keyword ();}
inline bool Token::is_keyword () const 
 { return _language == keyword_id; }
inline bool Token::is_wildcard () const 
 { return _language == wildcard_id; }
inline bool Token::is_preprocessor () const 
 { return _language == pre_id; }
inline bool Token::is_directive () const 
 { return _language == dir_id; }
inline bool Token::is_whitespace () const 
 { return _language == white_id; }
inline bool Token::is_comment () const 
 { return (_flags & TK_COMMENT) != 0; }
inline bool Token::is_macro_op () const 
 { return _language == macro_op_id; }
inline bool Token::is_open () const 
 { return _language == open_id; }
inline bool Token::is_comma () const 
 { return _language == comma_id; }
inline bool Token::is_close () const 
 { return _language == close_id; }

inline ostream &operator << (ostream &os, const Token &object) {
  object.print (os);
  return os;
}

} // namespace Puma

#endif /* __Token_h__ */
