// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __CTemplateInfo_h__
#define __CTemplateInfo_h__

// class and function template info
// knows the template parameters and the template definition
// knows the template instantiations

#include "Puma/CStructure.h"

namespace Puma {


class CT_TemplateDecl;
class CTemplateParamInfo;
class CT_ExprList;
class CT_TemplateName;

class CTemplateInfo : public CStructure {
  CObjectInfo *_ObjectInfo;
  CTemplateInfo *_Base;
  bool _isSpecial;
  CT_TemplateName *_SpecializationName;
  unsigned _Depth;

  Array<CTemplateParamInfo*> _Params;
  Array<CObjectInfo*> _Instances;
  Array<CObjectInfo*> _PseudoInstances;
  Array<CTemplateInfo*> _Specializations;

public: 
  CTemplateInfo ();
  ~CTemplateInfo ();
  
  void removeLinks ();

  bool isFunction () const;                // is function template?
  bool isClass () const;                   // is class template?
  bool isAttribute () const;               // is definition of static data member?
  bool isBaseTemplate () const;            // is not specialization?
  bool isSpecialization () const;          // is not base template?
  
  void increaseDepth ();                   // increase instantiation depth
  void decreaseDepth ();                   // decrease instantiation depth
  unsigned Depth () const;                 // return instantiation depth

  CObjectInfo *ObjectInfo () const;        // the template definition
  CT_TemplateDecl *Tree () const;
  CTemplateInfo *BaseTemplate () const;
  CTemplateParamInfo *newTemplateParam ();
  
  unsigned Parameters () const;
  CTemplateParamInfo *Parameter (unsigned) const;
  CT_ExprList *DefaultArgument (unsigned) const;

  unsigned Instances () const;
  CObjectInfo *Instance (unsigned);
  void addInstance (CObjectInfo *);
  void addPseudoInstance (CObjectInfo *);
  void removeInstance (const CObjectInfo *);
  
  unsigned Specializations () const;
  CTemplateInfo *Specialization (unsigned);
  void addSpecialization (CTemplateInfo *);
  void removeSpecialization (const CTemplateInfo *);
  void isSpecialization (bool);
  void SpecializationName (CT_TemplateName *);
  CT_TemplateName *SpecializationName () const;

  void deleteTemplateParam (const CTemplateParamInfo *); 
  void ObjectInfo (CObjectInfo *);
  void BaseTemplate (CTemplateInfo *);

  static bool equalArguments (CTemplateInstance *, CTemplateInstance *);

private:
  void linkInstance (CObjectInfo *);
  CTemplateInfo *nextTemplate (CTemplateInfo *tinfo) const; 
};

inline CTemplateInfo::CTemplateInfo () :
  CStructure (CObjectInfo::TEMPLATE_INFO),
  _ObjectInfo ((CObjectInfo*)0),
  _Base ((CTemplateInfo*)0),
  _isSpecial (false),
  _SpecializationName ((CT_TemplateName*)0),
  _Depth (0),
  _Instances (20, 20) // big enough?
 {}

inline void CTemplateInfo::increaseDepth () 
 { _Depth++; }
inline void CTemplateInfo::decreaseDepth ()
 { if (_Depth) _Depth--; }
inline unsigned CTemplateInfo::Depth () const
 { return _Depth; }

inline unsigned CTemplateInfo::Instances () const
 { return _Instances.length (); }
inline CObjectInfo *CTemplateInfo::Instance (unsigned i)
 { return _Instances.lookup (i); }
inline void CTemplateInfo::addInstance (CObjectInfo *inst)
 { _Instances.append (inst); linkInstance (inst); }
inline void CTemplateInfo::addPseudoInstance (CObjectInfo *inst)
 { _PseudoInstances.append (inst); linkInstance (inst); }

inline unsigned CTemplateInfo::Specializations () const
 { return _Specializations.length (); }
inline CTemplateInfo *CTemplateInfo::Specialization (unsigned i)
 { return _Specializations.lookup (i); }
inline void CTemplateInfo::isSpecialization (bool v)
 { _isSpecial = v; }
inline CT_TemplateName *CTemplateInfo::SpecializationName () const 
 { return _SpecializationName; }
inline void CTemplateInfo::SpecializationName (CT_TemplateName *name)
 { _SpecializationName = name; }

inline unsigned CTemplateInfo::Parameters () const
 { return _Params.length (); }
inline CTemplateParamInfo *CTemplateInfo::Parameter (unsigned i) const
 { return _Params.lookup (i); }

inline CObjectInfo *CTemplateInfo::ObjectInfo () const
 { return _ObjectInfo; }
inline void CTemplateInfo::ObjectInfo (CObjectInfo *info)
 { if (info && (void*)info < (void*)0x10) *(int*)info = 0;
   _ObjectInfo = info; }
inline CTemplateInfo *CTemplateInfo::BaseTemplate () const
 { return _Base; }
 
inline bool CTemplateInfo::isFunction () const
 { return ObjectInfo () && ObjectInfo ()->FunctionInfo (); }
inline bool CTemplateInfo::isClass () const
 { return ObjectInfo () && ObjectInfo ()->ClassInfo (); }
inline bool CTemplateInfo::isAttribute () const
 { return ObjectInfo () && ObjectInfo ()->AttributeInfo (); }
inline bool CTemplateInfo::isBaseTemplate () const
 { return ! _isSpecial && ! (_Base); }
inline bool CTemplateInfo::isSpecialization () const
 { return _Base || _isSpecial; }

inline CT_TemplateDecl *CTemplateInfo::Tree () const 
 { return (CT_TemplateDecl*)CObjectInfo::Tree (); }


} // namespace Puma

#endif /* __CTemplateInfo_h__ */
