// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __CAttributeInfo_h__
#define __CAttributeInfo_h__

// data object info (variables, class data members)

#include "Puma/CScopeRequest.h"

namespace Puma {


class CT_InitDeclarator;
class CT_ExprList;
class CTemplateInfo;
class CRecord;
class CNamespaceInfo;

/** \file 
 *  Representation of local/global variables and class data members. 
 *  \see Puma::CAttributeInfo */

/** \class CAttributeInfo CAttributeInfo.h Puma/infos/CAttributeInfo.h
 *  A CAttributeInfo object represents a local or global variable 
 *  or a class data member. */
class CAttributeInfo : public CScopeRequest {
  CTemplateInfo *_TemplateInfo;
  
protected:
  /** This constructor is to be used by classes inherited from
   *  CAttributeInfo. It makes it possible to set another object id
   *  than \c ATTRIBUTE_INFO.
   *  \param id The object id to be used instead of \c ATTRIBUTE_INFO. */
  CAttributeInfo (ObjectId id);

public: 
  /** Construct a new attribute info. */
  CAttributeInfo ();
  /** Destruct the attribute info. If the object id is 
   *  \c ATTRIBUTE_INFO, Puma::CObjectInfo::CleanUp() is called. */
  ~CAttributeInfo ();
  
  /** Get the base info object of the info object. <b>The 
   *  corresponding information is not yet set!</b>
   *  \return This method always returns \c NULL.
   *  \warning Do not use this method. */
  CAttributeInfo *BaseObject () const;
  /** Get the info object of the variable definition. 
   *  \return \e this or the info of the definition if the variable 
   *          is a static class data member initialized outside of 
   *          the class declaration. */
  CAttributeInfo *DefObject () const;         
  /** Get the syntax tree node of the variable declaration. */
  CT_InitDeclarator *Tree () const;  
  /** Get the initializer of the variable.
   *  \return The syntax tree node of the initializer or \c NULL
   *          if there is no initializer. */  
  CT_ExprList *Init () const;                  // initializer

  /** Get the corresponding template info of static data members 
   *  of class templates. 
   *  \return The info or \c NULL otherwise. */
  CTemplateInfo *TemplateInfo () const;
  /** Set the corresponding of a static data member or a class
   *  template. 
   *  \param info The info of the template. */
  void TemplateInfo (CTemplateInfo *info);

  /** Get the record info if this is a data member of a record. 
   *  \return The record info or \c NULL if not a member of a record. */
  CRecord *Record () const;
  /** Get the namespace info if the variable is declared in a namespace. 
   *  \return The info or \c NULL if not declared in a namespace. */
  CNamespaceInfo *Namespace () const;
  /** Set the qualification scope of the variable. This can be a namespace
   *  info for variables declared in a namespace or a record info for 
   *  data members of a class/struct/union. 
   *  \param scope The qualification scope of the variable. */
  void QualifiedScope (CStructure *scope);
};

inline CAttributeInfo::CAttributeInfo () :
  CScopeRequest (CObjectInfo::ATTRIBUTE_INFO),
  _TemplateInfo ((CTemplateInfo*)0)
 {}
inline CAttributeInfo::CAttributeInfo (CObjectInfo::ObjectId id) :
  CScopeRequest (id),
  _TemplateInfo ((CTemplateInfo*)0)
 {}

inline CAttributeInfo *CAttributeInfo::BaseObject () const
 { return (CAttributeInfo*)CObjectInfo::BaseObject (); }

inline CTemplateInfo *CAttributeInfo::TemplateInfo () const
 { return _TemplateInfo; }
inline void CAttributeInfo::TemplateInfo (CTemplateInfo *info) 
 { _TemplateInfo = info; }

inline void CAttributeInfo::QualifiedScope (CStructure *s)
 { _QualScope = s; }


} // namespace Puma

#endif /* __CAttributeInfo_h__ */
